// =====================================================================================================================================================================================================
// (c) 2021 Lynn Hansen, KU7Q															                                                                                                               |
// This Source Code Form is subject to the terms of the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007. A copy of this license can be found here: https://choosealicense.com/licenses/gpl-3.0/|
// =====================================================================================================================================================================================================



// THIS FILE CONTAINS CODE THAT MANAGES THE AUDIO THROUGH THE TEENSY 


void SetEqualizer(bool force)
{
	static bool lastModeWasTx = true; //flags change of tx/rx mode - false=was rx, true=was tx. setting it true to start forces rx change
	static bool lastRxEq = false;

	if (gTxModeSet2 <= RADIO_TX_ENAB && (force== true || lastModeWasTx == true || gRxEQ != gRxEQPrev || gRx0 != gRx0Prev || gRx1 != gRx1Prev || gRx2 != gRx2Prev || gRx3 != gRx3Prev || gRx4 != gRx4Prev))
	{
		//we're in Rx mode, check to see if rx equalizer has been enabled
		if (gRxEQ == true && (gOMode == MODE_VOICE || (gOMode == MODE_DIG && gDigProc == true)))
		{
			//SerialOut("Setting Rx equalizer", true);			
			if (lastRxEq == false)
			{
				Amp_Out.gain(LEV_OFF); //we just enabled the rx eq, mute rx while enabling equalizer otherwise we get a burst of noise
			}
			sgtl5000_1.eqSelect(1); //enable equalizer			
			sgtl5000_1.eqFilterCount(7);
			float gain0 = float(gRx0 - 10) - 20; //control range 0-20 in HMI = -10 to +10 dB, subtract 20 to compensate for equalizer gain
			float gain1 = float(gRx1 - 10) - 20;
			float gain2 = float(gRx2 - 10) - 20;
			float gain3 = float(gRx3 - 10) - 20;
			float gain4 = float(gRx4 - 10) - 20;

			//Q values from http://www.sengpielaudio.com/calculator-cutoffFrequencies.htm
			SetFilter(0, FILTER_PARAEQ, 200.0, gain0, 2.0);  //83-482hz
			SetFilter(1, FILTER_PARAEQ, 800.0, gain1, 1.1);  //515-1242hz
			SetFilter(2, FILTER_PARAEQ, 1400.0, gain2, 2.2); //1117-1754
			SetFilter(3, FILTER_PARAEQ, 2000.0, gain3, 3.3); //1734-2306 
			SetFilter(4, FILTER_PARAEQ, 2600.0, gain4, 3.6); //2264-2986 
			SetFilter(5, FILTER_HIPASS, 90.0, 2.0, 1.0);
			SetFilter(6, FILTER_HISHELF, 3200.0, 4.0, 8.0);				
		}
		else
		{
			//passthru
			sgtl5000_1.eqSelect(0); //disable equalizer
			//SerialOut("No Rx equalizer", true);			
		}
		//match prev to current
		gRx0Prev = gRx0;
		gRx1Prev = gRx1;
		gRx2Prev = gRx2;
		gRx3Prev = gRx3;
		gRx4Prev = gRx4;
		gRxEQPrev = gRxEQ;
		lastRxEq = gRxEQ; 
		lastModeWasTx = false;	
		ChkRxSquelch(true); //reset rx levels after changing gain values - we take 10 dB off gain when equalizer is on		
	}
	else if (gTxModeSet2 > RADIO_TX_ENAB && (force == true || lastModeWasTx == false || gTxEQ != gTxEQPrev || gTx1 != gTx1Prev || gTx2 != gTx2Prev || gTx3 != gTx3Prev || gTx4 != gTx4Prev))
	{
		//equalizer only used in voice mode
		if (gTxEQ == true && ((gOMode == MODE_VOICE) || (gOMode == MODE_DIG && gDigProc == true)))
		{
			//SerialOut("Setting Tx equalizer", true);
			sgtl5000_1.eqSelect(1); //select parametric equalizer			
			sgtl5000_1.eqFilterCount(7);
			float gain0 = float(gTx0 - 10); //control range 0-20 in HMI = -10 to +10 dB
			float gain1 = float(gTx1 - 10);
			float gain2 = float(gTx2 - 10);
			float gain3 = float(gTx3 - 10);
			float gain4 = float(gTx4 - 10);

			//Q values from http://www.sengpielaudio.com/calculator-cutoffFrequencies.htm
			SetFilter(0, FILTER_PARAEQ, 200.0, gain0, 2.0);  //83-482hz
			SetFilter(1, FILTER_PARAEQ, 800.0, gain1, 1.1);  //515-1242hz
			SetFilter(2, FILTER_PARAEQ, 1400.0, gain2, 2.2); //1117-1754
			SetFilter(3, FILTER_PARAEQ, 2000.0, gain3, 3.3); //1734-2306 
			SetFilter(4, FILTER_PARAEQ, 2600.0, gain4, 3.6); //2264-2986 
			SetFilter(5, FILTER_HIPASS, 90.0, 2.0, 1.0);
			SetFilter(6, FILTER_HISHELF, 3200.0, 4.0, 8.0);
		}
		else
		{
			//passthru
			sgtl5000_1.eqSelect(0); //disable equalizer
			//SerialOut("No Tx equalizer", true);
		}
		//match prev to current
		gTx0Prev = gTx0;
		gTx1Prev = gTx1;
		gTx2Prev = gTx2;
		gTx3Prev = gTx3;
		gTx4Prev = gTx4;
		gTxEQPrev = gTxEQ;
		lastModeWasTx = true;
	}
}

void SetFilter(int f_number, int type, float f_c, float dbGain, float Q) {
	//set the tx audio parametric equalizer - see https://wiki.uiowa.edu/display/teensymacos/Audio+Processing+on+the+Teensy for info

	calcBiquad(type, f_c, dbGain, Q, dspQ_UNIT, dspF_S, dspFilterParams);
	sgtl5000_1.eqFilter(f_number, dspFilterParams);
}



void SetFilterFreq(int callingSrc)
{
	//set tone detectors
	//callingSrc is used for debugging
	//SerialOut("SetFilterFreq call from " + String(callingSrc) + ", gOMode= " + String(gOMode) + ", gFFTFreq= " + String(gFFTFreq) + ", gLPFltr= " + String(gLPFltr) + ", gNFltr= " + String(gNFltr), true);
	//fix any filters that are off first - 0 causes audio crash
	if (gFFTFreq < 100.0 || gFFTFreq > 3200.0)
	{
		SerialOut("---> Invalid gFFTFReq= " + String(gFFTFreq) + " ,set to 700 hz)", true);
		gFFTFreq = 700.0;
	}
	if (gLPFltr < 1 || gLPFltr > 32)
	{
		SerialOut("---> Invalid gLPFltr= " + String(gLPFltr) + " ,set to 3200 hz)", true);
		gLPFltr = 32;
	}
	if (gNFltr < 1 || gNFltr > 32)
	{
		SerialOut("---> Invalid gNFltr= " + String(gNFltr) + " ,set to 3200 hz)", true);
		gNFltr = 32;
	}
	hmiFFTMarkerUpdate = 0; //update markers on fft next scan	
	if (gOMode == MODE_CW || gOMode == MODE_BEACON)
	{
		Filter_Rx.setBandpass(0, gFFTFreq, .7); //make sure this is enabled - compensates for rolloff on cable
		Filter_Rx.setLowpass(1, float(gLPFltr * 100), .7);
		Filter_Rx.setNotch(2, float(gNFltr * 100), 2);
		Filter_Rx.setNotch(3, float(gNFltr * 100), 2);
		Filter_Rx.update();
		Tone1.frequency(gFFTFreq, 8); //11 mS detect time @ 700 hz
		Tone1.update();		
	}
	else if (gOMode == MODE_RTTY)
	{
		//adjust bandpass for higher freq
		Filter_Rx.setBandpass(0, (rttyFreq[gRtyFrq] + (rttyShift[gRtySft] / 2)), .9);
		Filter_Rx.setLowpass(1, float(gLPFltr * 100), .7);
		Filter_Rx.setNotch(2, float(gNFltr * 100), 2);
		Filter_Rx.setHighpass(3, 1700.0, 2); //filter low freq
		Filter_Rx.update();
		Tone1.frequency(rttyFreq[gRtyFrq], 10);
		Tone1.update();
		if (gRtyFrq == 0)
		{
			//space is above mark if mark is lower freq
			Tone2.frequency(rttyFreq[gRtyFrq] + rttyShift[gRtySft], 10);
		}
		else
		{
			//space is below mark if mark is higher freq
			Tone2.frequency(rttyFreq[gRtyFrq] - rttyShift[gRtySft], 10);
		}
		Tone2.update();
	}
	else
	{
		//all other modes
		//SerialOut(">>>> Set Voice/Dig filter", true);
		Filter_Rx.setBandpass(0, 1600.0, .7); //mid scale
		Filter_Rx.setLowpass(1, float(gLPFltr * 100), .7);
		Filter_Rx.setNotch(2, float(gNFltr * 100), 2);
		Filter_Rx.setNotch(3, float(gNFltr * 100), 2);
		Filter_Rx.update();
	}
}


void SetOutputSource(uint8_t dir, bool force)
{
	//process output change only if direction or gOMode changes = otherwise it will lock up the proc because it's called so much
	static uint8_t lastDir = dir; //holds the last dir we set so we don't have to know the actual state
	static uint8_t lastTxLev = 0xff; //don't change if matches - causes tx click
	//if called with dir=0xff, use last dir	
	if (dir == 0 || (dir == 0xff && lastDir == 0))
	{
		ChkRxSquelch(false); //chk for squelch - this will set Amp_Out gain
	}

	if (dir == lastDir && gOMode == gOModePrev && force == false)
	{
		return;
	}
	if (gTxLev == lastTxLev && (dir == 0xff || force == false))
	{
		//ignore multiple calls to set tx dir
		return;
	}

	if (dir == 0xff)
	{
		dir = lastDir;
	}

	//SerialOut("Output Dir=" + String(dir), true);
	sgtl5000_1.muteLineout();
	sgtl5000_1.muteHeadphone();

	lastDir = dir;
	if (dir == 0)
	{
		//dir 0= use rx source
		//SerialOut("Setting Rx path", true);		
		
		lastTxLev = 0xff; //reset so it works when we go to tx mode

		sgtl5000_1.autoVolumeDisable(); //compressor not used in rx
		if (gTxModeSet2 > RADIO_TX_ENAB)
		{
			gTxModeSet2 = RADIO_TX_ENAB;
		}
		if (gSplit == RADIO_VFO_SPLIT)
		{
			//SerialOut("Updating radio from SetOutputSource == rx", true);
			TxFreq2Radio(gVFOA); //don't set hmi here, it causes problems with fft			
		}
		SetEqualizer(false); //allow eq in rx path in addition to lp and notch		
		//turn Tx outputs off
		Output_Source.gain(2, LEV_OFF); 
		Output_Source.gain(3, LEV_OFF); 
		if (gOMode == MODE_DIG)
		{			
			if (gDigRxPath == DIG_RX_USB)
			{
				if (!gDigProc)
				{
					//use unfiltered path
					Output_Source.gain(0, LEV_OFF);
					Output_Source.gain(1, LEV_OUT_SRC1);
				}
				else
				{
					//use filtered path
					Output_Source.gain(0, LEV_OUT_SRC0);
					Output_Source.gain(1, LEV_OFF);
				}
				Output_Source.update();
				//if - set to 0. - gain just inverts phase
				float g = gVolGain + (gUSBOutLv - 50.0);
				if (g < 0)
				{
					g = LEV_OFF;
				}
				Amp_USB_Out.gain(g);//send rx audio to usb out path					
				if (gMon == true)
				{
					sgtl5000_1.unmuteHeadphone();
				}
				else
				{
					sgtl5000_1.muteHeadphone();
					Amp_Out.gain(LEV_OFF);					
				}				
			}
			else
			{
				Amp_USB_Out.gain(LEV_OFF); //turn off rx audio to usb out											
				Output_Source.gain(0, LEV_OUT_SRC0); //send filtered audio to headphones & fft
				Output_Source.gain(1, LEV_OFF);
				Output_Source.update();
				sgtl5000_1.unmuteHeadphone();
			}			
		}
		else
		{
			//normal Rx audio
			if (gRadio != RADIO_TYPE_PCR1000 || gPCRBW == 0)
			{
				//if not a PCR1000, or it is and the bw = 3kHz, use filters
				Output_Source.gain(0, LEV_OUT_SRC0); //use filtered rx audio - level set on Amp_Out				
				Output_Source.gain(1, LEV_OFF);
			}
			else
			{
				//pcr1000 wide filter enabled, don't use filters
				Output_Source.gain(0, LEV_OFF);
				Output_Source.gain(1, LEV_OUT_SRC1);
			}
			sgtl5000_1.unmuteHeadphone(); //rx audio to headphone jack			
		}
		Amp_Out.update();
		Amp_USB_Out.update();
		Output_Source.update();
	}
	else
	{
		//dir 1= use tx source		
		//SerialOut("Setting Tx path levels", true);

		if (gOMode == MODE_VOICE && gTxComp == true)
		{
			//turn on compressor if enabled in voice mode
			//Use the following settings from https://www.pjrc.com/teensy/gui/?info=AudioControlSGTL5000
			//1=6 dB compression (0=0dB, 1=6db), 0=immediate response (1=25mS, 2=50mS, 3=100mS), 0=softLimit (1=hardLimit), -24dBFS (can be 0dBFS to -96dBFS), 15mS attack time, 40mS decay (attack and decay are in dB p/sec
			//these are experimental - more testing is required
			sgtl5000_1.autoVolumeControl(1, 0, 0, -24, .015, .04);
			sgtl5000_1.autoVolumeEnable(); //turn on with above values
			//SerialOut("Tx Comp On", true);
		}
		else
		{
			//compressor not used in other modes
			sgtl5000_1.autoVolumeDisable();
		}
		if (gSplit == RADIO_VFO_SPLIT)
		{
			//SerialOut("Updating radio from SetOutputSource == tx", true);

			delay(30); //keyup delay
			TxFreq2Radio(gVFOB); //don't set hmi here, it causes problems with fft			
		}
		SetEqualizer(false);
		Output_Source.gain(0, LEV_OFF); //turn off Rx filtered path
		Output_Source.gain(1, LEV_OFF); //turn off Rx unfiltered path			
		if (gOMode == MODE_DIG)
		{	
			float g = gUSBInLv - 50.0;
			if (g < 0)
			{
				g = LEV_OFF;
			}
			Amp_USB_In.gain(g);
			Amp_USB_In.update();

			if (!gDigProc)
			{
				Output_Source.gain(2, LEV_OUT_SRC2); //turn on unfiltered Tx path
				Output_Source.gain(3, LEV_OFF); //turn off filtered Tx path
			}
			else
			{
				Output_Source.gain(2, LEV_OFF); //turn off unfiltered Tx path
				Output_Source.gain(3, LEV_OUT_SRC3); //turn on filtered Tx path -  in the middle of the WSJT-X Pwr scale and 50 or so on gTxLevel
			}
		}
		else if (gOMode == MODE_VOICE || gOMode == MODE_RTTY)
		{
			float g = gMicInLv - 50.0;
			if (g < 0)
			{
				g = LEV_OFF;
			}
			Amp_Mic_In.gain(g);
			Amp_Mic_In.update();
			
			Output_Source.gain(2, LEV_OFF); //turn off unfiltered Tx path
			Output_Source.gain(3, LEV_OUT_SRC3 + 10.0); //turn on filtered tx path 
		}
		else
		{
			//CW & Beacon mode
			//turn off all tx audio paths
			Output_Source.gain(2, LEV_OFF); //turn off unfiltered Tx path
			Output_Source.gain(3, LEV_OFF);
		}	
		float eqBoost = 0;
		if (gTxEQ == true)
		{
			eqBoost = 10.0; //add 10 dB for equalizer loss
		}
		Output_Source.update();
		//if gain is - set to 0 (- #s just invert phase)
		float g = (float(gTxLev) / 10.0) + (gLnOutLv - 50.0) + eqBoost;
		if (g < 0)
		{
			g = LEV_OFF;
		}
		Amp_Out.gain(g);
		Amp_Out.update();
		sgtl5000_1.unmuteLineout();
		if (gMon == true)
		{
			sgtl5000_1.unmuteHeadphone();
		}		
		lastTxLev = gTxLev; //block multiple calls to set tx path and level if level hasn't changed
	}
}

void SetTxPath()
{
	//configure tx audio path based on gOMode - use SetOutputSource() to switch audio paths

	switch (gOMode)
	{
	case MODE_VOICE:
		Mixer_Tx.gain(0, LEV_MIXER_TX0); 
		Mixer_Tx.gain(1, LEV_OFF); //make sure unused sources are off
		Mixer_Tx.gain(2, LEV_OFF); //rtty off
		Mixer_Tx.gain(3, LEV_OFF); //sd raw audio
		//SerialOut("=====> Setting tx path to Voice-MIC", true);
		RTTY.amplitude(0);
		break;
	case MODE_RTTY:
		Mixer_Tx.gain(0, LEV_OFF);
		Mixer_Tx.gain(1, LEV_OFF);
		Mixer_Tx.gain(2, LEV_MIXER_TX2); //enable RTTY osc path through mixer, level set by RTTY.amplitude
		Mixer_Tx.gain(3, LEV_OFF);		
		//SerialOut("=====> Setting tx path to RTTY", true);
		break;
	case MODE_DIG:
		//set level using lev		
		if (gDigTxPath == DIG_TX_USB)
		{			
			Mixer_Tx.gain(0, LEV_OFF);
			Mixer_Tx.gain(1, LEV_MIXER_TX1); //enable USB tx path 	
			//SerialOut("=====> Setting tx path to DIG-USB", true);
		}
		else
		{
			Mixer_Tx.gain(0, LEV_MIXER_TX0);//use mic (right channel of line-in) for tx  					
			Mixer_Tx.gain(1, LEV_OFF);
			//SerialOut("=====< Setting tx path to DIG-MIC", true);
		}
		Mixer_Tx.gain(2, LEV_OFF);
		Mixer_Tx.gain(3, LEV_OFF);
		RTTY.amplitude(LEV_OFF);
		break;
	default:
		//cw or beacon modes - all audio out off
		Mixer_Tx.gain(0, LEV_OFF);
		Mixer_Tx.gain(1, LEV_OFF);
		Mixer_Tx.gain(2, LEV_OFF);
		Mixer_Tx.gain(3, LEV_OFF);		
		//SerialOut("=====> Turning off all Tx paths", true);
		RTTY.amplitude(LEV_OFF);
		break;
	}
	Mixer_Tx.update();
	RTTY.update();
}


void TxRTTYChr()
{
	//if we have a chr in the rtty tx buffer, clock it out now
	static int bitPtr = 5; //clocks out start and data bits
	static uint32_t bitTmr = 0; //holds length of bit in mSec
	static char chr = 0xff; //set to 0xff if empty
	static uint8_t mask = 0x10; //rotate mask to the right for each bit in chr

	static uint32_t mSec = millis(); //holds time of last bit start

	if (chr == 0xff)
	{
		//send chr LSB first

		//buffer is empty, load it		
		chr = rttyTxChr;
		bitPtr = 5; //start & data bits  
		bitTmr = rttyBaud[gRtyBaud]; //time in mS for full bit
		//debug
		//bitTmr = 5000; //set long to measure mark/space freq
		RTTY_KEY.amplitude(rttyFreqShift[gRtySft]); //start bit = space
		mSec = millis(); //start clock
		mask = 0x01; //reset mask
	}
	else if (bitPtr > 0)
	{
		//clock out data bits lsb first
		if (millis() - mSec > bitTmr)
		{
			//mask bit of chr
			//SerialOut("Sending bit " + String(bitPtr,HEX) + "=", false);
			if (chr & mask)
			{
				//mark if 1
				RTTY_KEY.amplitude(0.0);
			}
			else
			{
				//space if 0
				RTTY_KEY.amplitude(rttyFreqShift[gRtySft]);
			}
			bitPtr--;
			mask = mask << 1; //rotate for next bit
			mSec = millis();
		}
	}
	else if (bitPtr == 0)
	{
		if (millis() - mSec > bitTmr)
		{
			//last data bit has been sent, set bitTmr to # of stop bits			
			RTTY_KEY.amplitude(0.0); //mark for stop bit
			mSec = millis();
			switch (gRtyStp)
			{
			case 1:
				//1.5 stop bits
				bitTmr = bitTmr + uint32_t(bitTmr / 2);
				break;
			case 2:
				//2 stop bits
				bitTmr = bitTmr * 2;
				break;
			default:
				//leave bitTmr to 1 bit time for 1 stop bit
				break;
			}
			bitPtr--; //set to -1
		}
	}
	else
	{
		if (millis() - mSec > bitTmr)
		{
			//we're done, reset for next chr			
			chr = 0xff;
			bitPtr = 5;
			rttyTxChr = 0;
			rttyTxBusy = false;
		}
	}

}
