/*  RSL_VNA5 Arduino sketch for audio VNA measurements.
    Copyright (C) 2017 Robert Larkin  W7PUA

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* RSL_VNA5.ino    v0.5.7    8 Sept 2017
   Orig v0.1.0   18 Dec 2016
   Output a sine wave from the Left CODEC Line Output
   Receive that signal with Left Codec input.

   Use Right CODEC input to measure the voltage source.
   Use two mixers (multipliers) plus LPF to create i and q
   Measure sqrt(i^2 + q^2) and atan2(q, i), print to serial stream
   v0.2.0 revised to move I and Q through queue's.  RSL
   v0.2.2 brought out reference signal from Right channel. RSL
   v0.3.0 Create separate measurement function and measure power off I & Q  RSL
   v0.3.1 Removed post multiplier filters and made 1/freq a sub-multiple of
        measurement time.  Added tracking of source phase.   RSl 3 Jan 2017
   v0.4.0 test bed, added SerialCommandR
   v0.4.1 Re-did freq control structrue to recompute many variables when freq is changed.
        added setUpNewFreq() and modifyFreq()  RSL 25 Jan 2017
   v0.5.1 Revised to use Steber method of impedance determination (series R-Z). RSL 31 Jan 2017
   v0.5.2 Redid serial command structure to be orderly.RSL 3 March 17
   v0.5.3 Added corrections for input circuitry of op-amps  RSL 13 Mar 2017
   v0.5.4 Redid command structure.  Added numerous commands.  RSL 16 Mar 2017
   v0.5.6 Added ILI9341 320x240 display and touch screen.
   v0.5.7 Added vavnaState and lcdMenu[][].  Fixed LCD scan functions.  RSL 1 May 2017
   v0.5.7 Added column headings for un-annotated sweep output.  Removed baud control.
          RSL 8 Sept 2017
*/

/* TEENSY NOTE - This was tested using a Teensy 3.6.  It probably is OK for the slower 3.5.
   Versions like 3.2 may not have sufficient speed. If a Teensy board is to be
   purchased, buy the 3.6!
   This program was tested with Arduino 1.8.2 and Teensyduino 1.36.
*/

/* The audio board uses the following pins.
    9 - BCLK
   11 - MCLK
   13 - RX
   15 - VOL    (analog pot)
   18 - SDA
   19 - SCL
   22 - TX
   23 - LRCLK
   Five pins are used for control of the AVNA:
   35 - R5K relay
   36 - 50 Ohm relay
   37 - Measuretransmission sw
   38 - Measure Impedance
   39 - Connect inputs for calibration
   Two pins control the front panel bi-color LED
   0  - Bicolor 
   1  - Bicolor
   One pin inputs the weighted sum of all power supplies
   21 - Power Supply monitor
   Five pins control the 320 x 240 display:
   7  - SPI MOSI
   8  - SPI MISO
   10 - ILI9341 Display D/C
   14 - SPI SCK
   20 - Chip Select for 9341 Display
   PIns 7, 8 and 14 are also used for the same function, for touch screen
   In addition touch screen uses
   6  - Touch CS
   24 - Touch IRQ
*/

#include <Audio.h>
#include <Wire.h>
#include <SPI.h>
#include <SD.h>
#include <SerialFlash.h>
#include <EEPROM.h>
#include <ILI9341_t3.h>
#include <font_Arial.h> // from ILI9341_t3
#include <XPT2046_Touchscreen.h>
#include "complexR.h"
#include "SerialCommandR.h"
#include <stdio.h>
#include <math.h>

#define DIAGNOSTICS 0
// static params want to come from EEPROM.  Set following to 1 to reset these to initial values
#define RE_INIT_EEPROM 0

// Who is in charge
#define CTRL_LCD   1
#define CTRL_USB   2

// For control of panel LED
#define LOFF 0
#define LSTART 1
#define LGREEN 2
#define LRED 3
// and the digital pins
#define PANEL_LED0 0
#define PANEL_LED1 1

// Digital hardware control pins
#define R50_36 36
#define R5K_35 35
#define CAL_39 39
#define IMPEDANCE_38 38
#define TRANSMISSION_37 37

#define FILT_NONE 0
#define LPF2300  1
#define NUM_VNAF 14

// For setting sample rates:
#define S44100 0
#define S44117 1
#define S48K   2
#define S96K   3
#define S100K  4
#define FBASE  44117.64706

// msec delay before measurement
#define ZDELAY 10

#define NOCAL 0
#define OPEN 1
#define THRU 2

/* LCD touchscreen operation is based on avanState levels, set by menus (or serial commands):
 * avnaState 
 *     0 - IDLE     Idle, no measurment being made  
 *     1 - ZSINGLE  Z Meas 1 Freq
 *     2 - TSINGLE  T Meas 1 Freq  
 *     3 - ZSWEEP   Z Meas Sweep
 *     4 - TSWEEPT  Meas Sweep
 *     5 - WHATSIT  What Component
 *     6 - LCDHELP  Help
 */
#define IDLE     0
#define ZSINGLE  1
#define TSINGLE  2  
#define ZSWEEP   3
#define TSWEEP   4
#define WHATSIT  5
#define LCDHELP  6

// defines for Serial control
#define NO_MEASURE   0
#define IMPEDANCE    1
#define TRANSMISSION 2

#define SWEEP 0
#define SINGLE 1

#define R_OFF 0
#define R50   1
#define R5K   2

// Values for doRun
#define RUNNOT     0
#define CONTINUOUS 1
#define COUNTDOWN  2
#define SINGLE_NC  3
#define POWER_SWEEP 4

#define r2d(r) 57.2957795130823*r
#define d2r(d) 0.0174532925199433*d

// Pin assignment for AVNA rev 0.55 board
#define TFT_DC 10
#define TFT_CS 20
#define TFT_RST 255
#define TFT_MOSI 7
#define TFT_SCLK 14
#define TFT_MISO 8

// Touch CS
#define T_CS_PIN  6
// Touch IRQ pin
#define TIRQ_PIN 24
//=============================  OBJECTS  ========================================

SerialCommand SCmd;   // The SerialCommand object to control AVNA

// 320 x 240 Screen object
ILI9341_t3 tft = ILI9341_t3(TFT_CS, TFT_DC, TFT_RST, TFT_MOSI, TFT_SCLK, TFT_MISO);

// XPT2046_Touchscreen ts(T_CS_PIN);  // Param 2 - NULL - No interrupts
// XPT2046_Touchscreen ts(T_CS_PIN, 255);  // Param 2 - 255 - No interrupts
XPT2046_Touchscreen ts(T_CS_PIN, TIRQ_PIN);  // Param 2 - Touch IRQ Pin - interrupt enabled polling

//Audio objects          instance name
AudioControlSGTL5000     audioShield;
AudioInputI2S            audioInput;      // Measurement signal
AudioFilterFIR           firIn1;
AudioFilterFIR           firIn2;
AudioSynthWaveform       waveform1;       // Test signal
AudioSynthWaveform       waveform2;       // 90 degree
AudioEffectMultiply      mult1;           // I mixer
AudioEffectMultiply      mult2;           // Q mixer
AudioSynthWaveformDc     DC1;             // Gain control for DAC output
AudioEffectMultiply      multGainDAC;
AudioOutputI2S           i2s2;
AudioRecordQueue         queueNN[4];      // Two pairs of I and Q
// And for the reference channel
AudioEffectMultiply      multR1;          // I mixer
AudioEffectMultiply      multR2;          // Q mixer
// Additional measurments
AudioAnalyzePeak         pkDet;           // These 2 watch for overload
AudioAnalyzePeak         pkDetR;
//                                  Into cord     Out-of cord
//                                 coming from    going to
AudioConnection          patchCord0(waveform1, 0, multGainDAC, 0);  // Test signal
AudioConnection          patchCord0A(DC1, 0,      multGainDAC, 1);  // Set gain
AudioConnection          patchCord0C(multGainDAC, 0, i2s2, 0);      // Test signal to L output
// I-Q multipliers
AudioConnection          patchCord1(waveform1, 0, mult1, 1);        // In phase LO
AudioConnection          patchCord2(waveform2, 0, mult2, 1);        // Out of phase LO
AudioConnection          patchCordF1(audioInput, 0, firIn1, 0);     // Left input to FIR LPF
AudioConnection          patchCordF2(firIn1, 0, mult1, 0);          // FIR LPF to mixer I
AudioConnection          patchCordF3(firIn1, 0, mult2, 0);          // FIR LPF to mixer Q
AudioConnection          patchCord5(mult1, 0,   queueNN[0], 0);     // Q
AudioConnection          patchCord6(mult2, 0,   queueNN[1], 0);     // I
AudioConnection          patchCordpp(audioInput, 0, pkDet, 0);      // 2.0 max p-p
// For ref channel
AudioConnection          patchCordR1(waveform1, 0, multR1, 1);      // In phase LO
AudioConnection          patchCordR2(waveform2, 0, multR2, 1);      // Out of phase LO
AudioConnection          patchCordG1(audioInput, 1, firIn2, 0);     // Left input to FIR LPF
AudioConnection          patchCordG2(firIn2, 0, multR1, 0);         // FIR LPF to mixer I
AudioConnection          patchCordG3(firIn2, 0, multR2, 0);         // FIR LPF to mixer Q
AudioConnection          patchCordR5(multR1, 0, queueNN[2], 0);     // RQ
AudioConnection          patchCordR6(multR2, 0, queueNN[3], 0);     // RI
AudioConnection          patchCordRp(audioInput, 1, pkDetR, 0);

Complex Cone(1.0, 0.0);          // Complex number 1.0 + j0.0

// ================================  Global Variables  =====================================
// LPF, 100 taps, 2300 Hz Fco, 100 KHz sample rate
// -20 dB stopband
static int16_t lp2300_100K[100] = {
  1083, -751, -514, -336, -201, -99, -20, 42, 90, 130,
  163, 188, 205, 213, 215, 207, 188, 159, 124, 75,
  21, -40, -105, -173, -239, -300, -353, -394, -420, -428,
  -415, -379, -318, -232, -121, 13, 171, 346, 538, 741,
  950, 1160, 1366, 1561, 1740, 1897, 2029, 2132, 2203, 2238,
  2238, 2203, 2132, 2029, 1897, 1740, 1561, 1366, 1160, 950,
  741, 538, 346, 171, 13, -121, -232, -318, -379, -415,
  -428, -420, -394, -353, -300, -239, -173, -105, -40, 21,
  75, 124, 159, 188, 207, 215, 213, 205, 188, 163,
  130, 90, 42, -20, -99, -201, -336, -514, -751, 1083
};

/*  Frequency Table
    {0.0, 0.0} are complex zeros.
    Frequencies are channelized for AVNA4, with the exception
    that vnaF[0] is not part of the sweep and can be used as a
    manually controlled frequency. Thirteen frequencies are
    used for the sweep from 10 to 40,000 Hz.
*/
struct measureFreq {
  float    freqHz;            // Requested frequency (display freq, also)
  float    freqHzActual;      // Achieved frequency (used in line 2 data)
  uint16_t numTenths;
  double   vRatio;            // Vor/Vom magnitudes
  double   dPhase;            // phaseor - phaseom
  double   thruRefAmpl;
  double   thruRefPhase;
} vnaF[NUM_VNAF] =
{ 1000.0,   1000.0000,  5, 1.0, 0.0,  1.0, 0.0,
  10.0,       10.0000, 20, 1.0, 0.0,  1.0, 0.0,
  20.0,       20.0000, 10, 1.0, 0.0,  1.0, 0.0,
  50.0,       50.0000, 10, 1.0, 0.0,  1.0, 0.0,
  100.0,     100.0000,  5, 1.0, 0.0,  1.0, 0.0,
  200.0,     200.0000,  5, 1.0, 0.0,  1.0, 0.0,
  500.0,     500.0000,  2, 1.0, 0.0,  1.0, 0.0,
  1000.0,   1000.0000,  1, 1.0, 0.0,  1.0, 0.0,
  2000.0,   2000.0000,  1, 1.0, 0.0,  1.0, 0.0,
  5000.0,   5000.0000,  1, 1.0, 0.0,  1.0, 0.0,
  10000.0, 10000.0000,  1, 1.0, 0.0,  1.0, 0.0,
  20000.0, 20000.0000,  1, 1.0, 0.0,  1.0, 0.0,
  30000.0, 30000.9997,  1, 1.0, 0.0,  1.0, 0.0,
  40000.0, 39998.0006,  1, 1.0, 0.0,  1.0, 0.0
};

uint16_t nFreq = 0;             // 1.0 kHz default
uint16_t nFreqDisplay = 3;      // For LCD sweep display; start, plus 6 higher
uint16_t num256blocks;          // These control sampling time
uint16_t numCycles;
/* In getting data from audio objects, we have need for
    four different data types.  These 'NN' arrays are
    0 - Q or out-of-phase measurement
    1 - I or in-phase measurement
    2 - RQ or out-of-phase measurement of reference channel
    3 - RI or in-phase measurement of reference channel
*/
boolean NNReady[4] = {false, false, false, false};
float   sumNN[4];
double  superAveNN[4];
int16_t bufferNN[4][256];

// Data to be saved when updated and used at power up.
struct saveState {
  uint16_t startup;        // Detect valid EEPROM
  float    freqHz0;        // Goes with nFreq = 0
  float    freqHzActual0;  // This also
  uint16_t msDelay;
  uint16_t ZorT;
  uint16_t SingleorSweep;
  uint16_t iRefR;
  float    valueRRef[3];
  float    vMult;
  // capInput and resInput are used to compensate for high frequency, high impedance
  // measurements.  These are the stray capacity of the Measure wiring, and R1 1 Megohm.
  float    capInput;
  float    resInput;
  // capCouple compensates for the 0.22 uF at the R or M amp, and effects low frequencies.
  float    capCouple;
  // seriesR and seriesL compensate for the leads going to the DUT, and effect high frequencies
  // and low impedance measurements.
  float    seriesR;
  float    seriesL;
};
#define DEFAULT_PARAMETERS {0X4321, 1000.0, 999.9736, 1, IMPEDANCE, SWEEP, R50, {0.1, 50.00, 5000.0}, 1.0, 40.0E-12, 1000000.0, 0.22E-6, 0.07, 20.0E-9}
// Set up a union of bytes that allow EEPROM storage of VNA state when turned off. Init goes with first member
union {
  saveState lastState;
  uint8_t save_vnaF[sizeof(saveState)];
} uSave = DEFAULT_PARAMETERS;
// The values above are the "default values."  The variables will be changes by data from EEPROM to get
// the user selected values.  These defaults will never appear unless commanded by PARAM1 command,
// or by the RE_INIT_DEFAULT define.

bool     calZSingle = false;
bool     calZSweep = false;
bool     calTSingle = false;
bool     calTSweep = false;
bool     dataValidTSweep = false;
bool     dataValidZSweep = false;
uint16_t doRun = RUNNOT;  //  Set of measurements or sweeps, also CONTINUOUS or COUNTDOWN
uint16_t nRun = 0;        // count down until 0
// Control of the AVNA comes from either the USB communications, or from the touch screen,
// at start up. The touch screen can be turned off by command.  The USB command is always running..
// Output is to either USB or to the screen display, as controlled by useUSB and useLCD.
bool useUSB = true;
bool useLCD = true;

// There are difference between USB commands and LCD menu operations. The LCD is more automatic
// and the USB is more detailed.  Power up is with the LCD and the primary control can be changed
// via the USB CONTROL command.
uint16_t primaryControl = CTRL_LCD;
uint16_t avnaState = 0;
uint16_t currentMenu = 0;
// The table of menu numbers sets the next menu to be called, base on the 
// current menu, and the menu position touched.
uint16_t lcdMenu[11][6]    // Select next as 0 to 6. Indices are [current menu][menu item]
  = { 0, 1, 2, 7, 3, 5,    // Menu 0 Home
      0, 1, 1, 1, 1, 8,    // Menu 1 Single Frequency 
      0, 2, 2, 2,10, 9,    // Menu 2 Sweep Frequency
      0, 3, 3, 3, 3, 3,    // Menu 3 Settings
      0, 0, 0, 0, 0, 0,    // Menu 4 More settings
      0, 5, 6, 5, 5, 5,    // Menu 5 Help
      0, 5, 6, 6, 6, 6,    // Menu 6 More help
      0, 7, 7, 7, 7, 7,    // Menu 7 What Component
      0, 8, 8, 8, 8, 8,    // Menu 8 Measure single transmission
      0, 9, 9, 9, 9, 9,    // Menu 9 Measure swept transmission
      0,10,10,10,10,10 };  // Menu 10 Measure swept impedance

void tNothing(void);       // Declarations are required to reference function
void tToHome(void);        //   by pointer, just like C
void tDoSingleFreq(void);
void tDoSweep(void);
void tDoSweepZ(void);
void tDoSweepT(void);
void tSweepFreqDown(void);
void tSweepFreqUp(void);
void tWhatIntro(void);
void tDoWhatPart(void);
void tUseLF(void);
void tNoLF(void);
void tDoSettings(void);
void tDoHelp(void);
void tDoHelp2(void);
void tFreqUp(void);
void tFreqDown(void);
void tFreq0(void);
void tSet50 (void);
void tSet5K(void);
void tDoSingleZ(void);
void tDoSingleT(void);
void tCalCommand(void);

// 2D array of function pointers to be called when a menu item is touched 
// 11 menus and 6 buttons per menu    
void (*t[11][6])(void) =
{ tToHome, tDoSingleFreq, tDoSweep, tWhatIntro, tDoSettings, tDoHelp,     // 0 Home
  tToHome, tFreqDown, tFreqUp, tFreq0, tDoSingleZ, tDoSingleT,            // 1 Single Freq
  tToHome, tSweepFreqDown, tSweepFreqUp, tNothing, tDoSweepZ, tDoSweepT,  // 2 Sweep Freq
  tToHome, tSet50,  tSet5K, tToHome, tToHome, tToHome,                    // 3 Settings
  tToHome, tToHome, tToHome, tToHome, tToHome, tToHome,                   // 4 More Settings
  tToHome, tDoHelp, tDoHelp2, tNothing, tNothing, tNothing,               // 5 Help
  tToHome, tDoHelp, tDoHelp2, tNothing, tNothing, tNothing,               // 6 More Help
  tToHome, tNothing, tNothing, tNoLF, tUseLF, tDoWhatPart,                // 7 What Component
  tToHome, tNothing, tNothing, tNothing, tCalCommand, tDoSingleT,         // 8 Single T
  tToHome, tSweepFreqDown, tSweepFreqUp, tNothing, tCalCommand, tDoSweepT, // 9 Sweep T
  tToHome, tSweepFreqDown, tSweepFreqUp, tNothing, tCalCommand, tDoSweepZ // 9 Sweep Z
};

uint16_t nSampleRate = S100K;     // Current rate,  1 is S44117 or 44711.65Hz rate
double sampleRateExact = 100000000.00;
double factorFreq = 0.4411764706;

Complex Tmeas(0.0, 0.0);          // Result of measureT()
// Control serial printing for Z measurement (both can be true):
bool seriesRX = true;
bool parallelRX = true;
// annotate refers to serial print of CSV or annotated with "Freq=", etc.
bool annotate = true;
// verboseData serial prints the inner-workings.  Not normally done.
bool verboseData = false;
bool parallelGB = false;     // LCD, parallel-model single-freq, display G-B or parallel R and X
uint16_t firstLCDSweepFreq = 3;

float dacSweep = 0.0;   // Level during sweep
float dacLevel = 0.4;

// VNAMode 0-Single Freq, 0 to 9, 1-Sweep at nFreq, 2-Sweep with delay
uint16_t VNAMode = 1;
// Final results
double amplitudeV, amplitudeR;
double phaseV, phaseR;

// Complex is an object.  Store Z and T sweep results in arrays of these. Room for 15; 14 are used (March 17).
Complex Z[15] = {Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0)
                };
Complex Y[15] = {Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0)
                };
Complex T[15] = {Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0),
                 Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0), Complex(0.0, 0.0)
                };
float sLC[15] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
float pLC[15] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
float Q[15]   = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

boolean printReady = false;
uint32_t countMeasurements;   // The number achieved at a freq
unsigned long timeUL;
uint16_t teststate = 0;
uint16_t test_ry = 0;
uint16_t test_sw = 0;

boolean wastouched = true;  // Touch screen
boolean whatUseLF = true;   // Use 10, 20, 50 Hz for What? (speed up)

// The units for displaying RL and C values (2 char ea). Until we have an omega, will use 'O' as Ohms:
char *rUnits[7]={(char*)"pO", (char*)"nO", (char*)"uO", (char*)"mO", (char*)"  ", (char*)" K", (char*)" M"};
// Conductance basic unit is the mho (Ohm spelled backwards). Abbreviating as m produces some unfamiliar units:
char *gUnits[7]={(char*)"pm", (char*)"nm", (char*)"um", (char*)"mm", (char*)" m", (char*)"Km", (char*)"Mm"};
// Inductor and then capacitor units
char *lUnits[7]={(char*)"pH", (char*)"nH", (char*)"uH", (char*)"mH", (char*)" H", (char*)"kH", (char*)"MF"};
char *cUnits[7]={(char*)"pF", (char*)"nF", (char*)"uF", (char*)"mF", (char*)" F", (char*)"kF", (char*)"MF"};

unsigned long tms;

// =======================  SETUP  =============================================
void setup()
  {
  uint16_t i;
  panelLED(LSTART);
  panelLED(LRED);
  Serial.begin(9600);     // Use "BAUD 230400", etc., command to change rate
  delay(1000);
  // The following #if allows a way to reinitialize the EEPROM settings.  Notnormally needed.
#if !RE_INIT_EEPROM  
  if(EEPROM.read(0)==0X21 && EEPROM.read(1)==0X43)     // EEPROM data is valid
    { Serial.println("Loading EEPROM data");
     loadStateEEPROM();    }
  else
#endif
     saveStateEEPROM();   // Initialize EEPROM, coming from init values for saveState

  tft.begin();
  tft.setRotation(1);
  tft.fillScreen(ILI9341_BLACK);
  topLines();
  ts.begin();
  tft.setFont(Arial_8);
  tft.setTextColor(ILI9341_WHITE);
  writeMenus(0);
  tToHome();             // Adds Welcome message
  ts.begin();            // Touch screen
  pinMode(R50_36, OUTPUT);    // Five hardware control pins
  pinMode(R5K_35, OUTPUT);
  pinMode(CAL_39, OUTPUT);
  pinMode(IMPEDANCE_38, OUTPUT);
  pinMode(TRANSMISSION_37, OUTPUT);
  for(i=0; i<3; i++)   // Clean relay contacts
      {
      setRefR(R5K);
      delay(10);
      setRefR(R50);
      delay(10);
      }
  // For the FET switch, select, at most, one of the following three:
  digitalWrite(CAL_39, HIGH);       // On
  digitalWrite(IMPEDANCE_38, LOW);  // Off
  digitalWrite(TRANSMISSION_37, LOW);
  panelLED(LGREEN);
  Serial.println("VNA-LF Ver 0.58");
  Serial.print("Voltage Check (expect 145 to 175): ");
  Serial.println(analogRead(21));

  // Each additional AudioMemory uses 256 bytes of RAM (dynamic memory).
  // This stores 128-16-bit ints.   Dec 16 usage 8
  AudioMemory(30);   // Last seen peaking at 17

  AudioNoInterrupts();     // Use to synchronize all audio
  // Enable the SGTL5000 audio CODEC
  audioShield.enable();
  // Headphone volume, 0.0 to 0.8 is undistorted over full input ADC range
  audioShield.volume(0.0);
  /* Adjust the sensitivity/gain  of the line-level inputs, same foor both.
    Fifteen settings are possible for  lineInLevel(both), in 1.5 dB steps:
    0: 3.12 Volts p-p  0 dB gain  (pair with output 13 for net zero dB)
    1: 2.63 Volts p-p                               16
    2: 2.22 Volts p-p                               19
    3: 1.87 Volts p-p                               22
    4: 1.58 Volts p-p                               25
    5: 1.33 Volts p-p  (default)                    28
    6: 1.11 Volts p-p
    7: 0.94 Volts p-p  More gain
    8: 0.79 Volts p-p   is lower
    9: 0.67 Volts p-p     |
    10: 0.56 Volts p-p     |
    11: 0.48 Volts p-p     V
    12: 0.40 Volts p-p
    13: 0.34 Volts p-p
    14: 0.29 Volts p-p
    15: 0.24 Volts p-p
    For output, 19 sets 2.2 v p-p range.
  */
  audioShield.lineInLevel(1, 1);    // both 5 for full bridge
  audioShield.lineOutLevel(29);  // 29 lowers the output to not overdrive the amp
  // Next, see https://forum.pjrc.com/threads/27215-24-bit-audio-boards?p=78831&viewfull=1#post78831
  audioShield.adcHighPassFilterDisable();
  waveform1.begin(WAVEFORM_SINE);
  waveform1.frequency(factorFreq * vnaF[nFreq].freqHz);
  waveform1.amplitude(1.0);
  waveform1.phase(0);
  waveform2.begin(WAVEFORM_SINE);
  waveform2.frequency(factorFreq * vnaF[nFreq].freqHz);
  waveform2.amplitude(1.0);
  waveform2.phase(270);
  firIn1.begin(lp2300_100K, 100);
  firIn2.begin(lp2300_100K, 100);
  AudioInterrupts();
  DC1.amplitude(dacLevel);                          // Gain control for DAC output
  //factorFreq is global that corrects any call involving absolute frequency, like waveform generation.
  factorFreq = FBASE / sampleRateExact;
  // Setup callbacks for SerialCommand commands
  SCmd.addCommand("ZMEAS", ZmeasCommand);
  SCmd.addCommand("Z", ZmeasCommand);
  SCmd.addCommand("TRANSMISSION", TransmissionCommand);
  SCmd.addCommand("T", TransmissionCommand);
  SCmd.addCommand("FREQ", FreqCommand);
  SCmd.addCommand("F", FreqCommand);
  SCmd.addCommand("SWEEP", SweepCommand);
  SCmd.addCommand("CAL", CalCommand);
  SCmd.addCommand("C", CalCommand);
  SCmd.addCommand("RUN", RunCommand);
  SCmd.addCommand("R", RunCommand);
  SCmd.addCommand("POWER", PowerSweepCommand);
  SCmd.addCommand("SAVE", saveStateEEPROM);
  SCmd.addCommand("LOAD", loadStateEEPROM);
  SCmd.addCommand("DELAY", DelayCommand);
  SCmd.addCommand("CALDAT", CalDatCommand);
  SCmd.addCommand("SERPAR", SerParCommand);
  SCmd.addCommand("TEST", TestCommand);
  SCmd.addCommand("PARAM1", Param1Command);   // R50, R5K values and also use defaults
  SCmd.addCommand("PARAM2", Param2Command);   // Impedance correction factors`
  SCmd.addCommand("TUNEUP", TuneupCommand);   // Estimate strays 
  SCmd.addCommand("BAUD", BaudCommand);       // Change serial baud rate
  SCmd.addCommand("ANNOTATE", AnnotateCommand);
  SCmd.addCommand("VERBOSE", VerboseCommand);
  SCmd.addDefaultHandler(unrecognized);         // Handler for command that isn't matched
  // Response variations:  ECHO_FULL_COMMAND replies with full received line, even if command is not valid.
  // ECHO_COMMAND ((first token only) and ECHO_OK only respond if command is valid.  With EOL.
  // ECHO_ONE responds with numerical 1 if command is valid and numerical 0 if invalid.  No EOL.
  // NO_RESPONSE  does that.
  SCmd.setResponse(ECHO_FULL_COMMAND);
  VNAMode = 1;
  uSave.lastState.msDelay = 700;   // Needs adjustable param
  Serial.println("");
  tms = millis();
  }

// =====================================  LOOP  ============================================
void loop()
  {
  uint16_t menuItem;

  SCmd.readSerial();     // Process serial commands
  // Following if's will execute each loop() and are for touch control
  if(avnaState == 0)     // Idle
     {
     ;
     }
  else if(avnaState==ZSINGLE)
     {
     // This is for LCD control, not serial.  Always repetitive measurements until
     // stopped by a touch entry.
     setRefR(uSave.lastState.iRefR);    // Select relay for reference resistor
     setSwitch(IMPEDANCE_38);           // Connect for Z measure
     doZSingle(true);
     delay(1000);
     }
  else if(avnaState == TSINGLE)         // T Meas 1 Freq
     {
     setRefR(uSave.lastState.iRefR);    // Select relay for reference resistor
     setSwitch(TRANSMISSION_37);        // Connect for Z measure
     doTSingle(true);
     delay(1000);
     }
  else if(avnaState == ZSWEEP)
     {
     ;  // These are empty at this point.
     }
  else if(avnaState == TSWEEP)
     {
     ;
     }
  else if(avnaState == WHATSIT)      //What Component
     {
     doWhatPart();
     avnaState = IDLE;               // Force only one run
     }
  else if(avnaState == LCDHELP)
     {
     ; // No processing required
     }
 
  boolean istouched = ts.touched();
  if (istouched  &&  (millis()-tms) > 150)           // Set repeat rate and de-bounce or T_REPEAT
    {
    avnaState = 0;     // Stop measurements
    doRun = RUNNOT;       // Stop measurements
#if DIAGNOSTICS
    Serial.println(millis()-tms);
    tms = millis();
#endif
    TS_Point p = ts.getPoint();
    if(p.y > 3200)
       {
       menuItem = floor(p.x/642);                        // Make 3200 and 642 compilable <<<<<<
       // Act on menu item.  t is an 2-D array of pointers to functions to do what is needed
       // based on currentMenu and menuItem
       (*t[currentMenu][menuItem]) ();
       
       currentMenu = lcdMenu[currentMenu][menuItem];   // Select new menu ( or leave the same)
       writeMenus(currentMenu);                // Show the new menu
       }
    }    // end, screen was touched
  wastouched = istouched;
  delay(100);

  if(doRun == POWER_SWEEP)
     {
     dacSweep *= 1.04;
     dacLevel = dacSweep;
     if(dacLevel > 0.5)
        {
        doRun = RUNNOT;         // Pack up like we were never here
        dacLevel = 0.4;
        }
     delay(10);
     }

 if(doRun != RUNNOT && doRun != POWER_SWEEP)
   {
   if(teststate == 1)
     ;
   else     // Not test, response for serial control
     {
     if (nRun == 0)  // Counted down, time to stop   // REARRANGE 
        doRun = RUNNOT;
     else if (doRun == COUNTDOWN)
        nRun--;
     // Sort out Sweep or Single;  Impedance or Transmission
     if (doRun == COUNTDOWN || doRun == CONTINUOUS || doRun ==SINGLE_NC)
        {
        if (uSave.lastState.ZorT == IMPEDANCE)   // Measure Z
           {
           setRefR(uSave.lastState.iRefR);    // Select relay for reference resistor
           setSwitch(IMPEDANCE_38);
           if (uSave.lastState.SingleorSweep == SWEEP)
              {
              doZSweep(true, 1);     //  Do a impedance measurement with sweep
              if(useLCD)
                 LCDPrintSweepZ();
              }
          else
              {
              doZSingle(true);
              }
           }
        else                      // TRANSMISSION
           {
           setRefR(uSave.lastState.iRefR);   // Set relay for Ref R
           setSwitch(TRANSMISSION_37);
           if (uSave.lastState.SingleorSweep == SWEEP)
              doTSweep(true);     //  Do a transmission measurement with sweep
           else
              doTSingle(true);
              if(doRun==SINGLE_NC) doRun = RUNNOT;
           }
        delay(uSave.lastState.msDelay);       // Slow down, if desired

        }      // End repetitive loop
     }  // End not test
  }
#if DIAGNOSTICS
  Serial.print("Proc = ");
  Serial.print(AudioProcessorUsage());
  Serial.print(" (");
  Serial.print(AudioProcessorUsageMax());
  Serial.print("),  Mem = ");
  Serial.print(AudioMemoryUsage());
  Serial.print(" (");
  Serial.print(AudioMemoryUsageMax());
  Serial.println(")");
#endif
  }             // End loop()

// ------------------- Touch LCD Menu Response Functions  --------------------------
void tNothing(void)
  {
  // Typically here from a blank menu item.
  }

void tToHome(void)
  {
  avnaState = 0;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 60);
  tft.print("                    Welcome!");
  tft.setCursor(0, 82);
  tft.print("  Select from the bottom menu items.");
  dataValidTSweep = false;
  dataValidZSweep = false;
  clearStatus();
  }

// This may be extreme, but does the job.  Clears reference to data
// that may be out of date.
void clearStatus(void)
  {
  dataValidTSweep = false;
  dataValidZSweep = false;
  calZSingle = false;
  calTSingle = false;
  calZSweep = false;
  calTSweep = false;
  }

void tDoSingleFreq(void)
  {
  avnaState = 0;
  tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_14);
  tft.setCursor(20, 62);
  tft.print("Single Frequency Measurements");
  clearStatus();
  }

void tCalCommand(void)
  {
  tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(20, 60);
  tft.print("Wait for Cal to complete...");
  CalCommand();
  tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
  // After cal, but before measurement, data is no longer valid for display
  dataValidTSweep = false;
  dataValidZSweep = false;
  }

void tDoSweep(void)
  {
  avnaState = 0;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_14);
  tft.setCursor(20, 62);
  tft.print("Swept 13-Freq Measurements");
  displaySweepRange();
  }

// Intro screen for What? function
void tWhatIntro(void)
  {
  lcdTitle("What?");
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_11);
  tft.setCursor(10, 50);
  tft.print("         WHAT?  Component Identification");
  tft.setCursor(10, 80);
  tft.print("Component impedance will be measured at");
  tft.setCursor(10, 102);
  tft.print("13 frequencies and 2 impedance levels.");
  tft.setCursor(10, 124);
  tft.print("Based on these, values are estimated.");
  tft.setCursor(10, 146);
  tft.print("Omit 3 lowest frequencies for faster estimates.");
  tft.setCursor(10, 168);
  tft.print("Connect component and touch \"Search Value\"");
  }

// tDoWhatPart() is response to touching LCD.  The next function doWhatPart()
// is called from loop() to keep menus correct.
void tDoWhatPart(void)
  {
  avnaState=WHATSIT;
  }

// Does two Z sweeps, one at 50 Ohms and one at 5KOhm.
// Chooses frequencies with Z magnitude closest to 50 or 5K
// and prints part values.  Q>1.0 are L & C, otherwise R's.
void doWhatPart(void) 
  {
  uint16_t i, nBest, baseIndex, iRefRSave, saveZT, saveSS, savenFreq;
  double ZM, ZMBest, zDiff;
  double XX, RR;

  // Leave this function with the sate restored
  iRefRSave = uSave.lastState.iRefR;
  saveSS = uSave.lastState.SingleorSweep;
  saveZT = uSave.lastState.ZorT;
  savenFreq = nFreq;

  if(whatUseLF)  baseIndex = 1;
  else           baseIndex = 4;
  
  // avnaState = 5;
  tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 60);
  tft.print("     WAIT  -  Searching out component Z");
  uSave.lastState.iRefR = R50;
  setRefR(R50);            // Search out 50 Ohms first
  uSave.lastState.SingleorSweep = SWEEP;
  uSave.lastState.ZorT = IMPEDANCE;
  CalCommand0(baseIndex);
  setSwitch(IMPEDANCE_38);
  doZSweep(false, baseIndex);              // Result in Z[nf]
  nBest = 1;
  zDiff = 1E9;
  ZMBest = 0.0;
  for(i=baseIndex; i<=13; i++)
     {
     RR = Z[i].real();
     XX = Z[i].imag();
     ZM = sqrt(RR*RR + XX*XX);
     if(fabs(50.0 - ZM) < zDiff)
        {
        zDiff = fabs(50.0 - ZM);
        nBest = i;
        ZMBest = ZM;
        }
     }

    tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
    tft.setCursor(10, 40);
    tft.setFont(Arial_14);
    tft.print("Ref=50 Ohm, At Freq = ");
    if     (vnaF[nBest].freqHz < 100.0)
       tft.print(vnaF[nBest].freqHz, 2);
    else if(vnaF[nBest].freqHz < 1000.0)
       tft.print(vnaF[nBest].freqHz, 1);
    else    //  Must be 1000 or more
       tft.print(floor(0.5+vnaF[nBest].freqHz), 0);
    tft.print(" Hz:");

    RR = Z[nBest].real();
    XX = Z[nBest].imag();
    tft.setFont(Arial_18);
    tft.setCursor(20, 67);
    if(XX<=0.0 && (Q[nBest]>1.0 || Q[nBest]<0.0))          // Call it a capacitor
       {
       tft.print("C=");
       tft.print(valueString(sLC[nBest], cUnits));
       tft.print("  Q=");
       // Serial.print shows negative Q's as a number, +9999.9.  Here we can be less terse:
       if(Q[nBest]<9999.8 && Q[nBest]>1.0)
          tft.print(Q[nBest], 1);
       else
          tft.print(" High");
       }
    else if(XX>0.0 && (Q[nBest]>1.0 || Q[nBest]<0.0))      // Call it an inductor
       {
       tft.print("L=");
       tft.print(valueString(sLC[nBest], lUnits));
       tft.print("  Q=");
       if(Q[nBest] < 9999.8 && Q[nBest]>1.0)
          tft.print(Q[nBest], 1);
       else
          tft.print(" High");
       }
    else     // Low Q, show as a resistor with series L or C
       {
       tft.print("R=");
       tft.print(valueStringSign(RR, rUnits));
       if(XX > 0.0)
          {
          tft.print(" L=");
          tft.print(valueString(sLC[nBest], lUnits));
          }
       else
          {
          tft.print(" C=");
          tft.print(valueString(sLC[nBest], cUnits));
          }
        }
     printQuality(ZMBest / 50.0);

  // This is mostly duplicate stuff of 50 Ohm---make separate function <<<<<<<<<<<
  tft.setFont(Arial_12);
  tft.setCursor(10, 140);
  tft.print("  WAIT - Searching out Z for 5K Ref");
  uSave.lastState.iRefR = R5K;
  setRefR(R5K);            // Search out 5K Ohms

  CalCommand0(baseIndex);
  setSwitch(IMPEDANCE_38);
  doZSweep(false, baseIndex);              // Result in Z[nf]
  nBest = 1;
  zDiff = 1E9;
  for(i=baseIndex; i<=13; i++)
     {
     RR = Z[i].real();
     XX = Z[i].imag();
     ZM = sqrt(RR*RR + XX*XX);
     if(fabs(5000.0 - ZM) < zDiff)
        {
        zDiff = fabs(5000.0 - ZM);
        nBest = i;
        ZMBest = ZM;
        }
     }
    tft.fillRect(0, 100, tft.width(), 84, ILI9341_BLACK);
    tft.setCursor(10, 120);
    tft.setFont(Arial_14);
    tft.print("Ref=5K Ohm, At Freq = ");
    if     (vnaF[nBest].freqHz < 100.0)
       tft.print(vnaF[nBest].freqHz, 2);
    else if(vnaF[nBest].freqHz < 1000.0)
       tft.print(vnaF[nBest].freqHz, 1);
    else    //  Must be 1000 or more
       tft.print(floor(0.5+vnaF[nBest].freqHz), 0);
    tft.print(" Hz:");

    RR = Z[nBest].real();
    XX = Z[nBest].imag();
    
    tft.setFont(Arial_18);
    tft.setCursor(20, 147);

    if(XX<=0.0 && (Q[nBest]>1.0 || Q[nBest]<0.0))          // Call it a capacitor
       {
       tft.print("C=");
       tft.print(valueString(sLC[nBest], cUnits));
       tft.print("  Q=");
       // Serial.print shows negative Q's as a number, +9999.9.  Here we can be less terse:
       if(Q[nBest] < 9999.8 && Q[nBest]>1.0)
          tft.print(Q[nBest], 1);
       else
          tft.print(" High");
       }
    else if(XX>0.0 && (Q[nBest]>1.0 || Q[nBest]<0.0))      // Call it an inductor
       {
       tft.print("L=");
       tft.print(valueString(sLC[nBest], lUnits));
       tft.print("  Q=");
       if(Q[nBest] < 9999.8 && Q[nBest]>1.0)
          tft.print(Q[nBest], 1);
       else
          tft.print(" High");
       }
    else     // Low Q, show as a resistor with series L or C
       {
       tft.print("R=");
       tft.print(valueStringSign(RR, rUnits));
       if(XX > 0.0)
          {
          tft.print(" L=");
          tft.print(valueString(sLC[nBest], lUnits));
          }
       else
          {
          tft.print(" C=");
          tft.print(valueString(sLC[nBest], cUnits));
          }
        }
  printQuality(ZMBest / 5000.0);

  // Leave this function with the sate restored
  uSave.lastState.SingleorSweep = saveSS;
  uSave.lastState.ZorT = saveZT;
  nFreq = savenFreq;
  uSave.lastState.iRefR = iRefRSave;
  setRefR(uSave.lastState.iRefR);
  }

// Simple measurement quality estimate. Ratio is impedance/refR.
void printQuality(double ratio)
  {
  if(ratio < 1.0)   ratio = 1.0 / ratio;
  if(ratio < 10.0)        tft.print("  E");
  else if(ratio < 100.0)  tft.print("  G");
  else if(ratio < 1000.0) tft.print("  P");
  else                    tft.print("  0");
  }

// Low frequencies of 10, 20 and 50 Hz are especially slow to execute
// Z measurements.  For What? function, their use is optional.
void tUseLF(void)
  {
  whatUseLF = true;
  tft.fillRect(80, 185, tft.width(), 14, ILI9341_BLACK);
  }
void tNoLF(void)
  {
  whatUseLF = false;
  tft.fillRect(80, 185, tft.width()-80, 14, ILI9341_BLACK);
    tft.setTextColor(ILI9341_WHITE);
    tft.setFont(Arial_8);
    tft.setCursor(145, 185);
    tft.print("Not Using 10, 20 or 50 Hz");
  }

void tDoSettings(void)
  {
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  
  tft.setFont(Arial_12);
  tft.setCursor(0, 60);
  tft.print("   Currently, Ref R = "); tft.print(uSave.lastState.valueRRef[uSave.lastState.iRefR]);
          tft.print(" Ohms");
  tft.setCursor(0, 90);
  tft.print("  Select \"50\" or \"5K\" Ref R on menu below");
  }

void tSet50(void)
  {
  sRefR(R50);
  tft.print("  Ref R = 50 Ohms");
  }

void tSet5K(void)
  {
  sRefR(R5K);
  tft.print("  Ref R = 5000 Ohms");
  }

// All the common stuff for setting the reference resistance
void sRefR(int16_t indexRef)
  {
  uSave.lastState.iRefR = indexRef;   // Where we keep track of RefR
  setRefR(indexRef);           // Set relay
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 100);
  tft.print(" Select \"Back\" for Main Menu");
  topLines();
  saveStateEEPROM();
  clearStatus();
  // Get ready for printing "Ref R = 50 Ohms" etc
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_24);
  tft.setCursor(0, 60);
  }

void tDoHelp(void)
  {
  avnaState = 6;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 38);
  tft.print("HELP-Fixed Freq Measurements, Z or T");
  tft.setCursor(0, 62);
  tft.setFont(Arial_11);
  tft.print("Touch \"Single Freq\", select freq with 3 buttons");
  tft.setCursor(0, 86);
  tft.print("Touch \"Set Ref R\" and choose 50 or 5K Ohm");
  tft.setCursor(0,110);
  tft.print("Select \"Meas Z\" or T");
  tft.setCursor(0, 134);
  tft.print("Cal for Z is automatic, T is manual.");
  tft.setCursor(0, 158);
  tft.print("Measurements will repeat. Return with \"Back\"");
  tft.setCursor(0, 182);
  tft.print("HELP-What? Touch \"What?\"; follow instructions.");
  tft.setFont(Arial_12);
  }
void tDoHelp2(void)
  {
  avnaState = 6;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 38);
  tft.print("HELP-Swept Freq Measurements, Z & T");
  tft.setFont(Arial_11);
  tft.setCursor(0, 62);
  tft.print("Touch \"Set Ref R\" and choose 50 or 5K Ohm.");
  tft.setCursor(0, 86);
  tft.print("Touch \"Back\" and then \"Sweep\".");
  tft.setCursor(0, 110);
  tft.print("Touch \"Meas Z\" or \"Meas T\".");
  tft.setCursor(0, 134);
  tft.print("Z measurements proceed but T requires");
  tft.setCursor(0, 158);
  tft.print("  Cal. Follow prompts.");
  tft.setCursor(0, 182);
  tft.print("Use \"Disp Freq\" to scroll freq range up/down.");
  tft.setFont(Arial_12);
  }

void tFreqUp(void)
  {
  //Up if not at top, print top line
  if(nFreq==0)
     nFreq = 8;
  else if(nFreq<13)
     nFreq++;
  finish_tFreq();
  }
  
void tFreqDown(void)
  {
  if(nFreq==0)
     nFreq = 6;
  else if(nFreq>1)
     nFreq--;
  finish_tFreq();
  }
  
void tFreq0(void)
  {
  nFreq = 0;
  setUpNewFreq(nFreq);
  finish_tFreq();
  clearStatus();
  }

void finish_tFreq(void)
  {
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_24);
  tft.setCursor(0, 60);
  tft.print("Freq = ");  tft.print(vnaF[nFreq].freqHz, 0); tft.print(" Hz");
  saveStateEEPROM();
  topLines();
  calZSingle = false;
  calTSingle = false;
  }

void tDoSingleZ(void)
  {
  setRefR(uSave.lastState.iRefR);
  lcdTitle("Single Freq Z");
  uSave.lastState.SingleorSweep = SINGLE;
  uSave.lastState.ZorT = IMPEDANCE;
  if(calZSingle == false)
    {
    tCalCommand();
    calZSingle = true;
    }
  // Actual measurement will occur in loop().  Here we enable continuous measurement
  avnaState = ZSINGLE;
  }

void lcdTitle(char const *title)
  {
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_11);
  tft.setCursor(10, 185);
  tft.print(title);
  }

void tDoSingleT(void)
  {
  lcdTitle("Single Freq T");
  uSave.lastState.SingleorSweep = SINGLE;
  uSave.lastState.ZorT = TRANSMISSION;
  if(calTSingle == false)
    {
    tft.setTextColor(ILI9341_RED);
    tft.setFont(Arial_14);
    tft.setCursor(20, 100);
    tft.print("T - Single Freq  NEEDS CAL");
    tft.setTextColor(ILI9341_YELLOW);
    tft.setFont(Arial_12);
    tft.setCursor(20, 130);
    tft.print("Connect reference through path");
    tft.setCursor(20, 152);
    tft.print("and hit Cal below.");
    }
  // Actual measurement will occur in loop().  Here we enable continuous measurement
  avnaState = TSINGLE;
  }

void tSetParams(void)
  {
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
 // tft.setTextColor(ILI9341_YELLOW);
  //tft.setFont(Arial_24);
  //tft.setCursor(0, 60);
 // tft.print(" 1 Z ");
  }

void writeMenus(uint16_t nMenu)
  {
  uint16_t i;
  // Room for 9 menu sets of 6 menus each, and each menu has up to 8 chars + 0.
  static char line1[11][6][9]={ 
    " ", " Single", " Sweep", "  What", "  Set", " Help",       // Set 0 
    " Back", "  Freq", "  Freq", "  Freq", " Meas", " Meas ",   // Set 1
    " Back", "Disp Frq", "Disp Frq", " ", " Meas", " Meas",     // Set 2
    " Back", "    50", "    5K", " ", " ", " ",                 // Set 3
    " Back", " 50 Ohm", " 5K Ohm", " Power", " Power", " ",     // Set 4
    " Back", " Screen", " Screen", " ", " ", " ",               // Set 5
    " Back", " Screen", " Screen", " ", " ", " ",               // Set 6 Help
    " Back", " ", " ", "  No 10,", " Use 10,", " Search",       // Set 7 What?
    " Back", " ", " ", " ", "  Cal", " Meas",                   // Set 8
    " Back", "Disp Frq", "Disp Frq", " ", "  Cal", " Single",   // Set 9
    " Back", "Disp Frq", "Disp Frq", " ", "  Cal", " Single"    // Set 9
    };
  static char line2[11][6][9]={
    " ", "  Freq", " ", "   ?", " Ref R", " ",
    " ", "  Down", "   Up", "    0", "    Z", "    T",
    " ", "  Down", "   Up", " ", "    Z", "    T",
    " ", "  Ohms", "   Ohms ", " ", " ", " ",
    " ", " ", " ", "    Up", " Down", " ",
    " ", "    1", "    2", " ", " ", " ",
    " ", "    1", "    2", " ", " ", " ",
    " ", " ", " ", "20 or 50", "20 & 50", " Value",
    " ", " ", " ", " ", " ", "    T",
    " ", "  Down", "   Up", " ", " ", "T Sweep",
    " ", "  Down", "   Up", " ", " ", "Z Sweep"
    };

    tft.fillRect(0, 200, tft.width(), 39, ILI9341_BLACK);
  for(i=0; i<6; i++)
    {
    tft.drawRect(53*i, 200, 51, 39, ILI9341_GREEN);
    tft.setTextColor(ILI9341_WHITE);
    tft.setFont(Arial_8);
    tft.setCursor(53*i+5, 209);
    tft.print(&line1[nMenu][i][0]);
    tft.setCursor(53*i+5, 222);
    tft.print(&line2[nMenu][i][0]);    
    }
  }

/* Setup and cal info is saved to a block, starting at address 256
    in EEPROM.  This is the byte array save_vnaF[].  Works out
    to 14 x 64 = 896 bytes.
*/
void saveStateEEPROM(void)
  {
  uint16_t i;
  for (i = 0; i < sizeof(saveState); i++)
     EEPROM.write(i, uSave.save_vnaF[i]);
  }

// Reverse to read state from EEPROM.
void loadStateEEPROM(void)
  {
  uint16_t i;
  for (i = 0; i < sizeof(saveState); i++)
     uSave.save_vnaF[i] = EEPROM.read(i);   // Byte at a time
  }

// doZSingle() measures impedance at a single frequency. The calling
// parameter, out sets Serial.print.  nFreq sets the frequency.
void doZSingle(bool out)
  {
  // Check that cal done, settings OK
  if (!calZSingle && !(doRun==SINGLE_NC))
     {
     Serial.println("Single frequency measurements without Cal---use CAL or C");
     doRun = RUNNOT;
     return;
     }
  DC1.amplitude(dacLevel);     // Turn on sine wave
  setUpNewFreq(nFreq);
  delay(ZDELAY);   // Delay until level is constant
  measureZ(nFreq);
  // if (useUSB)
  serialPrintZ(nFreq);
  // if(useLCD)
  LCDPrintSingleZ(nFreq);
  }

// doZSweep() measures impedance at all sweep frequencies. The calling
// parameter, out, sets Serial.print.
void doZSweep(bool out, uint16_t iStart)
  {
  if (!calZSweep)
    {
    if(out)
       Serial.println("Sweep cannot run without Cal---use CAL or C");
    doRun = RUNNOT;
    return;
    }
  // Check that cal done, settings OK
  DC1.amplitude(dacLevel);     // Turn on sine wave
  // Add column headings if not annotated
  if(!annotate && seriesRX && !parallelRX)
    Serial.println(" Freq,  R,  X,  L or C,  Q");
  if(!annotate && parallelRX && !seriesRX)
    Serial.println(" Freq,  G,  B,  Res,  L or C,  Q");
  for (nFreq = iStart; nFreq <= 13;  nFreq++) // Over all frequencies; nFreq is global freq index
    {
    setUpNewFreq(nFreq);
    delay(ZDELAY);        // Delay until level is constant
    measureZ(nFreq);      // result is Z[nFreq]
    serialPrintZ(nFreq);
    }
  nFreq = 0;    // Don't leave at 14!
  setUpNewFreq(nFreq);
  }

// This is for touch-LCD controlled sweep. It is a single sweep, commanded
// by "Single Sweep" on touch LCD. The display is separate, as it needs to change
/// withthe displayed frequencies. 
void tDoSweepZ(void)                        // Impedance Sweep
  {
  setRefR(uSave.lastState.iRefR);
  lcdTitle("Sweep Z");
  tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
  uSave.lastState.SingleorSweep = SWEEP;
  uSave.lastState.ZorT = IMPEDANCE;
  if(calZSweep == false)
     {
     tft.setTextColor(ILI9341_YELLOW);
     tft.setFont(Arial_12);
     tft.setCursor(20, 60);
     tft.print("Wait - Doing Z-Sweep Cal");
     CalCommand();
     }
  avnaState = ZSWEEP;
  if(calZSweep)
     {
     DC1.amplitude(dacLevel);     // Turn on sine wave   <<<<<<<<<MAKE PROGRAMMABLE
     setSwitch(IMPEDANCE_38);
     tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
     tft.setTextColor(ILI9341_YELLOW);
     tft.setFont(Arial_12);
     tft.setCursor(20, 60);
     tft.print("Wait - Doing Z-Sweep Measure");
     for (nFreq = 1; nFreq <= 13;  nFreq++) // Over all frequencies; nFreq is global freq index
        {
        setUpNewFreq(nFreq);
        delay(ZDELAY + (unsigned long)(1000.0 / vnaF[nFreq].freqHz));
        measureZ(nFreq);   // result is Z[nFreq], Y[nFreq], sLC[nFreq], pLC[nFreq] and Q[nFreq]
        }
     nFreq = 0;
     setUpNewFreq(nFreq);
     tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
     dataValidZSweep = true;
     display7Z();
     }
  }

// Display 7 lines of impedance data on LCD
void display7Z(void)
  {
  uint16_t i, nfr;
  double XX, RR, ZM;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(20, 38);
  tft.print("Freq        R    +j    X           L-C");
  if(dataValidZSweep)
     for(i=0; i<7; i++)
        {
        nfr = i + nFreqDisplay;
        RR = Z[nfr].real();
        XX = Z[nfr].imag();
        ZM = sqrt(RR*RR + XX*XX);
        tft.setCursor(0, 57+19*i);
        if     (vnaF[nfr].freqHz < 100.0)
           tft.print(vnaF[nfr].freqHz, 2);
        else if(vnaF[nfr].freqHz < 1000.0)
           tft.print(vnaF[nfr].freqHz, 1);
        else    //  Must be 1000 or more
           tft.print(floor(0.5+vnaF[nfr].freqHz), 0);
        tft.print("Hz");
        tft.setCursor(70, 57+19*i);
        if(fabs(RR)<10.0)
           tft.print(RR, 3);
        else if(fabs(RR)<100.0)
           tft.print(RR, 2);
        else if(fabs(RR)<1000.0)
           tft.print(RR, 1);
        else
           tft.print(RR, 0);
        tft.setCursor(155, 57+19*i);
        if(fabs(XX)<10.0)
           tft.print(XX, 3);
        else if(fabs(XX) < 100.0)
           tft.print(XX, 2);
        else if(fabs(XX) <1000.0)
           tft.print(XX, 1);
        else
           tft.print(XX, 0);
        tft.setCursor(220, 57+19*i);
        if(XX<=0.0 && (Q[nfr]>1.0 || Q[nfr]<0.0))          // Call it a capacitor
           {
           tft.print(valueString(sLC[nfr], cUnits));
           }
        else if(XX>0.0 && (Q[nfr]>1.0 || Q[nfr]<0.0))      // Call it an inductor
           {
           tft.print(valueString(sLC[nfr], lUnits));
           }
        else     // Low Q, show as a resistor with series L or C
           {
           if(XX > 0.0)
              {
              tft.print(valueString(sLC[nfr], lUnits));
              }
           else
              {
              tft.print(valueString(sLC[nfr], cUnits));
              }
           }
        if(uSave.lastState.iRefR == R5K)      
            printQuality(ZM / 5000.0);
        else
            printQuality(ZM / 50.0);       
        }
  }

// =========================  TRANSMISSION  =========================

// doTSingle() measures transmission at single frequency, set by global nFreq.
void doTSingle(bool out)
  {
  if (!calTSingle && out)
    {
    Serial.println("Single frequency measurements without Cal---use CAL or C");
    doRun = RUNNOT;
    return;
    }
  DC1.amplitude(dacLevel);     // Turn on sine wave
  setUpNewFreq(nFreq);
  // Delay until level is constant
  delay(ZDELAY + (unsigned long)(1000.0 / vnaF[nFreq].freqHz));
  measureT();      // result is Tmeas
  T[nFreq] = Tmeas;
  
  LCDPrintSingleT(nFreq);
  }

// doTSweep() measures transmission at all sweep frequencies. The calling
// parameter, out sets Serial.print.  This is for serial controlled sweep.
void doTSweep(bool out)
  {
  // Check that cal done, settings OK
  if (!calTSweep)
    {
    if(out)
       Serial.println("Sweep cannot run without Cal---use CAL or C");
    doRun = RUNNOT;
    return;
    }
  DC1.amplitude(dacLevel);     // Turn on sine wave   <<<<<<<<<MAKE PROGRAMMABLE
  for (nFreq = 1; nFreq <= 13;  nFreq++) // Over all frequencies; nFreq is global freq index
    {
    setUpNewFreq(nFreq);
    delay(ZDELAY + (unsigned long)(1000.0 / vnaF[nFreq].freqHz));
    measureT();      // result is Tmeas
    T[nFreq] = Tmeas;
    }
  }

// This is for touch-LCD controlled sweep. It is a single sweep, commanded
// by "Single Sweep" on touch LCD. The display is separate, as it needs to change
/// withthe displayed frequencies. 
void tDoSweepT(void)
  {
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  uSave.lastState.SingleorSweep = SWEEP;
  uSave.lastState.ZorT = TRANSMISSION;
  if(calTSweep == false)
    {
    tft.setTextColor(ILI9341_RED);
    tft.setFont(Arial_14);
    tft.setCursor(20, 100);
    tft.print("T - Sweep 13-Freq  NEEDS CAL");
    tft.setTextColor(ILI9341_YELLOW);
    tft.setFont(Arial_12);
    tft.setCursor(20, 130);
    tft.print("Connect reference through path");
    tft.setCursor(20, 152);
    tft.print("and hit Cal below.");
    }
  avnaState = TSWEEP;
  if(calTSweep)
     {
     DC1.amplitude(dacLevel);     // Turn on sine wave  <<<MAKE PROGRAMMABLE
     tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
     tft.setTextColor(ILI9341_YELLOW);
     tft.setFont(Arial_12);
     tft.setCursor(20, 60);
     tft.print("Wait for Sweep to complete...");
     for (nFreq = 1; nFreq <= 13;  nFreq++) // Over all frequencies; nFreq is global freq index
        {
        setUpNewFreq(nFreq);
        delay(ZDELAY + (unsigned long)(1000.0 / vnaF[nFreq].freqHz));
        measureT();      // result is Tmeas
        T[nFreq] = Tmeas;
        }
     tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
     dataValidTSweep = true;
     display7T();
     nFreq = 0;  // Back in range
     setUpNewFreq(nFreq);
     }
  }
// Display 7 lines of transmission data on LCD
void display7T(void)
  {
  uint16_t i, nfr;
  double XT, RT, MT, PT, MTdB;
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  if(dataValidTSweep)
     for(i=0; i<7; i++)
        {
        nfr = i + nFreqDisplay;
        RT = T[nfr].real();
        XT = T[nfr].imag();
        MT = sqrt(RT*RT + XT*XT);
        MTdB = 20.0 * log10(MT);
        PT = r2d(atan2(XT, RT));
        tft.setCursor(10, 43+19*i);
        tft.print(vnaF[nfr].freqHz, 0);   tft.print(" Hz");
        tft.setCursor(95, 43+19*i);
        tft.print("G=");  tft.print(MTdB);  tft.print("dB");
        tft.setCursor(190, 43+19*i);
        tft.print("Ph=");
        tft.print(PT);  tft.print("Deg");
        }
  }

void tSweepFreqDown(void)
  {
  if((nFreqDisplay--) <= 1)
      nFreqDisplay++;
  if     (dataValidTSweep)
     display7T();
  else if(dataValidZSweep)
     display7Z();
  else
     displaySweepRange();
  }

void tSweepFreqUp(void)
  {
  if((nFreqDisplay++) >= 7)
      nFreqDisplay--;
  if     (dataValidTSweep)
     display7T();
  else if(dataValidZSweep)
     display7Z();
  else
     displaySweepRange();
  }

void displaySweepRange(void)
  {
  tft.fillRect(0, 140, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(20, 141);
  tft.print("Display ");   tft.print(vnaF[nFreqDisplay].freqHz, 0);   tft.print(" to ");
  tft.print(vnaF[6 + nFreqDisplay].freqHz, 0);   tft.print(" Hz");
  }

//===========================  IMPEDANCE  ============================

// measureZ does a single impedance data point, including applying CAL.
// Freq needs to be setup before calling, but the frequency index is iF.
// No delays for settling. No print.  Complements measureT.
// Results are global Z[iF], Y[iF], sLC[iF], pLC[iF] and Q[iF]
void measureZ(uint16_t iF)
  {
  Complex Yin(0.0, 0.0);
  Complex Z22(0.0, 0.0);
  Complex Yi(0.0, 0.0);
  Complex Yinput(0.0, 0.0);
  Complex Zmeas(0.0, 0.0);
  Complex Ymeas(0.0, 0.0);
  Complex Vm(0.0, 0.0);
  Complex Vr(0.0, 0.0);
  double w, ZM;

  w = 6.28318530718 * vnaF[nFreq].freqHz;
  getFullDataPt();    // Gets amplitudes and phases
  checkOverload();
  // Do corrections for analog amplifier errors, vRatio and dPhase:
  Vm = polard2rect(amplitudeV * vnaF[nFreq].vRatio, phaseV + vnaF[nFreq].dPhase);
  Vr = polard2rect(amplitudeR, phaseR);
  // The next Complex(,) is the reference resistor value, set up to be changed and made sticky
  Zmeas = Complex(uSave.lastState.valueRRef[uSave.lastState.iRefR], 0.0) * Vm / (Vr - Vm);

  // The inpuput capacity and the input to U1D needs de-embedding.  This consists of 3 elements,
  // the 0.22 uF coupling cap, the 1.0 megohm shunt resistor and stray capacitance.
  // In parts:
  Z22 = Complex(0.0, -1.0) / (w*uSave.lastState.capCouple);  // Impedance of 0.22 coupling cap
  // Yi is admittance of input R and C in parallel
  Yi = Complex((1.0 / uSave.lastState.resInput), 0.0) + Complex(0.0, w * uSave.lastState.capInput);
  Yinput = Cone / (Z22 + (Cone / Yi));                    // Admittance shunting input
  Ymeas = Cone / Zmeas;                                   // Uncorrected measured suseptance
  if(verboseData && useUSB)
     {
     Serial.println("");
     Serial.print("Embedded Z measured: "); Serial.print(Zmeas.real(),4);
             Serial.print("+j"); Serial.println(Zmeas.imag(),4);
     Serial.print("Embedded Y measured:");  Serial.print(Ymeas.real(), 9);
             Serial.print(" <re(Y)   im(Y)> "); Serial.print(Ymeas.imag(), 9);
             Serial.print("  pF > ");  Serial.println(1.0E12*Ymeas.imag() / w);
     }
  // Re-use Zmeas and Ymeas for final corrected values:
  Ymeas = Ymeas - Yinput;                      // De-embed correction to getunknown complex Y
  // And a final correction for resistance and inductive reactance of measuring leads, seriesR, seriesL.
  Zmeas = (Cone / Ymeas) - Complex(uSave.lastState.seriesR, w*uSave.lastState.seriesL);
  Ymeas = Cone / Zmeas;        // Include the correction back to Y

  // Indicate that a better refR may be available
  ZM = sqrt(Zmeas.real() * Zmeas.real() + Zmeas.imag() *Zmeas.imag());
  if(avnaState != WHATSIT)
     {
     if(uSave.lastState.iRefR == R50  &&  ZM > 500.0)
        LCDPrintError("Consider using refR=5K");
     else if(uSave.lastState.iRefR == R5K  &&  ZM < 500.0)
        LCDPrintError("Consider using refR=50");
     else
        LCDPrintError("");                   // Clear error
     }

  if(verboseData && useUSB)
     {
     Serial.print("De-embedded Z measured: "); Serial.print(Zmeas.real(),4);
             Serial.print("+j"); Serial.println(Zmeas.imag(),4);
     Serial.print("De-embedded Y measured: "); Serial.print(Ymeas.real(), 9);
             Serial.print(" <re(Y)   im(Y)> "); Serial.print(Ymeas.imag(), 9);
             Serial.print("  pF > ");  Serial.println(1.0E12*Ymeas.imag() / w);
     Serial.print("Measured: V="); Serial.print(amplitudeV);
             Serial.print(" VR="); Serial.print(amplitudeR);
             Serial.print(" Cal Ratio="); Serial.print(vnaF[nFreq].vRatio, 5);
             Serial.print(" dPhase="); Serial.println(vnaF[nFreq].dPhase, 3);
     }
  // Put results into global arrays
  Z[iF] = Zmeas;
  Y[iF] = Ymeas;
  if (Zmeas.imag() < 0.0)          // series R-C
     {
     sLC[iF] = -1.0 / (w * Zmeas.imag());
     // Zmeas.real() can be negative (usually very high Q with errors)
     if(Zmeas.real() > 0.0)
        Q[iF] = -Zmeas.imag() / Zmeas.real();     // Same Q, series or parallel
     else
        Q[iF] = 9999.9;
     }
  else                             // series R-L
     {
     sLC[iF] = Zmeas.imag() / w;
     if(Zmeas.real() > 0.0)
        Q[iF] = Zmeas.imag() / Zmeas.real();
     else
        Q[iF] = 9999.9;
     }

  if (Ymeas.imag() < 0.0)          // parallel R-L
     pLC[iF] = -1.0 / (w * Ymeas.imag());
  else                             // parallel R-C
     pLC[iF] = Ymeas.imag() / w;
  }

/* LCDPrintSingleZ()  -  Produces a screen based on data in globals
 * Z[0], Y[0], etc.
 * useLCD is checked before calling this fcn
 */
void LCDPrintSingleZ(uint16_t nfr)
  {
  double XX, RR, GG, BB;

  RR = Z[nfr].real();
  XX = Z[nfr].imag();
  GG = Y[nfr].real();
  BB = Y[nfr].imag();

  if(useLCD)         // Print screen of single frequency data
    {
   tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
    tft.setTextColor(ILI9341_YELLOW);
    tft.setFont(Arial_14);
    tft.setCursor(20, 38);
    tft.print("Z - Freq-");
    if     (vnaF[nfr].freqHz < 100.0)
       tft.print(vnaF[nfr].freqHz, 2);
    else if(vnaF[nfr].freqHz < 1000.0)
       tft.print(vnaF[nfr].freqHz, 1);
    else    //  Must be 1000 or more
       tft.print(floor(0.5+vnaF[nfr].freqHz), 0);
    tft.print(" Hz  Ref=");

    if (uSave.lastState.iRefR==R50)
       tft.print("50.0 Ohm");
    else
       tft.print("5.0 KOhm");

    tft.setCursor(20, 62);
    tft.print("Series: R=");
    tft.print(valueStringSign(RR, rUnits));
    tft.print("  X=");
    tft.print(valueStringSign(XX, rUnits));
    tft.setFont(Arial_18);
    tft.setCursor(30, 80);
    if(Q[nfr]>1.0 && XX<0.0)
       {
       tft.print("C=");
       tft.print(valueString(sLC[nfr], cUnits));
       tft.print("  Q=");
       // Serial.print shows negative Q's as a number, +9999.9.  Here we can be less terse:
       if(Q[nfr] < 9999.8)
          tft.print(Q[nfr], 1);
       else
          tft.print(" Neg");
       }
    else if(Q[nfr]>1.0 && XX>=0.0)
       {
       tft.print("L=");
       tft.print(valueString(sLC[nfr], lUnits));
       tft.print("  Q=");
       if(Q[nfr] < 9999.8)
          tft.print(Q[nfr], 1);
       else
          tft.print(" Neg");
       }
    else     // Low Q, show as a resistor with series L or C
       {
       tft.print("R=");
       tft.print(valueStringSign(RR, rUnits));
       if(XX > 0.0)
          {
          tft.print(" L=");
          tft.print(valueString(sLC[nfr], lUnits));
          }
       else
          {
          tft.print(" C=");
          tft.print(valueString(sLC[nfr], cUnits));
          }
        }
    tft.setTextColor(ILI9341_YELLOW);
    tft.setFont(Arial_14);
    tft.setCursor(20, 124);
    // G and B are the correct wasy to express a complex admittance.
    // parallelGB = false causes the inverted values of parallal R and X to be shown
    if(parallelGB)
       {
       tft.print("Par.: G=");
       tft.print(valueStringSign(GG, gUnits));
       tft.print("  B=");
       tft.print(valueStringSign(BB, gUnits));
       }
    else
       {
       tft.print("Par.: R=");
       if(fabsf(GG) > 1.0E-15)                        // What is #define for min??
          tft.print(valueStringSign(1.0/GG, rUnits));  // Equiv parallel res
       else
          tft.print("  inf   ");
       tft.print("  X=");
       if(fabsf(BB) > 1.0E-15)
          tft.print(valueStringSign(1.0/BB, rUnits));
       else
          tft.print("  inf   ");
       }    
    tft.setFont(Arial_18);
    tft.setCursor(30, 146);
    if(Q[nfr]>1.0 && BB>=0.0)
       {
       tft.print("C=");
       tft.print(valueString(pLC[nfr], cUnits));
       tft.print("  Q=");
       if(Q[nfr] < 9999.8)
          tft.print(Q[nfr], 1);
       else
          tft.print(" Neg");
       }
    else if(Q[nfr]>1.0 && BB<0.0)
       {
       tft.print("L=");
       tft.print(valueString(pLC[nfr], lUnits));
       tft.print("  Q=");
       if(Q[nfr] < 9999.8)
          tft.print(Q[nfr], 1);
       else
          tft.print(" Neg");
       }
    else
       {
       tft.print("R=");
       if(fabsf(GG) > 1.0E-37)                        // What is #define for min??
          tft.print(valueStringSign(1.0/GG, rUnits));
       else
          tft.print("  inf   ");
       if(BB < 0.0)
          {
          tft.print(" Lp=");
          tft.print(valueString(pLC[nfr], lUnits));
          }
       else
          {
          tft.print(" Cp=");
          tft.print(valueString(pLC[nfr], cUnits));
          }
        }   
     }
// primaryControl XCTRL_LCD  XCTRL_USB
  }      // end LCDPrintSingle()

void LCDPrintError(char const *emsg)
  {
   tft.fillRect(130, 185, tft.width(), 14, ILI9341_BLACK);
    tft.setTextColor(ILI9341_RED);
    tft.setFont(Arial_11);
    tft.setCursor(130,185);
    tft.print(emsg);
    tft.setTextColor(ILI9341_YELLOW);   // Leave it in yellow
  }
  

/* LCDPrintSweepLineZ()  -  Produces a sweep line based on data in globals
 * Z[], Y[], etc.
 * useLCD is checked before calling this fcn
 */
void LCDPrintSweepZ(void)
  {
  double XX, RR;
  uint16_t ii, nfr;

  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  tft.setTextColor(ILI9341_YELLOW);
  tft.setFont(Arial_12);
  tft.setCursor(0, 38);
  tft.print(" FREQ      R        X          L/C");

  for(ii=0; ii<7; ii++)
    {
    nfr = ii+firstLCDSweepFreq;
    RR = Z[nfr].real();
    XX = Z[nfr].imag();
    // Print line of Z data
    tft.setCursor(0, 60+20*ii);
    if     (vnaF[nfr].freqHz < 100.0)
       tft.print(vnaF[nfr].freqHz, 2);
    else if(vnaF[nfr].freqHz < 1000.0)
       tft.print(vnaF[nfr].freqHz, 1);
    else    //  Must be 1000 or more
       tft.print(floor(0.5+vnaF[nfr].freqHz), 0);
    tft.print("  ");
    tft.print(valueStringSign(RR, rUnits));
    tft.print("  ");
    tft.print(valueStringSign(XX, rUnits));
    tft.print("  ");
    if(XX<0.0)
       tft.print(valueString(sLC[nfr], cUnits));
    else if(XX>=0.0)
       tft.print(valueString(sLC[nfr], lUnits));
    }
  }      // end LCDPrintSingle()

// Output a line to Serial over USB.  Annotated or not, and corresponding
// to frequency index iF.  Assumes that useUSB is in effect
void serialPrintZ(uint16_t iF)
  {

  if(seriesRX)         // Series R and X values, plus L or C needed
    {
    // Always output freq
    Serial.print(vnaF[iF].freqHz, 3);
    // Output series Z,  L or C, and Q.  Annotation is optional.
    if (annotate)
       Serial.print(" Hz Series RX: R=");
    else
       Serial.print(",");
    Serial.print(Z[iF].real(), 3);
    if (annotate)
       Serial.print(" X=");
    else
       Serial.print(",");
    Serial.print(Z[iF].imag(), 3);
    // Serial print C or L value
    if (Z[iF].imag() < 0.0)
      {
      if (annotate)
        {
        Serial.print(" C=");
        Serial.print(valueString(sLC[iF], cUnits));
        Serial.print(" Q=");
        }
      else       // just CSV
        {
        Serial.print(",");
        Serial.print(sLC[iF], 12);
        Serial.print(",");
        }
      Serial.println(Q[iF]);
      }
    else    // Looks like an inductor
      {
      if (annotate)
        {
        Serial.print(" L= ");
         Serial.print(valueString(sLC[iF], lUnits));
        Serial.print(" Q=");
        }
      else       // just CSV
        {
        Serial.print(",");
        Serial.print(sLC[iF], 9);
        Serial.print(",");
        }
      Serial.println(Q[iF]);
      }
    }
  if(parallelRX)   // Not exclusive with series RX
    {
    Serial.print(vnaF[iF].freqHz, 3);
    // Output Y, L or C, and Q.  Annotation is optional.
    if (annotate)
       Serial.print(" Hz Parallel GB: G=");
    else
       Serial.print(",");
    Serial.print(Y[iF].real(), 9);
    if (annotate)
       Serial.print(" B=");
    else
       Serial.print(",");
    Serial.print(Y[iF].imag(), 9);
    if (annotate)
       Serial.print(" R= ");
    else
       Serial.print(",");
    Serial.print(1.0/Y[iF].real());
    if (Y[iF].imag() >= 0.0)
      {
      if (annotate)
        {
        Serial.print(" C=");
        Serial.print(valueString(pLC[iF], cUnits));
        Serial.print(" Q=");
        }
      else       // just CSV
        {
        Serial.print(",");
        Serial.print(pLC[iF], 12);
        Serial.print(",");
        }
      Serial.println(Q[iF]);
      }
    else    // Looks like an inductor
      {
      if (annotate)
        {
        Serial.print(" L= ");
         Serial.print(valueString(sLC[iF], lUnits));
        Serial.print(" Q=");
        }
      else       // just CSV
        {
        Serial.print(",");
        Serial.print(sLC[iF], 9);
        Serial.print(",");
        }
      Serial.println(Q[iF]);
      }
    }
  }        // end serialPrintZ(uint16_t iF)


// measureT does a single transmission data point, including applying CAL.
// Freq needs to be setup before calling, being found in vnaF[nFreq].freqHz.
// No printing is done.  No delays for settling.  Complements measureZ.
// Output is a single global Complex, Tmeas.  This is the ratio of the transmission
// transfer function to the reference value found by the thru-cal, with both
// adjusted for vRatio and dPhase.
void measureT(void)
  {
  float vGain, vPhase, vGainNorm, vPhaseNorm;
  //Complex Vm(0.0, 0.0);
  //Complex Vr(0.0, 0.0);

  getFullDataPt();
  checkOverload();
  // .vRatio for transmission is the through cal voltage magnitude.
  // .dPhase for transmission is the through cal voltage phase.
  vGain = (amplitudeV / amplitudeR) * vnaF[nFreq].vRatio;
  vPhase = phaseV - phaseR + vnaF[nFreq].dPhase;
  // Now normalize these gains to the through path gain from CAL.
  vGainNorm = vGain / vnaF[nFreq].thruRefAmpl;
  vPhaseNorm = vPhase - vnaF[nFreq].thruRefPhase;
  if (vPhaseNorm < (-180.0))
     vPhaseNorm += 360.0;
  else if (vPhaseNorm > 180.0)
     vPhaseNorm -= 360.0;
  Tmeas = polard2rect(vGainNorm, vPhaseNorm);
  Serial.print(vnaF[nFreq].freqHz, 3);
  if (annotate)
     Serial.print(" Hz  Magnitude normalized Av= ");
  else
     Serial.print(",");
  Serial.print(vGainNorm, 5);
  if (annotate)
     Serial.print(" V/V  Phase normalized (deg)= ");
  else
     Serial.print(",");
  Serial.println(vPhaseNorm, 3);
  }


/* LCDPrintSingleT()  -  Produces a screen based on data in global
 * T[0]
 */
void LCDPrintSingleT(uint16_t nfr)
  {
  double XT, RT, MT, PT, MTdB;

  RT = T[nfr].real();
  XT = T[nfr].imag();
  MT = sqrt(RT*RT + XT*XT);
  MTdB = 20.0 * log10(MT);
  PT = r2d(atan2(XT, RT));
 
  if(useLCD)         // Print screen of single frequency data
    {
    tft.fillRect(0, 38, tft.width(), 146, ILI9341_BLACK);
    tft.setTextColor(ILI9341_YELLOW);
    tft.setFont(Arial_13);
    tft.setCursor(20, 38);
    tft.print("T - Single Freq = ");
    if     (vnaF[nfr].freqHz < 100.0)
       tft.print(vnaF[nfr].freqHz, 2);
    else if(vnaF[nfr].freqHz < 1000.0)
       tft.print(vnaF[nfr].freqHz, 1);
    else    //  Must be 1000 or more
       tft.print(floor(0.5+vnaF[nfr].freqHz), 0);
    tft.print(" Hz");

    tft.setCursor(20, 62);
    if (uSave.lastState.iRefR==R50)
       tft.print("Ref=50.0 Ohm");
    else
       tft.print("Ref=5.0 KOhm");

    tft.setCursor(20, 86);
    tft.print("Voltage Gain = ");
    tft.print(MT, 5);
    tft.print(" V/V");

    tft.setFont(Arial_18);
    tft.setCursor(30, 120);
    tft.print("Gain = ");
    tft.print(MTdB, 3);
    tft.print(" dB");

    tft.setCursor(30, 155);
    tft.print("Phase = ");
    tft.print(PT, 2);
    tft.print(" deg");
    }
  }

// Someday:  Provide a field in non-annotated output for warnings and errors  <<<<<<<
void checkOverload(void)
  {
  float ppM, ppR;
  if (annotate && pkDet.available() && (ppM = pkDet.readPeakToPeak()) > 1.95)
    {
    Serial.print("Warning: P-P from Measure ADC=");
    Serial.print(50.0 * ppM);
    Serial.println("%");
    }
  if (annotate && pkDetR.available() && (ppR = pkDetR.readPeakToPeak()) > 1.95)
    {
    Serial.print("Warning: P-P from Reference ADC=");
    Serial.print(50.0 * ppR);
    Serial.println("%");
    }
  }

// -------------------------------------------------------------------------
/*  Not having filtering after the mixers improves the transient
    performance greatly.  But, it requires that one or more cycles
    of the measuring frequency fit exactly into the averaging period.
    Part of this is done by selecting the number of measurements to
    be averaged.  The frequency is then offset from the "desired" very
    slightly to make the fit "exact." The init below is exact frequency,
    the number of samples taken, the number of full 256 block samples
    used, the number of samples less than 256 needed to fill out the
    number of samples, the number of 0.1 sec measurements for AVE=1,
    the analog phase offset and finally the "almost" frequency to display.
*/
/* Request frequency f.  Depending on the sample rate, a slightly different
   frequency will be used, and set.  That frequency is returned here.
   Call this after the Sample Rate has been set (nSampleRate on 0,4).
*/
double modifyFreq(double f)
  {
  double tMeasMin, dNmin, dN, nSamples, time_samples, time_freq, fNew;
  uint16_t nSamplesI;
  tMeasMin = 0.1;             // At least 0.1 sec
  // Figure number of f periods, including part periods
  dNmin = tMeasMin * f;
  if ((dNmin - floor(dNmin)) < 0.000001)  dNmin = dNmin - 1;
  // dN is number of whole input frequency cycles
  dN = floor(dNmin) + 1;
  nSamples = floor(0.5 + (sampleRateExact * dN / f));   // Whole number of ADC samples
  nSamplesI = (uint16_t)nSamples;

  // Leave nSamples as is and alter freq to fit
  time_samples = nSamples / sampleRateExact;     // Time for ADC full sampling
  time_freq = dN / f;        // Time for N cycles of f

  // Alter f for 'perfect' fit.  This is the frequency from the VNA
  fNew = f * time_freq / time_samples;

  // Number of 256 word blocks and remainder.   Global variables
  num256blocks = (uint16_t) (nSamplesI / 256);
  numCycles = (uint16_t)(nSamplesI - 256 * num256blocks);
  vnaF[nFreq].freqHzActual = fNew;
#if DIAGNOSTICS
  Serial.print("Freq desired="); Serial.print(f); Serial.print(" Sample Rate="); Serial.print(sampleRateExact);
         Serial.print(" Num periods="); Serial.println(dN);
  Serial.print("Num adc samp="); Serial.print(nSamplesI); Serial.print(" Altered f="); Serial.print(fNew, 5);
#endif
  return fNew;
  }

void setSample(uint16_t nS)
  {
  sampleRateExact = setI2SFreq(nS);
  //factorFreq is global that corrects any call involving absolute frequency, like waveform generation.
  factorFreq = FBASE / sampleRateExact;
#if DIAGNOSTICS
  Serial.print(" S Rate=");  Serial.print(sampleRateExact); Serial.print( "ff=");
  Serial.println(factorFreq);
#endif
  }

/* Fir coefficient array and length.
   Array may also be FIR_PASSTHRU (length = 0), to directly pass the input to output
   without filtering.
*/
void setFilter(uint16_t firFilt)
  {
  if (firFilt == LPF2300)           // LPF with 0 to 2300 Hz passband and -20 dB above.
    { // 100 KHz sample rate only
    firIn1.begin(lp2300_100K, 100);
    firIn2.begin(lp2300_100K, 100);
    }
  else if (firFilt == FILT_NONE)
    {
    firIn1.begin(FIR_PASSTHRU, 0);
    firIn2.begin(FIR_PASSTHRU, 0);
    }
  }

// setUpNewFreq(uint16_t nF)  -  All steps necessary to
// use a frequency.  fr is in Hz. In most cases this is ADC spur avoidance.
void setUpNewFreq(uint16_t nF)
  {
  double fr = vnaF[nF].freqHz;
  topLines();
  if (fr < 2300.0)
    {
    setSample(S100K);
    setFilter(LPF2300);
    }
  else if (fr < 3800.0)
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  else if (fr < 5000.0)
    {
    setSample(S48K);
    setFilter(FILT_NONE);
    }
  else if (fr < 12000.0)
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  else if (fr < 13000.0)
    {
    setSample(S48K);
    setFilter(FILT_NONE);
    }
  else if (fr < 20400.0)
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  else if (fr < 21500.0)
    {
    setSample(S96K);
    setFilter(FILT_NONE);
    }
  else if (fr < 28600.0)
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  else if (fr < 29800.0)
    {
    setSample(S96K);
    setFilter(FILT_NONE);
    }
  else if (fr < 37000.0)
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  else if (fr < 38000.0)
    {
    setSample(S96K);
    setFilter(FILT_NONE);
    }
  else
    {
    setSample(S100K);
    setFilter(FILT_NONE);
    }
  modifyFreq(fr);
  AudioNoInterrupts();
  waveform1.frequency(factorFreq * vnaF[nFreq].freqHz);
  waveform1.phase(0);
  waveform2.frequency(factorFreq * vnaF[nFreq].freqHz);
  waveform2.phase(270);
  AudioInterrupts();
  }

/* getFullDataPt()  -  Measure average amplitude and phase (rel to DSP generated
   wave).  This function does not return until printReady istrue, meaning that the
   measurement is complete.  The frequency and ZorT must be set before calling here.
   Answers are global floats amplitudeV, phaseV, amplitudeR, phaseR.
   This function is useable for single or sweep, impedance or transmission.
*/
boolean getFullDataPt(void)
  {
  uint16_t kk;

  getNmeasQI();    // Measure and average   << Combine back here
  // The four data items will not become available in any particular order.
  // So track their status with NNReady[]
  // serial transmit data
  if (NNReady[0] && NNReady[1] && NNReady[2] && NNReady[3])    // Everybody ready
    {
    // Find ave power, correct for noise, and convert to an amplitude
    // The voltage at the top of the measurement resistor is measured
    // as ref Q, superAveNN[2], and ref I, superAveNN[3]. These are doubles.
    amplitudeV = sqrt(superAveNN[1] * superAveNN[1] + superAveNN[0] * superAveNN[0]);
    amplitudeR = sqrt(superAveNN[3] * superAveNN[3] + superAveNN[2] * superAveNN[2]);
    // Minus signs reflect the hookup of the inputs to U1A and U1D:
    phaseV = r2d(atan2(-superAveNN[0], -superAveNN[1]));
    phaseR = r2d(atan2(superAveNN[2], superAveNN[3]));
    // Diagnostic printout
    if(0)
      {
      Serial.print("V values: "); Serial.print(amplitudeV); Serial.print("  ang in deg = "); Serial.println(phaseV);
      Serial.print("R values: "); Serial.print(amplitudeR); Serial.print("  ang in deg = "); Serial.println(phaseR);
      }
    for (kk = 0; kk < 4; kk++)
      NNReady[kk] = false;   // Only print these once
    printReady = true;
    return true;
    }
  return false;
  }

void print2serial(void)
  {
  double p;
  delay(50);
  //  if (annotate)
  //      Serial.print("Fr=");
  //  Serial.print(vnaF[nFreq].freqHz);
  // CSV output if not annotating
  if (uSave.lastState.ZorT == TRANSMISSION)
     {
     double normV = amplitudeV / amplitudeR;   //ADD CAL
     if (annotate)  Serial.print(" Ampl=");
     else           Serial.print(",");
     // Scale up to be compatible with Serial.print
     Serial.print(normV, 4);
     if (annotate)  Serial.print(" Phase=");    // ADD CAL
     else           Serial.print(",");
     p = phaseV - phaseR - 180.0;
     if (p > 180.0)
        p = p - 360.0;
     else if (p < -180.0)
        p = p + 360.0;
     Serial.println(p);
     }
  // Individual vectors for analysis - not normally needed
  if (verboseData)
     {
     Serial.print(" L & R Measurements V=");
     Serial.print(amplitudeV, 4);
     Serial.print("  ");
     Serial.print(phaseV, 2);
     Serial.print(" R=");
     Serial.print(amplitudeR, 4);
     Serial.print("  ");
     Serial.println(phaseR, 2);
     }
  printReady = false;
  }

/* getNmeasQI(Nmeas, zDelay)
   Gather  numTenths x numCycles   samples of I, Q, RI and RQ and average.
   zdelay = the settling time in ms before starting.
   Results are normalized to one measurement.
   Results in double superAveNN[]
*/
void getNmeasQI(void)
  {
  uint16_t hh, jj, kk;

  for (jj = 0; jj < 4; jj++)
     {
     queueNN[jj].begin();    // Start loading the queueNN[]
     queueNN[jj].clear();    //  Clear everything
     sumNN[jj] = 0.0;        // 4 places for results
     }
   countMeasurements = 0;  // Total number of measurements, like up to 4411 for 0.1 sec.
   for (hh = 0; hh < vnaF[nFreq].numTenths; hh++)  // All tenth seconds    <<<<<
     {
     // Get most of data for measurement
     for (kk = 0; kk < num256blocks; kk++)
        {
        // Get data, after all queues are loaded
        while ((queueNN[0].available() < 2) || (queueNN[1].available() < 2) ||
               (queueNN[2].available() < 2) || (queueNN[3].available() < 2));   // Just wait
        // All are available;  Go get 'em
        countMeasurements += 256;
        for (jj = 0; jj < 4; jj++)
           {
           superAveNN[jj] += (double)get2blockAve(jj, 256);
           }
        }            // End, over all full blocks
    // In general a partial block of less than 256 is still needed
    while ((queueNN[0].available() < 2) || (queueNN[1].available() < 2) ||
           (queueNN[2].available() < 2) || (queueNN[3].available() < 2));   // Just wait
    // numCycles of them are available;  Go get 'em
    countMeasurements += numCycles;
    for (jj = 0; jj < 4; jj++)
       {
       superAveNN[jj] += (double)get2blockAve(jj, numCycles);
       }
     }      // End, over all tenth seconds

  // Divide by countMeasurements to get average
  for (jj = 0; jj < 4; jj++)
     {
     queueNN[jj].end();     //  Stop queues
     superAveNN[jj] = superAveNN[jj] / ((double)countMeasurements);
     NNReady[jj] = true;     // Mark as ready to use and print
     }
  }

/* get2blocks(uint16_t nn)  The basic measurements are averaged here.
   nn is the meauremnt 0 to 3 (Q, I, RQ, RI).
   The average, a float, is returned in sumNN[nn]
*/
double get2blockAve(uint16_t nn, uint16_t npts)
  {
  uint16_t ii;

  // Fetch 2 blocks from nn'th of 4 queues and copy
  // into a 512 byte (256 16-bit word) buffer.
  // The return is the sum of up to 256 and is left in global float sumNN(nn)
  // memcpy() works on bytes, bufferNN[] works on 16-bit words
  memcpy(&bufferNN[nn][0], queueNN[nn].readBuffer(), 256);
  queueNN[nn].freeBuffer();
  memcpy(&bufferNN[nn][128], queueNN[nn].readBuffer(), 256);
  queueNN[nn].freeBuffer();
  sumNN[nn] = 0.0;
  // Add npts samples (16-bit numbers) to sumNN[nn]
  for (ii = 0; ii < npts; ii++)
     {
     sumNN[nn] += (float)bufferNN[nn][ii];
     }
  return sumNN[nn];
  }

// Utility to convert complex vector in amplitude and phase in
// degrees into conventional rectangular complex vector.
Complex polard2rect(double a, double p)
  {
  return Complex( a * cos(d2r(p)), a * sin(d2r(p)) );
  }
//=================================================================================
// Routines to convert basic measurement into impedance or transmission values

// TRANSMISSION command
void TransmissionCommand()
  {
  char *arg;
  doRun = RUNNOT;  // Stop any measurements
  uSave.lastState.ZorT = TRANSMISSION;

  arg = SCmd.next();
  if (arg != NULL)
    {
    if (atoi(arg) == 50)
      {
      uSave.lastState.iRefR = R50;
      }
    else if (atoi(arg) == 5000)
      {
      uSave.lastState.iRefR = R5K;
      }
    if (verboseData)
      {
      Serial.print("Transmission measure; Ref R = ");
      Serial.println(uSave.lastState.valueRRef[uSave.lastState.iRefR], 2);
      }
    }
  tft.fillRect(0, 38, tft.width(), 161, ILI9341_BLACK);
  topLines();
  }

// ---------------------  xxxxCommand()  Responses to serial commands  -------------

// FreqCommand changes the single frequency being used for serial control.  This is saved
// in   uSave.lastState.freqHz0  to endure shut down.  At startup, it will
// be moved also to vnaF[0] to allow calibration.
// This automatically sets the single frequency operation (non-sweep).
void FreqCommand(void)
  {
  char *arg;
  arg = SCmd.next();
  doRun = RUNNOT;  // Stop any measurements
  if (arg != NULL)
    {
    uSave.lastState.freqHz0 = (float)atof(arg);
    vnaF[0].freqHz = uSave.lastState.freqHz0;
    nFreq = 0;                   // For single frequency operation
    setUpNewFreq(0);
    uSave.lastState.SingleorSweep = SINGLE;
    saveStateEEPROM();
    if (verboseData)
      {
      Serial.print("New Single Frequency: ");
      Serial.print(uSave.lastState.freqHz0);
      Serial.println(" - Needs cal");
      }
    }                 // ELSE ERROR RESPONSE
  topLines();
  }

// SweepCommand sets up sweep of fixed table of frequencies.
// Parallels FreqCommand().  Use CAL and RUN to complete

void SweepCommand(void)
  {
  doRun = RUNNOT;  // Stop any measurements
  nFreq = 1;                   // For sweep
  uSave.lastState.SingleorSweep = SWEEP;
  saveStateEEPROM();
  if (verboseData)
    Serial.println("Sweep data setup - needs cal.");
  topLines();
  }

void CalCommand(void)
  {
  CalCommand0(1);
  }

// CAL command.  If Single freq, does that freq of cal.
// If sweep, does standard logarithmic set.
// baseI allows for leaving out bottom frequencies.
// baseI = 1 for all freq
void CalCommand0(uint16_t baseI)
  {
  clearStatus();
  doRun = RUNNOT;  // Stop any measurements
  if(verboseData)  Serial.print("Doing CAL");
  if (uSave.lastState.ZorT == IMPEDANCE)
    {
    if(verboseData)  Serial.println(" of Impedance.");
    if (uSave.lastState.SingleorSweep == SINGLE)
      {
      setUpNewFreq(nFreq);
      setRefR(R_OFF);         // Disconnect relays
      setSwitch(CAL_39);
      DC1.amplitude(dacLevel);     // Turn on sine wave
      delay(50);

      getFullDataPt();
      //if (printReady)
      //   print2serial();
      checkOverload();
      // These are in polar form.  vnaF[nFreq].vRatio is a minor correction for
      // differences in the gain of V and R channels. dPhase is same but for phase.
      vnaF[nFreq].vRatio = amplitudeR / amplitudeV;  // For use later
      vnaF[nFreq].dPhase = phaseR - phaseV;          // For use later
      setRefR(uSave.lastState.iRefR);           // Restore relay settings
      printCalDetails();
      calZSingle = true;
      if(verboseData)  Serial.println("...Cal complete.");
      }
    else if (uSave.lastState.SingleorSweep == SWEEP)  // Impedance Sweep
      {
      setRefR(uSave.lastState.iRefR);
      setSwitch(CAL_39);
      DC1.amplitude(dacLevel);     // Turn on sine wave
      delay(50);
      for (nFreq = baseI; nFreq <= 13;  nFreq++) // Over all frequencies; nFreq is global freq index
        {
        if(!verboseData) Serial.print(nFreq);
        setUpNewFreq(nFreq);
        delay(ZDELAY);   // Delay until level is constant
        getFullDataPt();
        checkOverload();
        vnaF[nFreq].vRatio = amplitudeR / amplitudeV;  // For use later
        vnaF[nFreq].dPhase = phaseR - phaseV;          // For use later
        printCalDetails();
        }
      calZSweep = true;
      if(verboseData)  Serial.println("");
      }
    }
  else             // Transmission
    {
    // Cal here is in two parts.  First the vRatio/dPhase correction
    // for the ADC input circuitry.  Then a second correction to normalize the
    // measurement
    if(verboseData)  Serial.print(" of Transmission. A reference thru-path is needed.");
    if (uSave.lastState.SingleorSweep == SINGLE)
      {
      setUpNewFreq(nFreq);

      // Get numbers for first correction
      setRefR(R_OFF);         //  Disconnect the output
      setSwitch(CAL_39);      // Drive both ADC channels together
      delay(200);
      getFullDataPt();
      checkOverload();
      // These are in polar form.  vnaF[nFreq].vRatio is the voltage at the
      // measurement input. dPhase is the measured phase.  Small difference between
      // the 2 ADC channels.
      vnaF[nFreq].vRatio = amplitudeR / amplitudeV;
      vnaF[nFreq].dPhase = phaseR - phaseV;

      // And for second correction
      // (for low Z output, manual delay here) <<<<<<<<<<<<<<<<<<<<<<<<
      setRefR(uSave.lastState.iRefR);   // Connect to 50 or 5K ohm output
      setSwitch(TRANSMISSION_37);      // Thru path
      delay(200);
      getFullDataPt();
      //if (printReady)
      //   print2serial();
      checkOverload();
      // The through path voltage gain, using first correction
      vnaF[nFreq].thruRefAmpl = (amplitudeV / amplitudeR) * vnaF[nFreq].vRatio;
      vnaF[nFreq].thruRefPhase = phaseV - phaseR + vnaF[nFreq].dPhase;
      printCalDetails();
      calTSingle = true;
      if(verboseData)  Serial.println("...Cal complete.");
      }
    else           // Swept transmission cal
      {
      setUpNewFreq(1);
      // Get numbers for first correction
      setRefR(R_OFF);         //  Disconnect the output
      setSwitch(CAL_39);      // Drive both ADC channels together
      delay(200);
      for (nFreq = baseI; nFreq <= 13;  nFreq++) // Over all freqs; nFreq is global freq index
        {
        if (verboseData)  Serial.print(nFreq);
        setUpNewFreq(nFreq);
        delay(100);            // To be sure
        getFullDataPt();
        checkOverload();
        vnaF[nFreq].vRatio = amplitudeR / amplitudeV;
        vnaF[nFreq].dPhase = phaseR - phaseV;
        }
      // And for second correction
      // (for low Z output, manual delay here) <<<<<<<<<<<<<<<<<<<<<<<<
      setRefR(uSave.lastState.iRefR);   // Connect to 50 or 5K ohm output
      setSwitch(TRANSMISSION_37);      // Thru path
      delay(200);
      for (nFreq = baseI; nFreq <= 13;  nFreq++) // Over all freq; nFreq is global freq index
        {
        if (verboseData)  Serial.print(13 + nFreq);
        setUpNewFreq(nFreq);
        delay(100);
        getFullDataPt();
        checkOverload();
        // The through path voltage gain, using first correction
        vnaF[nFreq].thruRefAmpl = (amplitudeV / amplitudeR) * vnaF[nFreq].vRatio;
        vnaF[nFreq].thruRefPhase = phaseV - phaseR + vnaF[nFreq].dPhase;
        printCalDetails();
        }
      if (verboseData)  Serial.println("");
      calTSweep = true;
      Serial.println("...Cal complete.");
    }
  }
}          // End CalCommand

// Verbose printing for Cal
void printCalDetails(void)
  {
  if(verboseData)
      {
      Serial.print("Fr=");       Serial.print(vnaF[nFreq].freqHz, 3);
      Serial.print("  V=");      Serial.print(amplitudeV);
      Serial.print(" Vph=");     Serial.print(phaseV);
      Serial.print("  VR=");      Serial.print(amplitudeR);
      Serial.print(" VRph=");     Serial.print(phaseR);
      Serial.print("  vRatio="); Serial.print(vnaF[nFreq].vRatio, 5);
      Serial.print(" dPhase=");  Serial.println(vnaF[nFreq].dPhase);
      }
  }

// RunCommand, in the command,  takes a parameter n that means to take n single measurements
// or to do n sweeps.  An zero value for n is to never stop (except with "RUN n" with n>0).
// A negative value is to not measure.
// The command is either "RUN n" or "R n".
//      doRun:  RUNNOT, COUNTDOWN, CONTINUOUS, or SINGLE_NC (single serial meas, no cal)
//      nRun: -1=stop measuring, 0 measure continuous, and 0<n is remaining measurements
void RunCommand(void)
  {
  char dr[16];
  char *arg;
  arg = SCmd.next();
  if (arg != NULL)
    {
    nRun = atoi(arg);
    if (nRun == 0)
      {
      strcpy(dr, " (Continuous)");
      doRun = CONTINUOUS;
      }
    else if (nRun > 0)
      {
      doRun = COUNTDOWN;
      strcpy(dr, " (Countdown)");
      }
    else if(nRun == -1)
      {
      doRun = SINGLE_NC;
      strcpy(dr, " (Single, no Cal)");
      }
    else
      {
      doRun = RUNNOT;
      nRun = 0;
      strcpy(dr, " (Not measuring)");
      }
    if (verboseData)
      {
      Serial.print("Set RUN n = ");
      Serial.print(nRun);
      Serial.println(dr);
      }
    }
  }

void PowerSweepCommand(void)
  {
  doRun = POWER_SWEEP;
  dacSweep = 0.01;   // Level during sweep
  Serial.println("Starting Power Sweep");
  }

// setRefR() controls the two relays to determine the reference resistance
// for impedance measurements.  R_OFF. R50 or R5K
void setRefR(uint16_t refState)
{
  if (refState == R_OFF)
  {
    digitalWrite(R50_36, LOW);        // Off
    digitalWrite(R5K_35, LOW);        // Off
  }
  else if (refState == R50)
  {
    digitalWrite(R50_36, HIGH);        // On
    digitalWrite(R5K_35, LOW);
  }
  else if (refState == R5K)
  {
    digitalWrite(R50_36, LOW);
    digitalWrite(R5K_35, HIGH);         // On
  }
}

// setSwitch(uint16_t what) Sets the measurement channel to
// CAL_39, IMPEDANCE_38 or TRANSMISSION_37.
void setSwitch(uint16_t what)
  {
  if (what == CAL_39)
    {
    digitalWrite(CAL_39, HIGH);          // On
    digitalWrite(IMPEDANCE_38, LOW);     // Off
    digitalWrite(TRANSMISSION_37, LOW);  // Off
    }
  else if (what == IMPEDANCE_38)
    {
    digitalWrite(CAL_39, LOW);             // Off
    digitalWrite(IMPEDANCE_38, HIGH);      // On
    digitalWrite(TRANSMISSION_37, LOW);    // Off
    }
  else if (what == TRANSMISSION_37)
    {
    digitalWrite(CAL_39, LOW);             // Off
    digitalWrite(IMPEDANCE_38, LOW);       // Off
    digitalWrite(TRANSMISSION_37, HIGH);   // On
    }
  }

// ZMEAS Command - Only sets up Z measurements
void ZmeasCommand()
  {
  char *arg;
  // Complex RR50(uSave.lastState.valueRRef[1], 0.0);
  // Complex RR5K(uSave.lastState.valueRRef[2], 0.0);
  doRun = RUNNOT;  // Stop any measurements
  uSave.lastState.ZorT = IMPEDANCE;
  arg = SCmd.next();
  if (arg != NULL)
    {
    if (atoi(arg) == 50)
       uSave.lastState.iRefR = R50;
    else if (atoi(arg) == 5000)
       uSave.lastState.iRefR = R5K;
    if (verboseData)
      {
      Serial.print("Set Ref R = ");
      Serial.println(uSave.lastState.valueRRef[uSave.lastState.iRefR]);
      }
    topLines();
    }
  }

void DelayCommand()
{
  char *arg;
  arg = SCmd.next();
  if (arg != NULL)
  {
    uSave.lastState.msDelay = (uint16_t)atoi(arg);  // New loop delay
    saveStateEEPROM();
    if (verboseData)
    {
      Serial.print("New Loop Delay: ");
      Serial.println(uSave.lastState.msDelay);
    }
  }                 // ELSE ERROR RESPONSE
}


//  From CALDAT command.
void CalDatCommand(void)
  {
    
  }

// From SERPAR command.  Sets type of output data for Z meas
//    SERPAR s p  s sets serial output, p sets parallel output choice
void SerParCommand(void)
  {
  char *arg;
  // First parameter after SERPAR
  arg = SCmd.next();  // Returns ptr to null terminated string, parse series RX
  if (arg == NULL)
    return;
  else if (strcmp(arg, "0") == 0)
    {
    seriesRX = false;
    if(annotate)
      Serial.println("Series RX not outputted");
    }
  else if (strcmp(arg, "1") == 0)
    {
    seriesRX = true;
    if(annotate)
      Serial.println("Series RX outputted");
    }

  // p for parallel options
  arg = SCmd.next();      // now parse parallel RX desires
  if (arg == NULL)
    return;
  else if (strcmp(arg, "0") == 0)
    {
    parallelRX = false;
    if(annotate)
      Serial.println("Parallel RX not outputted");
    }
  else if (strcmp(arg, "1") == 0)
    {
    parallelRX = true;
    if(annotate)
      Serial.println("Parallel RX outputted");
    }
  // Add possibility of S11 and S21 output?
  }
  
void TestCommand(void)
  {
  char *arg;

  teststate = 1;
  arg = SCmd.next();
  if (arg != NULL)
    {
    test_ry = (uint16_t)atoi(arg);
    }
  arg = SCmd.next();
  if (arg != NULL)
    {
    test_sw = (uint16_t)atoi(arg);
    }
    digitalWrite(CAL_39, LOW);           // Off
    digitalWrite(IMPEDANCE_38, LOW);     // Off
    digitalWrite(TRANSMISSION_37, LOW);  // Off
 // TEST 0 0  will leave all three switches off
 if(test_sw & 0X0001)
     digitalWrite(CAL_39, HIGH);          // On
 if(test_sw & 0X0002)    
    digitalWrite(IMPEDANCE_38, HIGH);     // On
 if(test_sw & 0X0004)
    digitalWrite(TRANSMISSION_37, HIGH);  // On
 setRefR(test_ry);

 DC1.amplitude(dacLevel);     // Turn on sine wave
 setUpNewFreq(7);         // 1000 Hz

  Serial.print("Test Mode -  RY = ");
  Serial.print(test_ry);
  Serial.print("  Test SW = ");
  Serial.println(test_sw);
  Serial.println("Restart required to reset to ordinary measurements.");
  }

// Param1Command()  -  "PARAM1 0 50.22 5017.3"
// The '0' means "no default."  Set to the number 99 and ALL EEPROM values will be set to the
// defaults.  For this reset, the next two parameters are ignored, and are not required.
// A value 0 causes the reference resistor values to be set to the two parameter values.
// PARAM1 with no following parameters  prints the two reference resistor values.
void Param1Command(void)
  {
  char *arg;

  arg = SCmd.next();
  if(arg == NULL)
     {
     Serial.println("Currrent PARAM1 values:");
     Serial.print("  50 Ohm reference  = "); Serial.println(uSave.lastState.valueRRef[1]);
     Serial.print("  5K Ohm reference  = "); Serial.println(uSave.lastState.valueRRef[2]);
     return;
     }
  else if(atoi(arg) == 99)
     {
     if(annotate)  Serial.println("Full reset of parameters!");
     uSave.lastState = DEFAULT_PARAMETERS;
     saveStateEEPROM();
     topLines();
     return;
     }
  else if(atoi(arg) != 0)
     return;
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.valueRRef[1] = atof(arg);
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.valueRRef[2] = atof(arg);
  saveStateEEPROM();
  topLines();
  }

/* param2Command() - Serves the changing of correction factors for the impedance measurements.
 * arg in order:     capInput resInput capCouple seriesR seriesL  
 *             "PARAM2 37.0  1000000.0    0.22    0.07   20.0"
 * With units           pF      Ohm        uF      Ohm    nH
 *
 * Note capInput is the most likely item to change, and it can be changed with simply "PARAM2 34.8"
 * To obtain current values, type  PARAM2  with no parameters
 */
void Param2Command(void)
  {
  char *arg;
 
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.capInput = 1.0E-12 * atof(arg);
  else
     {
     Serial.println("Currrent PARAM2 values:");
     Serial.print("  Input cap  = "); Serial.println(1.0E12 * uSave.lastState.capInput);
     Serial.print("  Input res  = "); Serial.println(uSave.lastState.resInput);
     Serial.print("  Couple cap = "); Serial.println(1.0E6 * uSave.lastState.capCouple);
     Serial.print("  Series res = "); Serial.println(uSave.lastState.seriesR, 4);
     Serial.print("  Series ind = "); Serial.println(1.0E9 * uSave.lastState.seriesL);
     return;
     }
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.resInput = atof(arg);
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.capCouple = 1.0E-6 * atof(arg);
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.seriesR = atof(arg);
  arg = SCmd.next();
  if (arg != NULL)
     uSave.lastState.seriesL = 1.0E-9 * atof(arg);
  arg = SCmd.next();
  saveStateEEPROM();
  }

// Command for providing lots of feedback
void VerboseCommand()
  {
  char *arg;
  arg = SCmd.next();      // Returns ptr to null terminated string
  if (arg == NULL)
    return;
  else if (strcmp(arg, "0") == 0)
    {
    verboseData = false;
    if(annotate)
      Serial.println("Verbose off");
    }
  else if (strcmp(arg, "1") == 0)
    {
    verboseData = true;
    if(annotate)
      Serial.println("Verbose enabled");
    }
  }

// Tuneup is used at setup time, or whenever parameters need to be checked.  These are
// the values that correct for circuit "stray" components to improve the accuracies at
// impedances near the extreme values.  This is available only over the serial port and
// is directed manually, since components, opens and shorts need to be connected.
// NOTE:  This is EXPERIMENTAL. In general, the results are better by using externally
// calculated values, except especially Cin.  Use this command with care and remember that
// default values are available from "PARAM1 99".
void TuneupCommand(void)
  {
  static float eRs, eR50, eR5K, e1Meg, eCin;    //  , eLs;
  static float eRsSave, eLsSave, eR50Save, eR5KSave, e1MegSave, eCinSave;
  char *arg;
  Complex Zmeas(0.0, 0.0);
  Complex Ymeas(0.0, 0.0);
  Complex Vm(0.0, 0.0);
  Complex Vr(0.0, 0.0);

  // A copy for "go back"
  eRsSave=  uSave.lastState.seriesR;        eLsSave= uSave.lastState.seriesL;
  eR50Save= uSave.lastState.valueRRef[R50]; eR5KSave=uSave.lastState.valueRRef[R5K];
  e1MegSave=uSave.lastState.resInput;       eCinSave=uSave.lastState.capInput;
  
  arg = SCmd.next();
  if (arg == NULL || atoi(arg) == 0)
    {
    Serial.println("Tuneup requires steps 1 to 4, in order, no omissions:");
    Serial.println("   1-Connect a solid short and type  TUNEUP 1");
    Serial.println("   2-Connect known value R=50 and type  TUNEUP 2 R  (the true value of R)");
    Serial.println("   3-Connect known value R=5K and type  TUNEUP 3 R  (the true value of R)");
    Serial.println("   4-Leave measurement terminals open and type  TUNEUP 4");
    Serial.println("   5-Type either  TUNEUP 5  to enter tuneup values to corrections");
    Serial.println("or 6-Type  PARAM1 99  to restore default values");
    return;
    }
    
  else if (atoi(arg) == 1)
    {
    Serial.println("Doing short circuit tests...");
    verboseData = true;
    uSave.lastState.valueRRef[1] = 50.0;        // We need a known value for these two variables
    uSave.lastState.valueRRef[2] = 5000.0;
    uSave.lastState.iRefR = R50;

    // First compare meas and ref channels
    uSave.lastState.ZorT = IMPEDANCE;
    uSave.lastState.SingleorSweep = SINGLE;
    nFreq = 6;
    setUpNewFreq(6);        // 500 Hz
    CalCommand();
    // Now measure  a short with input series corrections turned off
    uSave.lastState.seriesR = 0.0;         // Remove corrections
    uSave.lastState.seriesL = 0.0;
    setRefR(R50);           // 50 Ohm relay
    setSwitch(IMPEDANCE_38);
    DC1.amplitude(dacLevel);     // Turn on sine wave
    delay(50);

    measureZ(nFreq);
    checkOverload();
    eRs = Z[6].real();
    uSave.lastState.seriesR=eRsSave;  uSave.lastState.seriesL=eLsSave;
    Serial.print("Measured series R = ");  Serial.println(Z[6].real(), 6);
    Serial.println("Reactance is not currently estimated.");
    }

  else if (atoi(arg) == 2)
    {
    Serial.println("Measuring 50 Ohms...");
    arg = SCmd.next();
    if (arg != NULL)
      {
      measureZ(nFreq);
      checkOverload();
      eR50 = ((amplitudeR-amplitudeV)/amplitudeV) * (atof(arg) + eRs);
      Serial.print("Measured internal series R = ");  Serial.println(eR50, 3);    
      }
    else
      Serial.println("TUNEUP 2 requires the test resistor value, like  TUNEUP 2 49.85");
    }
    
  else if (atoi(arg) == 3)
    {
    Serial.println("Measure 5K Ohms...");
    uSave.lastState.iRefR = R5K;
    setRefR(R5K);           // 5K Ohm relay
    delay(50);
    arg = SCmd.next();
    if (arg != NULL)
      {
      measureZ(nFreq);
      checkOverload();
      eR5K = ((amplitudeR-amplitudeV)/amplitudeV) * (atof(arg) + eRs);
      Serial.print("Measured internal series R = ");  Serial.println(eR5K, 1);    
      }
    else
      Serial.println("TUNEUP 3 requires the test resistor value, like  TUNEUP 3 5012.0");
    }
   
  else if (atoi(arg) == 4)
    {
    Serial.println("Measure Input resistance...");
    nFreq = 13;
    setUpNewFreq(13);        // 40 KHz
    setRefR(R5K);           // 5K Ohm relay
    uSave.lastState.iRefR = R5K;
    delay(50);
    CalCommand();

    setSwitch(IMPEDANCE_38);
    delay(100);
    getFullDataPt();    // Gets amplitudes and phases
    checkOverload();
    // Do corrections for analog amplifier errors, vRatio and dPhase:
    Vm = polard2rect(amplitudeV * vnaF[nFreq].vRatio, phaseV + vnaF[nFreq].dPhase);
    Vr = polard2rect(amplitudeR, phaseR);
    Zmeas = Complex(eR5K, 0.0) * Vm / (Vr - Vm);

 Serial.println("av*vRatio  ar  pv+dPhase  pr");
 Serial.println(amplitudeV * vnaF[nFreq].vRatio);
 Serial.println(amplitudeR);
 Serial.println(phaseV+vnaF[nFreq].dPhase);
 Serial.println(phaseR);

    Ymeas = Cone / Zmeas;
    e1Meg = 1.0 / Ymeas.real();
    eCin = Ymeas.imag() / (6.28318 * vnaF[nFreq].freqHzActual);
#if DIAGNOSTICS
    Serial.print("Open Circuit, no corrections, parallel model G + jB: ");
    Serial.print(Ymeas.real(), 9);  Serial.print(" +j "); Serial.println(Ymeas.imag(), 9);
#endif
    Serial.print("Parallel Resistance = ");
    Serial.println(e1Meg);  
    Serial.print("Parallel Capacity, pF = ");
    Serial.println(1.0E12 * eCin); 
    }

  else if (atoi(arg) == 5)
    {
    Serial.println("Making measurements permanent. Use PARAM1 or PARAM2 to alter these.");
    uSave.lastState.seriesR=eRs;       // uSave.lastState.seriesL=eLs;
    uSave.lastState.valueRRef[R50]=eR50;  uSave.lastState.valueRRef[R5K]=eR5K;
    uSave.lastState.resInput=e1Meg;       uSave.lastState.capInput=eCin;
    }

  else if (atoi(arg) == 6)
    {
    Serial.println("Restoring current parameters (not defaults).  Use PARAM1 or PARAM2 to alter these.");
    uSave.lastState.seriesR=eRsSave;      //  uSave.lastState.seriesL=eLsSave;
    uSave.lastState.valueRRef[R50]=eR50Save;  uSave.lastState.valueRRef[R5K]=eR5KSave;
    uSave.lastState.resInput=e1MegSave;       uSave.lastState.capInput=eCinSave;
    }
  }

// Command to change baud rate
void BaudCommand(void)
  {
  char *arg;
  arg = SCmd.next();
  if (arg == NULL)
    return;
  Serial.end();
  delay(100);
  Serial.begin(atol(arg));
  delay(1000);
  Serial.print("New baud rate: ");
  Serial.println(atof(arg), 0);   
  }

// Command for annotating printout; Otherwise, CSV
void AnnotateCommand()
  {
  char *arg;
  arg = SCmd.next();
  if (arg == NULL)
    return;
  else if (strcmp(arg, "0") == 0)
    annotate = false;
  else if (strcmp(arg, "1") == 0)
  {
    annotate = true;
    Serial.println("Annotation enabled");
  }
}

// This is the default Command handler
void unrecognized()
{
  Serial.println("What cmd?");
}

// ----------------------------------------------------------------
// Use pins 2 and 3 to control panel 2-color LED
void panelLED(uint16_t what)
  {
  if (what == LSTART)           // Ready, but off
    {
    pinMode(PANEL_LED0, OUTPUT);
    pinMode(PANEL_LED1, OUTPUT);
    digitalWrite(PANEL_LED0, LOW);
    digitalWrite(PANEL_LED1, LOW);
    }
  else if (what == LGREEN)
    {
    digitalWrite(PANEL_LED0, LOW);
    digitalWrite(PANEL_LED1, HIGH);
    }
  else if (what == LRED)
    {
    digitalWrite(PANEL_LED0, HIGH);
    digitalWrite(PANEL_LED1, LOW);
    }
  else if (what == LOFF)
    {
    digitalWrite(PANEL_LED0, LOW);
    digitalWrite(PANEL_LED1, LOW);
    }
  }

/* Variable sample rate from Frank B., see
   https://forum.pjrc.com/threads/38753-Discussion-about-a-simple-way-to-change-the-sample-rate
   As pointed out by Frank, the various routines, like waveform generator, are calibrated
   for 44.11765 and need to be corrected when other sample rates. See factorFreq below.

   K12 Peripheral manual:
   MCLD Divide sets the MCLK divide ratio such that: MCLK output = MCLK input * ( (FRACT + 1) / (DIVIDE + 1) ).
   FRACT must be set equal or less than the value in the DIVIDE field.
   The MCLK divide ratio
   can be altered while an SAI is using that master clock, although the change in the divide
   ratio takes several cycles. MCR[DUF] can be polled to determine when the divide ratio
   change has completed.
*/
const uint16_t numFreqs = 5;
// sampleFreqs[] is of limited utility, as the index nSampleRate describes the selection,
// and sampleRateExact is the true rate, sometimes slightly different.
const int sampleFreqs[numFreqs] = {44100, 44117, 48000, 96000, 100000};
// Note Teensy 3.6:  F_CPU == 180000000, F_PLL == 180000000
// setI2SFreq(if) returns exact sample frequency, that may differ very slightly from sampleFreqs[]
double setI2SFreq(uint16_t iFreq)
{
  typedef struct
  {
    uint8_t mult;
    uint16_t div;
  } __attribute__((__packed__)) tmclk;
  // 44117 is nickname for 44117.64706
#if   (F_PLL==72000000)
  const tmclk clkArr[numFreqs] = {{98, 625}, {8, 51}, {64, 375}, {128, 375}, {16, 45} };
#elif (F_PLL==96000000)
  const tmclk clkArr[numFreqs] = {{147, 1250}, {2, 17}, {16, 125}, {32, 125}, {20, 75} };
#elif (F_PLL==120000000)
  const tmclk clkArr[numFreqs] = {{205, 2179}, {8, 85}, {64, 625}, {128, 625}, {16,75} };
#elif (F_PLL==144000000)
  const tmclk clkArr[numFreqs] = {{49, 625}, {4, 51}, {32, 375}, {64, 375}, {8, 45} };
#elif (F_PLL==16000000)
  const tmclk clkArr[numFreqs] = {{151, 214}, {12, 17}, {96, 125}, {192, 125}, {20,125} };
#elif (F_PLL==168000000)
  const tmclk clkArr[numFreqs] = {{42, 625}, {8, 119}, {64, 875}, {128, 875}, {16, 185} };
#elif (F_PLL==180000000)
  const tmclk clkArr[numFreqs] = {{196, 3125}, {16, 255}, {128, 1875}, {219, 1604}, {32, 225} };
#elif (F_PLL==192000000)
  const tmclk clkArr[numFreqs] = {{147, 2500}, {1, 17}, {8, 125}, {16, 125}, {2, 15} };
#elif (F_PLL==216000000)
  const tmclk clkArr[numFreqs] = {{98, 1875}, {8, 153}, {64, 1125}, {128, 1125}, {16, 135} };
#elif (F_PLL==240000000)
  const tmclk clkArr[numfreqs] = {{147, 3125}, {4, 85}, {32, 625}, {64, 625}, {8, 75} };
#endif
  while (I2S0_MCR & I2S_MCR_DUF) ;  // This is to make sure I2S controller is up to speed NEEDED??
  I2S0_MDR = I2S_MDR_FRACT((clkArr[iFreq].mult - 1)) | I2S_MDR_DIVIDE((clkArr[iFreq].div - 1));
  return  (double)F_PLL * (double)clkArr[iFreq].mult / (256.0 * (double)clkArr[iFreq].div);
}

// valueStringSign()
// Same as valueString() but one char bigger(9) and adds blank or minus sign.
char *valueStringSign(float a, char *unitStr[])
  {
  static char str1[9];
 
  if(a < 0.0)
     str1[0] = '-';
  else
     str1[0] = ' ';
  strcpy(&str1[1], valueString(fabsf(a), &unitStr[0]));
  return str1;
  }

// Use this to create a print string for unsigned numbers
// In order to handle R, L and C values, the range is from
// 0.1 pico unit to 99 mega units (21 decades!). The units
// are set by the 2-char strings in unitString.  The resulting
// str1 is 7 char long plus a terminating zero.
char *valueString(float a, char *unitString[])
  {
  float aa;
  static char str1[8];
  char str0[8];
  strcpy(str1, "      ");
  // Print a to printf-type precision of 1 and four
  // significant digits, if possible.  Use leading spaces, if not.
  // 0.1 <= a < 1000.0
  if(fabsf(a) < 1.0E-13)
     {
     strcpy(str1, " 1/inf ");
     return str1;
     }
  if(a < 1.0E-9)       // The pF
     {
     aa = 1.0E12*a;
     sprintf(str0, "%.1f", aa);
     if(strlen(str0) <= 5)      // Add leading spaces
        {
        str1[5-strlen(str0)] = 0;
        strcat(str1, str0);
        }
     else
        {
        strcpy(str1, str0);
        }
     strcat(str1, unitString[0]);
     }
  else if(a < 1.0E-8)
     {
     aa = 1.0E9*a;                    // For C, 1 to 10 nF
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[1]);
     }
  else if(a < 1.0E-7)
     {
     aa = 1.0E9*a;                    // 10 to 100 nF
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[1]);
     }
  else if(a < 1.0E-6)
     {
     aa = 1.0E9*a;                    // 100 to 1000 nF
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[1]);
     }

  else if(a < 1.0E-5)
     {
     aa = 1.0E6*a;                    // 1 to 10 uF
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[2]);
     }
  else if(a < 1.0E-4)
     {
     aa = 1.0E6*a;                    // 10 to 100 uF
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[2]);
     }
  else if(a < 1.0E-3)
     {
     aa = 1.0E6*a;                    // 100 to 1000 uF
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[2]);
     }
  else if(a < 1.0E-2)
     {
     aa = 1.0E3*a;                    // 1000 to 10,000 uF
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[3]);
     }
  else if(a < 0.1)
     {
     aa = 1.0E3*a;                    // 10,000 to 100,000 uF
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[3]);
     }
  else if(a < 1.0)
     {
     aa = 1.0E3*a;                    // 0.1 to 1
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[3]);
     }
  else if(a < 10.0)
     {
     aa = a;                    // 1 to 10
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[4]);
     }
  else if(a < 100.0)
     {
     aa = a;                    // 10 to 100
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[4]);
     }
  else if(a < 1000.0)
     {
     aa = a;                    // 100 to 1000
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[4]);
     }
  else if(a < 10000.0)
     {
     aa = 1.0E-3*a;                    // 1000 to 10,000
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[5]);
     }
  else if(a < 1.0E5)
     {
     aa = 1.0E-3*a;                    // 10,000 to 100,000
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[5]);
     }
  else if(a < 1.0E6)
     {
     aa = 1.0E-3*a;                    // 100,000 to 1.0E6
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[5]);
     }
  else if(a < 1.0E7)
     {
     aa = 1.0E-6*a;                    // 1.0E6 to 1.0E7
     sprintf(str1, "%.3f", aa);
     strcat(str1, unitString[6]);
     }
  else if(a < 1.0E8)
     {
     aa = 1.0E-6*a;                    // 1.0E7  to 1.0E8
     sprintf(str1, "%.2f", aa);
     strcat(str1, unitString[6]);
     }
  else if(a<1.0E9)
     {
     aa = 1.0E-6*a;                    // 1.0E7  to 1.0E8
     sprintf(str1, "%.1f", aa);
     strcat(str1, unitString[6]);
     }
  else
     strcpy(str1, "  ***  ");         // It can happen for X at 40 KHz
  return str1;
  }

void topLines(void)
  {
  tft.fillRect(0, 0, tft.width(), 38, ILI9341_BLACK);
  tft.setTextColor(ILI9341_WHITE);
  tft.setFont(Arial_16);
  tft.setCursor(5, 0);
  tft.print("Audio Vector Network Analyzer");
  tft.setFont(Arial_9);
  tft.setCursor(5, 22);
  // Setup a function for this to allow changing settins
  tft.print("ver 0.60 de W7PUA");
  //    if(SingleorSweep == SINGLE)
  if(uSave.lastState.ZorT == IMPEDANCE)
    tft.print(" Z  f=");
  else
    tft.print("   f=");
  tft.print(vnaF[nFreq].freqHzActual, 3);
  tft.print("Hz    Ref=");
  tft.print(uSave.lastState.valueRRef[uSave.lastState.iRefR]);
  tft.setTextColor(ILI9341_YELLOW);      // Others expect this
  }


