/* Using most of a FoxDelta AAZ-0914A RF scalar analyzer kit, monitor KX3 transmit frequency and
 * automatically keep an MFJ 1786/1788 mag loop antenna tuned for a match. The Fox Delta generates a test
 * signal using an AD9850 DDS and measures mismatch compared to 50 ohms with an AD8307 log amp. These parts
 * could be provided in any number of ways but the FoxDelta was a nice implementation that also includes
 * an SGA3486 MMIC wide band amp to improve measurement noise margin.
 *
 * The FoxDelta unit uses a PIC to provide a nice serial command language but rather than go through an
 * extra control layer we decided to just remove it and control the raw devices directly from the Arduino.
 * This also allows this code to more easily be adapted to other AD9850/AD8307 implementations.
 *
 * When first powered on, all four LEDs are turned on for one second as a nod to the classic "lamp
 * test" of days gone by. Then the ComOK LED flashes slowly until a valid frequency is received from the
 * radio. From then on the ComOK LED flickers whenever a valid frequency is received.
 *
 * The basic idea of automatic tuning mimics what one would do automatically using the MFJ Remote Control.
 * The DDS is set to the radio frequency then the antenna is swept quickly until it moves through a match.
 * Based on the last antenna frequency, if known, the direction always approaches the radio frequency. If
 * the match is missed for any reason, it automatically reverses if the end of travel limit is encountered.
 * When a match candidate is found, the antenna is rocked back and forth a few times looking for the best
 * match. Note that auto tuning is initially disabled until the first time the Tune switch is used.
 *
 * When the radio moves significantly away from the antenna frequency, the procedure starts automatically
 * as soon as the frequency stops changing. This feature allows you to tune around leisurely without
 * being interrupted. A search can also be initiated immediately at any time by tapping the Tune switch.
 *
 * For added convenience, we control a coax relay to temporarily route the antenna to the FoxDelta while
 * tuning then route it back to the KX3 when auto tuning is complete. By snooping the KX3 responses we try
 * to avoid switching while the radio is transmitting but this may not be perfect so operator beware.
 *
 * During automatic tuning two pots control motor speed. The one labeled "Slew" sets the steady fast rate
 * used for initial sweeping and manual moves. Use the manual Up and Down buttosn to adjust this pot so a
 * complete move from one end to the other takes about 20 seconds. The other pot, labeled "Step", controls
 * the length of the short pulses used to refine the match. This pot is a little tricky to get right. If it
 * is too small, it takes forever to refine the initial match. If too large, one step can jump right over
 * the best match position. On my loop, the best setting for this pot is mid range.
 *
 * The antenna may also be tuned manually. Holding the Up or Down switch will change the antenna frequency
 * in the corresponding direction until released or the end of travel if encountered. The rate of this tuning
 * is controlled by the Slew pot. The coax switch is not activated during these manual moves, leaving the
 * KX3 connected to the antenna. After a manual tune, auto tuning is disabled until the next Tune operation.
 *
 * We assume the KX3 will send the IF command response regularly, from which we learn the current frequency
 * and whether the transmitter is on. If the KX3 tx comm line (ACC1 ring) is connected to a computer 
 * running a program that is controlling the radio this is almost surely the case already. But if unsure
 * or if the KX3 is operating stand-alone without a computer, enter the Menu mode and set AUTOINF to
 * ANT CTRL.
 *
 * During tuning, the DDS signal can leak into the radio and make a very annoying tone. To avoid this, the
 * DDS can be zero beat exactly to match the radio. This procedure is invoked by holding the Up switch
 * while resetting the Arduino. The ComOK light will flash until the radio reports its frequency (just
 * rotate the KX3 main knob slightly) and then the Up light will flash to indicate you are now to tune
 * for zero beat. Adjust the Step pot to find zero beat, release the Up switch and the Up light will stop
 * flashing. This offset is stored in EEPROM and the procedure need not be repeated unless the radio
 * calibration is changed. The total pot range is about += 2 KHz.
 *
 * Another feature that can be triggered at reset is bad match calibration. To perform this function,
 * first temporarily disconnect the antenna from the tuner if you are not using the coax relay. Then hold
 * the Tune switch during reset. The Arduino will measure the mismatch for several seconds to accumulate
 * statistics about the mean and variance of the mismatch values under a poor match condition. These are
 * saved in EEPROM so the test need not be repeated unless the band background noise change dramatically.
 * This calibration should be performed any time the unit seems unable to perform tuning reliably.
 *
 * Easter eggs:
 *
 *   General purpose frequency generator:
 *
 *     Tap both manual switches at the same time to enter a mode that allows the AAZ-0914A to serve as a
 *     general purpose signal generator. To indicate this mode is in effect, both Up and Down LEDs are lit
 *     at the same time. Now the Slew pot is a course frequency control and the Step pot is a fine tuning
 *     control. The coax relay will connect the antenna to the Fox Delta during this mode. Briefly tap
 *     any switch to exit this mode.
 *
 *   Full Frequency sweep:
 *
 *     Hold Manual Up switch and tap Tune to perform a full frequency sweep. The limit LED flashes
 *     rapidly while the sweep is in progress. The frequency and mismatch are printed to the serial
 *     connection. The coax is activated during the sweep to connect the antenna to the DDS. Briefly tap
 *     any switch to cancel this mode.
 *
 * FoxDelta web site: http://foxdelta.com/products/aaz-0914a.htm
 *
 * (C) 2015 Elwood Downey, WB0OEW
 *
 * Code version number is in the first trace() statement in setup().
 */


/* EEPROM is used to store the mismatch when there is nothing connected
 */
#include <EEPROM.h>

/* Our trace() function works just like printf(3) to the standard Arduino TX line. This allows sending
 * helpful messages to the Arduino Serial Monitor. We can't use the usual Serial package because it
 * redefines the same interrupt vectors we need here.
 * Note: In order to see the trace() messages in the IDE Serial Monitor, it must be set to the same baud
 *      rate as the radio, ie, 38400. Some versions have this speed as an option, some do not. If yours
 *      does not try getting the "nightly build" from http://arduino.cc/en/Main/Software.
 */
void trace (const char *fmt, ...);


/* The antenna bandwidth is modeled to increase as the square of frequency scaled from the following
 * pair; adjust to taste.
 */
const uint16_t ANT_BW = 7;		// half bandwidth, KHz
const uint16_t ANT_F0 = 14000;		// frequency of ANT_BW, KHz


/* Maximum time to wait for new frequency from radio
 */
const uint16_t FREQ_AGE = 1500;		// ms


/* Mismatch values above BAD_MATCH are known to be very poor match. This are stored in EEPROM after being
 *   measured with the Fox Delta connection open. It must be set once and then may be set again at any
 *   time by holding the Tune switch while the Arduino is being reset, then letting go to make the
 *   measurements.
 * Similarly, FREQ_OFFSET is stored in EEPROM and may be set by holding Up during reset and adjusting the
 *   Step pot for zero beat.
 */
uint16_t BAD_MATCH;			// value of open circuit match
int16_t FREQ_OFFSET;			// DDS offset to match radio


/* antenna frequency range, change for other loop models
 */
const uint32_t LO_HZ =  6800000UL;
const uint32_t HI_HZ  = 22000000UL;


/* tuning constants, adjust to taste
 */
const uint16_t RUNNING_START = 1500;	// initial slew in wrong direction so real Tune moves fast over match
const uint16_t KICK_FACTOR = 10;	// initial move back after fast Tune for overshoot and backlash


/* assign coax relay output pin : HIGH to route antenna to FoxDelta, LOW to route antenna to KX3
 */
const uint8_t coaxRelayPin  = 6;


/* assign input pin for antenna end-of-travel limit switch detection : HIGH true
 */
const uint8_t limitDetect   = 12;


/* assign H-bridge control output : one held 0, PWM to the other depending on direction
 */
const uint8_t hDownPin	    = 11;
const uint8_t hUpPin	    = 10;


/* assign output status LED pins : HIGH lit
 */
const uint8_t goingUpLED    = 9;
const uint8_t goingDownLED  = 8;
const uint8_t limitLED      = 7;
const uint8_t ComOkLED      = 13;


/* handy array of all LEDs in left-right order on panel
 */
const uint8_t all_leds[] = { limitLED, goingDownLED, goingUpLED, ComOkLED };
const uint8_t N_LEDS = sizeof(all_leds)/sizeof(all_leds[0]);


/* assign user input switch pins : LOW true
 */
const uint8_t tuneSw        = 5;
const uint8_t manualUpSw    = 4;
const uint8_t manualDownSw  = 3;


/* assign Analog inputs pins
 */
const uint8_t slewADCPin    = A0;
const uint8_t stepADCPin    = A1;
const uint8_t AD8307Pin     = A2;


/* assign DDS output pins.
 * These could all be digitial pins but we're out of those.
 */
const uint8_t DDSresetPin   = A3;
const uint8_t DDSstrobePin  = A4;	// AKA FQ_UD
const uint8_t DDSclockPin   = A5;
const uint8_t DDSdataPin    = 2;


/* define system states, we are always in one of these
 */
typedef enum {
    STATE_STOPPED,			// no motion
    STATE_POT_FREQ,			// pots control freq
    STATE_SWEEP,			// full sweep
    STATE_MOVING,			// manual move, moving_up or not
    STATE_SRCH_SCAN,			// scanning for first minimum mismatch
    STATE_SRCH_FINAL,			// seeking back on final approach
} SystemState;


/* define some error codes for use with blinkErr().
 * N.B. blink code is index into descriptions array.
 */
typedef enum {
    ERR_NONE,
    ERR_TXON,
    ERR_EEPROM,
    ERR_NOFREQ,
    ERR_TOOLO,
    ERR_TOOHI,
    ERR_N
} Err;
const char *const blink_descriptions[ERR_N] = {
    NULL,
    "Trasmitter is on",
    "EEPROM error",
    "Radio frequency is unknown",
    "Radio frequency is below antenna range",
    "Radio frequency is above antenna range",
};



/* All serial IO uses interrupts via these circular buffer queues.
 * kx3rxchars[] is a circular buffer of all inbound chars.
 *   usage: store new char at rxch_head then increment; used endlessly so no tail. When see terminator
 *   add to kx3rxcmds (see next).
 * kx3rxcmds[] is a circular buffer of indices into kx3rxchars[], one for each complete command.
 *   usage: increment rxcm_head then store, read from rxcm_tail then increment. This usage simplfies
 *   adding entries when find the command terminator character ';' in kx3rxchars.
 *   rxcm_head == rxcm_tail means empty. no state for "full" just overwrite oldest is good enough.
 * txchars[] is a circular buffer of all outbound chars. Overflow is harmless other than losing chars.
 *   usage: store new char at txch_head then increment; txch_head == txch_tail signifies empty.
 * N.B. sizes must be power of two to allow using a fast mask for modulus, and fit in 8 bits to use uint8_t.
 */

/* macro to perform fast Q index increment with wrap
 */
#define QADVANCE(index,mask)   ((index) = (((index) + 1) & (mask)))

#define NRXCHARS        128     	// max number of pending characters in inbound queue
#define RXCHARSMASK     (NRXCHARS-1)	// modulo mask
#if NRXCHARS > 255 || (NRXCHARS & RXCHARSMASK)
    #error NRXCHARS must be power of 2 and fit in 8 bits
#endif
volatile uint8_t kx3rxchars[NRXCHARS];
volatile uint8_t rxch_head;

#define NRXCMDS         16       	// max number of pending inbound commands
#define RXCMDSMASK      (NRXCMDS-1)	// modulo mask
#if NRXCMDS > 255 || (NRXCMDS & RXCMDSMASK)
    #error NRXCMDS must be power of 2 and fit in 8 bits
#endif
volatile uint8_t kx3rxcmds[NRXCMDS];
volatile uint8_t rxcm_head, rxcm_tail;

#define	NTXCHARS	128		// max number of characters in outbound queue
#define	TXCHARSMASK	(NTXCHARS-1)	// modulo mask
#if NTXCHARS > 255 || (NTXCHARS & TXCHARSMASK)
    #error NTXCHARS must be power of 2 and fit in 8 bits
#endif
volatile char txchars[NTXCHARS];
volatile uint8_t txch_head, txch_tail;


/* class to hold a list of match readings.
 * methods for reset, add new reading and check for possible dip.
 */
class MatchReadings {

    private:

	// record last N_READINGS, newest reading added at highest index
	static const uint8_t N_READINGS = 4;
	uint16_t readings[N_READINGS];

	// record dip estimage
	uint16_t dip_estimage;

    public:

	// constructor resets
	MatchReadings (void) {
	    reset();
	}

	// reset
	void reset(void) {
	    for (uint8_t i = 0; i < N_READINGS; i++)
		readings[i] = BAD_MATCH;
	    dip_estimage = BAD_MATCH;
	}

	// fake approach down to start
	void reverse (uint16_t start) {
	    readings[N_READINGS-1] = start;
	    for (uint8_t i = N_READINGS-1; i > 0; --i)
		readings[i-1] = readings[i] + 1;
	}

	// push down and add new reading at highest index
	void add (uint16_t new_reading) {
	    memmove (readings, readings+1, sizeof(readings[0])*(N_READINGS-1));
	    readings[N_READINGS-1] = new_reading;
	}

	// check for having passed through the high speed dip.
	// if so, return true and record an estimate of the best match
	bool dip (void) {

	    // scan readings for correct shape and depth
	    bool real_dip = true;
	    uint16_t center_mean = 0;
	    for (uint8_t i = 0; real_dip && i < N_READINGS; i++) {
		if (i == 0) {
		    // oldest pair must be strictly decreasing (improving)
		    if (readings[0] <= readings[1])
			real_dip = false;
		} else if (i == N_READINGS-1) {
		    // newest pair must be strictly increasing (getting worse)
		    if (readings[N_READINGS-2] >= readings[N_READINGS-1])
			real_dip = false;
		} else {
		    // all inside must be below BAD_MATCH
		    uint16_t ri = readings[i];
		    if (ri >= BAD_MATCH) {
			real_dip = false;
		    } else {
			// accumulate insider mean for use as best dip estimate
			center_mean += ri/(N_READINGS-2);
		    }
		}
	    }

	    if (real_dip) {
		// ok but using center mean could still be too lucky
		dip_estimage = (BAD_MATCH + center_mean)/2;
	    }

	    // report
	    return (real_dip);
	}

	// trace readings contributing to dip
	void traceDip (void) {
	    ::trace (" %u :", dip_estimage);
	    for (uint8_t i = 0; i < N_READINGS; i++)
		::trace (" %u", readings[i]);
	    ::trace ("\n");
	}

	// check if we've just passed a dip
	bool passed (void) {

	    // increasing and started better than dip estimage?
	    uint16_t ri = readings[N_READINGS-2];
	    uint16_t rj = readings[N_READINGS-1];
	    if (ri < rj && ri < dip_estimage)
		return (true);
	    return (false);
	}

	// trace readings contributing to passed
	void tracePassed (void) {
	    ::trace (":");
	    for (uint8_t i = N_READINGS-2; i < N_READINGS; i++)
		::trace (" %u", readings[i]);
	    ::trace ("\n");
	}

} readings;


/* one bit for each switch, used in switches
 */
const uint8_t SW_UP   = 1;
const uint8_t SW_DOWN = 2;
const uint8_t SW_TUNE = 4;


/* local persistent variables
 */
SystemState current_state;      	// current state
SystemState last_state;			// previous state
uint16_t slew_speed;           		// moving speed, PWM units, set from pot
uint16_t step_time;            		// duration of step, ms, set from pot
bool moving_up;                 	// whether current move is up in frequency, else down
bool last_dir;				// last motion command direction
uint8_t switches;			// current switch state, made from SW_*
bool hit_limit;				// whether just hit a limit
bool enable_autotune;			// disable automatically engaging tune after either manual move
bool transmitting_now;			// whether radio is currently transmitting, else receiving
uint32_t antenna_freq;			// antenna is currently tuned to this freq, Hz, 0 if unknown
uint32_t radio_freq;			// radio is currently tuned to this freq, Hz, 0 if unknown
uint16_t last_speed;			// last motion commanded speed
uint32_t sweep_hz;			// running sweep frequency
uint32_t tune_t0;			// tune start time, millis()
uint32_t newfreq_time;			// double duty: ComOkLED blinker then time of new radio freq
uint32_t manual_t0;			// manual move start time, millis()
uint32_t now;				// millis() at start of each loop()


/* one-time initial setup
 */
void
setup()
{
    // setup coax relay control output
    pinMode (coaxRelayPin, OUTPUT);
    digitalWrite (coaxRelayPin, LOW);

    // setup limit detector input
    pinMode (limitDetect, INPUT_PULLUP);

    // setup h-bridge control outputs
    pinMode (hDownPin, OUTPUT);
    pinMode (hUpPin, OUTPUT);

    // setup LED outputs
    pinMode (goingUpLED, OUTPUT);
    pinMode (goingDownLED, OUTPUT);
    pinMode (limitLED, OUTPUT);
    pinMode (ComOkLED, OUTPUT);

    // setup operator switch inputs
    pinMode (manualDownSw, INPUT_PULLUP);
    pinMode (manualUpSw, INPUT_PULLUP);
    pinMode (tuneSw, INPUT_PULLUP);

    // default (5V) analog reference
    analogReference (DEFAULT);

    // setup speed control pot inputs
    pinMode (slewADCPin, INPUT);
    pinMode (stepADCPin, INPUT);

    // setup reading AD8307 mismatch
    pinMode (AD8307Pin, INPUT);

    // setup DDS outputs
    pinMode (DDSresetPin, OUTPUT);
    pinMode (DDSstrobePin, OUTPUT);
    pinMode (DDSclockPin, OUTPUT);
    pinMode (DDSdataPin, OUTPUT);

    // set up USART0 for communication from radio and trace output
    UCSR0B = (1<<RXEN0)|(1<<RXCIE0) | (1<<TXEN0)|(1<<UDRIE0);   // enable RX TX and their done interrupts
    UCSR0C = 0x80|(1<<UCSZ01)|(1<<UCSZ00);                      // 8 bit chars
    #define BAUDRATE 38400                                      // radio serial baud rate
    #define BAUD_PRESCALER (((F_CPU / (BAUDRATE * 16UL))) - 1)
    UBRR0 = BAUD_PRESCALER;

    // put AD9850 in serial load mode, see data sheet page 12 (should be default but just to be sure)
    pulsePinHigh (DDSresetPin);
    pulsePinHigh (DDSclockPin);
    pulsePinHigh (DDSstrobePin);

    // enable CPU interrrupts
    sei();

    // init state
    current_state = STATE_STOPPED;

    // here we go
    trace ("\n\n\nVersion 1.04\n\n");

    // lamp test -- nod to the past
    for (uint8_t i = 0; i < 2; i++) {
	delay(500);
	digitalWrite (goingUpLED, !i);
	digitalWrite (goingDownLED, !i);
	digitalWrite (limitLED, !i);
	digitalWrite (ComOkLED, !i);
	delay(1000);
    }

    // require EEPROM settings or allow setting again
    readEEPROM();
    if (BAD_MATCH == 65535 || digitalRead (tuneSw) == LOW)
	calibrateBadMatch();
    trace ("BAD_MATCH set from EEPROM to %u\n", BAD_MATCH);
    trace ("(Recalibrate by resetting while holding Tune)\n");
    if (abs(FREQ_OFFSET) > 2000 || digitalRead (manualUpSw) == LOW)
	calibrateFreqOffset();
    trace ("FREQ_OFFSET set from EEPROM to %d Hz\n", FREQ_OFFSET);
    trace ("(Recalibrate by resetting while holding Up)\n");
}


/* main infinite loop
 */
void
loop()
{
    // record time now (used in multiple places)
    now = millis();

    // read user command inputs
    readSwitches();

    // read course speed pot
    slew_speed = analogRead (slewADCPin) / 4;		// 0 .. 255 PWM Hz

    // check for any new messages from radio
    checkRadioMessages();

    // act on and update current state
    checkState();

    // control coax switch
    setCoaxSwitch();
}


/* Update state depending on current state, user input and measured mismatch.
 */
void
checkState()
{

    switch (current_state) {

	case STATE_STOPPED:

	    // idle now: check switches for possible new state
	    if (switches == SW_UP) {

		// only manual up switch is on: start manual slew going up
		moving_up = true;
		enable_autotune = false;
		trace ("Manual up @ %u\n", slew_speed);
		manual_t0 = now;
		current_state = STATE_MOVING;

	    } else if (switches == SW_DOWN) {

		// only manual down switch is on: start manual slew going down
		moving_up = false;
		enable_autotune = false;
		trace ("Manual down @ %u\n", slew_speed);
		manual_t0 = now;
		current_state = STATE_MOVING;

	    } else if (switches == SW_TUNE) {

		// only auto tune switch is on: start automatic tune
		enable_autotune = true;
		waitForNoSwitches();
		trace ("\nStart tuning\n");
		startTuning();

	    } else if (enable_autotune && radio_freq > 0) {

		// no switches are on, see if freq error large enough to start an automatic tune
		// unless op (or software) is still spinning the dial
		if (now > newfreq_time + FREQ_AGE) {
		    uint32_t tuning_bw = antBandwidth(antenna_freq);
		    if (radio_freq > antenna_freq + tuning_bw || radio_freq < antenna_freq - tuning_bw)
			startTuning();
		}
	    }

	    break;

	case STATE_POT_FREQ:

	    // continue pot control until cancelled
	    updatePotFreq();	// may change current_state

	    break;

	case STATE_SWEEP:

	    // continue sweep until finished
	    updateSweep();	// may change current_state

	    break;

	case STATE_MOVING:

	    // update limit state
	    checkLimit();

	    // manually moving: just continue unless see new switch states
	    if (switches == (SW_UP | SW_DOWN)) { 

		// both manual up and down switches are on: now pots control frequency
		stop();
		digitalWrite (goingUpLED, HIGH);
		digitalWrite (goingDownLED, HIGH);
		trace ("Start pot freq control\n");
		waitForNoSwitches();
		current_state = STATE_POT_FREQ;

	    } else if (switches == (SW_UP | SW_TUNE)) {

		// Tune on during manual up: switch to sweep mode
		startSweep();
		waitForNoSwitches();
		current_state = STATE_SWEEP;

	    } else if ((moving_up && !(switches&SW_UP)) || (!moving_up && !(switches&SW_DOWN)) || hit_limit) {

		// manual switch released or hit limit: stop moving
		trace ("for %lu ms\n", now - manual_t0);
		hit_limit = false;
		stop();
		current_state = STATE_STOPPED;

	    } else {

		slew();

	    }

	    break;

	case STATE_SRCH_SCAN:

	    // update limit state
	    checkLimit();

	    // resume tune in progress
	    updateTuning();	// may change current_state

	    break;

	case STATE_SRCH_FINAL:

	    // update limit state
	    checkLimit();

	    // final apprach
	    finalTune();	// may change current_state

	    break;
    }
}


/* set up for tuning, set state to SRCH_SCAN.
 */
void
startTuning ()
{
    // start idle
    setDDSFreq (0UL);
    stop();

    // cancel if transmitter is on
    if (transmitting_now) {

	blinkErr(ERR_TXON);
	current_state = STATE_STOPPED;
	return;

    }

    // cancel if don't know radio frequency
    if (!radio_freq) {

	blinkErr(ERR_NOFREQ);
	current_state = STATE_STOPPED;
	return;

    }

    // cancel if out of antenna range
    if (radio_freq < LO_HZ) {

	blinkErr(ERR_TOOLO);
	current_state = STATE_STOPPED;
	return;

    }
    if (radio_freq > HI_HZ) {

	blinkErr(ERR_TOOHI);
	current_state = STATE_STOPPED;
	return;

    }

    // init DDS to radio frequency
    setDDSFreq (radio_freq);

    // change state to tuning
    current_state = STATE_SRCH_SCAN;

    // record time
    tune_t0 = now;

    // switch coax
    setCoaxSwitch();

    // decide direction
    if (!antenna_freq || radio_freq > antenna_freq) {

	// really want to go up but nudge down briefly first to insure full speed scan over match
	// if antenna position is unknown, this is good a guess as any.
	moving_up = false;
	step(RUNNING_START);
	moving_up = true;

    } else {

	// really want to go down but nudge up briefly first to insure full speed scan over match
	moving_up = true;
	step(RUNNING_START);
	moving_up = false;

    }

    // invalidate mismatch history readings
    readings.reset();
    trace ("Slewing %s @ %u to %lu:", getDirName(), slew_speed, radio_freq);

    // start moving
    slew();
}


/* scan antenna looking for dip in match, change state to SRCH_FINAL when find it.
 * turn around if encounter limit.
 * can be cancelled by tapping any switch.
 */
void
updateTuning()
{
    if (switches) {

	// cancelled
	trace ("\nTuning cancelled by operator\n\n");
	enable_autotune = false;
	stop();
	setDDSFreq (0UL);
	waitForNoSwitches();
	current_state = STATE_STOPPED;

    } else if (hit_limit) {

	// hit limit so turn around
	hit_limit = false;
	moving_up = !moving_up;
	slew();
	trace ("\nLimit\n");
	trace ("Slewing %s @ %u to %lu:", getDirName(), slew_speed, radio_freq);

    } else {

	// update mismatch using current radio frequency.
	// not crazy fast so we can detect trends
	setDDSFreq (radio_freq);
	activeDelay(5);
	uint16_t mismatch = readMismatch();
	readings.add(mismatch);
	trace (" %u", mismatch);

	// check for dip trend
	if (readings.dip()) {
	    trace ("\nSlewing %s found dip estimage ", getDirName()); readings.traceDip();

	    // check limits just before stopping
	    checkLimit();

	    // full stop
	    fullStop();

	    // reverse
	    moving_up = !moving_up;

	    // read step duration from pot
	    step_time = analogRead (stepADCPin) / 32;		// 0 .. 31 ms

	    // move one initial kick back over much of the overshoot
	    step (step_time*KICK_FACTOR);

	    // init history from mismatch at this location
	    mismatch = readMismatch();
	    readings.reverse(mismatch);

	    // start stepping
	    current_state = STATE_SRCH_FINAL;

	    trace ("Stepping %s @ %u: %u", getDirName(), step_time, mismatch);
	}
    }
}


/* repeatedly step, measure, check for dip
 * can be cancelled by tapping any switch.
 */
void
finalTune()
{
    if (switches) {

	// cancelled
	trace ("\nTuning canceled by operator\n\n");
	enable_autotune = false;
	stop();
	setDDSFreq (0UL);
	waitForNoSwitches();
	current_state = STATE_STOPPED;

    } else if (hit_limit) {

	// hit limit: start over
	hit_limit = false;
	trace ("\nLimit during final tuning\n");
	startTuning();

    } else {

	// move one small step then come to a complete stop
	step(step_time);

	// update mismatch using current radio frequency
	setDDSFreq(radio_freq);
	uint16_t mismatch = readMismatch();
	readings.add(mismatch);
	trace (" %u", mismatch);

	// check for promising increasing trend
	if (readings.passed()) {

	    trace ("\nStepping %s passed dip", getDirName());
	    readings.tracePassed();

	    // turn around, go slower
	    moving_up = !moving_up;
	    step_time = 7*step_time/10;			// half power

	    // reverse history from here
	    readings.reverse(mismatch);

	    // done if essentially stopped
	    if (step_time <= 5) {

		uint32_t dt = now - tune_t0;
		trace ("Final mismatch %u at %lu took %lu.%lu secs.\n\n",
			readMismatch(), radio_freq, dt/1000, (dt%1000)/100);

		// turn off DDS
		setDDSFreq (0UL);

		// record antenna is now at radio frequency
		antenna_freq = radio_freq;

		// done
		current_state = STATE_STOPPED;

	    } else {

		trace ("Stepping %s @ %u: %u", getDirName(), step_time, mismatch);

	    }
	}
    }
}


/* bump the motor for dt ms, wait here until full stop
 */
void
step (uint16_t dt)
{
    // start moving
    if (moving_up) {
	digitalWrite (goingUpLED, HIGH);
	digitalWrite (goingDownLED, LOW);
	analogWrite (hUpPin, 255);
	analogWrite (hDownPin, 0);
    } else {
	digitalWrite (goingUpLED, LOW);
	digitalWrite (goingDownLED, HIGH);
	analogWrite (hUpPin, 0);
	analogWrite (hDownPin, 255);
    }

    // wait for desired time
    activeDelay (dt);

    // check limits before stopping
    checkLimit();

    // full stop
    fullStop();
}


/* command a constant move at slew_speed in direction determined by moving_up.
 * N.B. only change PWM when value changes to avoid rapid resets.
 */
void
slew(void)
{
    // new PWM if anything changed
    if (current_state != last_state || last_dir != moving_up || last_speed != slew_speed) {
	if (moving_up) {
	    digitalWrite (goingUpLED, HIGH);
	    digitalWrite (goingDownLED, LOW);
	    analogWrite (hUpPin, slew_speed);
	    analogWrite (hDownPin, 0);
	} else {
	    digitalWrite (goingUpLED, LOW);
	    digitalWrite (goingDownLED, HIGH);
	    analogWrite (hUpPin, 0);
	    analogWrite (hDownPin, slew_speed);
	}
    }

    // save conditions
    last_state = current_state;
    last_dir = moving_up;
    last_speed = slew_speed;
}


/* turn off the motor and the direction LEDs.
 */
void
stop(void)
{
    digitalWrite (goingUpLED, LOW);
    digitalWrite (goingDownLED, LOW);
    analogWrite (hDownPin, 0);
    analogWrite (hUpPin, 0);

    last_speed = 0;
}


/* stop and wait here for antenna to definitely stop moving.
 * N.B. requires DDS to be On
 */
void
fullStop()
{
    // stop pulse train
    stop();

    // wait for coasting to really stop as indicated by a largely unchanging match reading
    int lm, m = readMismatch();
    do {
	activeDelay(100);
	lm = m;
	m = readMismatch();
    } while (abs(lm-m) > m/100);
}


/* set up to start a full sweep
 */
void
startSweep()
{
    // insure motor is off
    stop();

    // init sweep frequency
    sweep_hz = LO_HZ;

    // connect antenna to DDS
    digitalWrite (coaxRelayPin, HIGH);

}


/* perform next sweep step, set to STOPPED when finished.
 * can be cancalled by tapping any switch
 */
void
updateSweep()
{
    static uint8_t blink, led;

    // set frequency, report and increment
    setDDSFreq (sweep_hz);
    trace ("%8lu %4u\n", sweep_hz, readMismatch());
    sweep_hz += 2000;

    // display sweeping pattern
    blink = (blink+1) & 0XF;
    if (!blink) {
	// change LED
	digitalWrite (all_leds[led], LOW);
	led = (led+1)%N_LEDS;
	digitalWrite (all_leds[led], HIGH);
    }

    // finished when at upper frequency end or user taps any switch
    if (sweep_hz > HI_HZ || switches) {
	setDDSFreq (0UL);			// turn off DDS
	digitalWrite (coaxRelayPin, LOW);	// turn off relay
	for (uint8_t i = 0; i < N_LEDS; i++)
	    digitalWrite (all_leds[i], LOW);	// turn off lights
	if (switches)				// report reason for stopping
	    trace ("User cancelled freq sweep\n");
	else
	    trace ("Sweep complete\n");
	current_state = STATE_STOPPED;		// stopped state
	waitForNoSwitches();			// wait for release
    }
}


/* control frequency with pots.
 * Slew gives course control, Step gives fine control.
 * can be cancalled by tapping any switch
 */
void
updatePotFreq()
{
    // define freq range
    const uint32_t hz_range = HI_HZ - LO_HZ;
    const uint32_t course_step = hz_range/1024;
    static uint16_t course_avg, fine_avg;
    static uint32_t last_hz;

    // read new values
    uint16_t course_raw = analogRead (slewADCPin);
    uint16_t fine_raw   = analogRead (stepADCPin);

    // fold into running exponential average
    course_avg = (9*course_avg + course_raw)/10;
    fine_avg = (9*fine_avg + fine_raw)/10;

    // find new hz
    uint32_t hz = LO_HZ + course_step*course_avg + course_step*fine_avg/1024;

    // install and measure if different
    if (hz != last_hz) {
	setDDSFreq (hz);
	trace ("%4u %4u %7lu %4u\n", course_avg, fine_avg, hz, readMismatch());
	last_hz = hz;
    }

    // don't go crazy fast
    activeDelay(10);

    // cancel if any switch is tapped
    if (switches) {
	setDDSFreq (0UL);			// turn off DDS
	digitalWrite (goingUpLED, LOW);		// turn off lights indicating pot mode
	digitalWrite (goingDownLED, LOW);	// turn off lights indicating pot mode
	trace ("User cancelled pot freq\n");	// report
	current_state = STATE_STOPPED;		// stopped state
	waitForNoSwitches();			// wait for release
    }
}


/* control the coax switch to connect antenna to Fox Delta if not transmitting and
 * either manually controlling frequency or tuning.
 */ 
void
setCoaxSwitch()
{
    bool toFoxDelta = !transmitting_now && 
	    (current_state == STATE_SWEEP || current_state >= STATE_SRCH_SCAN);
    digitalWrite (coaxRelayPin, toFoxDelta);
}


/* read switch states and set switches mask
 */
void
readSwitches()
{
    switches = 0;
    if (digitalRead (manualDownSw) == LOW)
	switches |= SW_DOWN;
    if (digitalRead (manualUpSw) == LOW)
	switches |= SW_UP;
    if (digitalRead (tuneSw) == LOW)
	switches |= SW_TUNE;
}


/* wait for no switches to be active.
 * this is used to cancel an operation without subsequently activating something else.
 */
void
waitForNoSwitches()
{
    while (switches) {
	activeDelay (50);
	readSwitches();
    }
}


/* indicate with the LEDs error code e.
 * [arg should be type Err but Ardunio IDE munging breaks it]
 */
void
blinkErr (uint8_t e)
{
    // avoid recursion with checkRadioMessages() with ERR_TOOLO and ERR_TOOHI
    static uint8_t working_e;
    if (e == working_e)
	return;
    working_e = e;

    trace ("Error %u: %s\n", e, blink_descriptions[e]);

    for (uint8_t i = 0; i < 4; i++) {
	for (uint8_t j = 0; j < e; j++) {
	    for (uint8_t k = 0; k < 2; k++) {
		for (uint8_t led = 0; led < N_LEDS; led++)
		    digitalWrite (all_leds[led], !k);
		activeDelay (200);
	    }
	}
	activeDelay (500);
    }

    working_e = ERR_NONE;

}


/* like delay() but avoids missing any radio messages.
 * we also keep now up to date.
 */
void
activeDelay(uint16_t ms)
{
    for (uint32_t done = now + ms; (now = millis()) < done; delay(1))
	checkRadioMessages();
}


/* given an antenna frequency in Hz, return the maximum useable half-bandwidth at that frequency.
 */
uint32_t
antBandwidth (uint32_t hz)
{
    uint32_t khz = hz/1000UL;				// reduce precision to avoid overflow
    return (1000UL*ANT_BW*khz/ANT_F0*khz/ANT_F0);	// goes as f^2
}


/* pulse the given pin HIGH then LOW
 */
void
pulsePinHigh (uint8_t pin)
{
    digitalWrite (pin, HIGH);
    digitalWrite (pin, LOW);
}


/* send the given byte to the DDS, LSBit first
 */
void
sendDDSByte (uint8_t b)
{
    for (uint8_t i = 0; i < 8; i++) {
	digitalWrite (DDSdataPin, (b&1) ? HIGH : LOW);
	pulsePinHigh (DDSclockPin);
	b >>= 1;
    }
}


/* set the DDS to the given frequency plus FREQ_OFFSET.
 * call with 0 to disable output.
 * don't keep hammering with the same freq
 */
void
setDDSFreq (uint32_t hz)
{
    static uint32_t last_tune;

    // compute tuning word = Hz * 2^32 / 125E6 clock freq, see data sheet pg 8.
    uint32_t tune = (uint64_t)(hz+FREQ_OFFSET) * (1ULL<<32) / 125000000ULL;

    // skip if same
    if (tune == last_tune)
	return;
    last_tune = tune;

    // send tuning word LSByte first, see data sheet pg 12
    for (uint8_t i = 0; i < 4; i++) {
	sendDDSByte (tune & 0xff);
	tune >>= 8;
    }

    // send final control byte
    sendDDSByte (0);

    // engage!
    pulsePinHigh (DDSstrobePin);

    // delay a bit for stability
    delay (5);
}


/* set hit_limit if we just encountered a real limit and update LED.
 * N.B. only call while ostensibly moving.
 * N.B. require limit detect to be on for a short time to allow for PWM and motor inductance.
 * N.B. we only _set_ hit_limit, consumer must _clear_
 */
void
checkLimit()
{
    static unsigned long t_start;
    static bool raw_was_on, was_real;
    bool raw_is_on, is_real = false;

    // get instantaneous apparent state
    raw_is_on = (digitalRead (limitDetect) == HIGH);

    // consider real only if stays on at least a short while
    if (raw_is_on) {
	if (!raw_was_on)
	    t_start = now;
	else if (t_start > 0UL && now > t_start + 1000)
	    is_real = true;
    } else {
	// effectively cancel the timer
	t_start = 0;
    }
    digitalWrite (limitLED, is_real);

    // report if hit a real limit
    if (is_real && !was_real)
	hit_limit = true;

    // save for next time
    raw_was_on = raw_is_on;
    was_real = is_real;
}

/* return the current level of mismatch from 50 ohms: the smaller the better.
 */
uint16_t
readMismatch(void)
{
    return (analogRead (AD8307Pin));
}


/* set BAD_MATCH and FREQ_OFFSET from EEPROM
 */
void
readEEPROM()
{
    // stored as little endian
    BAD_MATCH = ((EEPROM.read(1) << 8) | (EEPROM.read(0)));
    FREQ_OFFSET = ((EEPROM.read(3) << 8) | (EEPROM.read(2)));
}


/* write BAD_MATCH and FREQ_OFFSET to EEPROM
 */
void
writeEEPROM()
{
    // stored as little endian
    EEPROM.write (0, BAD_MATCH & 0xFF); EEPROM.write (1, (BAD_MATCH>>8));
    EEPROM.write (2, FREQ_OFFSET & 0xFF); EEPROM.write (3, (FREQ_OFFSET>>8));
}


/* flash lights to indicate we will read the mismatch and store in EEPROM when Tune switch is released.
 */
void
calibrateBadMatch()
{
    uint8_t rotate = 0;

    // flash until Tune released
    trace ("Release Tune to perform calibration ...");
    while (digitalRead (tuneSw) == LOW) {
	digitalWrite (all_leds[rotate], HIGH);
	delay(50);
	digitalWrite (all_leds[rotate], LOW);
	delay (50);
	rotate = (rotate + 1) % N_LEDS;
    }
    trace ("\n");

    // turn on DDS to upper freq where mismatch tends to be lower
    setDDSFreq (HI_HZ);

    // disconnect antenna from foxdelta
    digitalWrite (coaxRelayPin, false);		
    delay(500);

    // sample several times for five seconds to get noise stats
    const uint16_t N_CAL_MS = 5000;
    const uint16_t N_CAL_NOISE = 100;
    float sum = 0, sum2 = 0;
    trace ("BAD_MATCH Calibration values:");
    rotate = 0;
    for (uint8_t i = 0; i < N_CAL_NOISE; i++) {
	float m = readMismatch();
	trace (" %u", (uint16_t)m);
	sum += m;
	sum2 += m * m;

	digitalWrite (all_leds[rotate], HIGH);
	delay(N_CAL_MS/N_CAL_NOISE);
	digitalWrite (all_leds[rotate], LOW);
	rotate = (rotate - 1 + N_LEDS) % N_LEDS;
    }

    // set BAD_MATCH well below the mean noise
    uint16_t mean = 0.95*sum/N_CAL_NOISE + 0.5;	// antenna open is a little worse than any connected value
    uint16_t stddev = sqrt(N_CAL_NOISE*sum2 - sum*sum)/(N_CAL_NOISE-1) + 0.5;
    BAD_MATCH = mean - 5*stddev;
    trace ("\ncorrected mean %u stddev %u => BAD_MATCH = %u\n", mean, stddev, BAD_MATCH);

    // DDS off
    setDDSFreq (0UL);

    // store and confirm
    uint16_t bm = BAD_MATCH;
    writeEEPROM ();			// write BAD_MATCH to EEPROM
    BAD_MATCH = 12345;			// any unlikely value
    readEEPROM();			// read BAD_MATCH from EEPROM
    if (bm != BAD_MATCH) {
	trace ("EEPROM error: wrote %u read %u\n", bm, BAD_MATCH);
	blinkErr(ERR_EEPROM);
    }
}


/* adjust Step pot for zero beat whole holding Up.
 */
void
calibrateFreqOffset(void)
{
    uint16_t blinky;

    // wait to read freq from radio
    while (!radio_freq) {
	checkRadioMessages();
	digitalWrite (ComOkLED, (++blinky & 0x1000) ? HIGH : LOW);
    }
    digitalWrite (ComOkLED, LOW);
    uint32_t rf0 = radio_freq;

    // wait for op to tune and release Up
    while (digitalRead (manualUpSw) == LOW) {
	FREQ_OFFSET = ((int16_t)(analogRead (stepADCPin)) - 512)*4;	// +- 2 KHz
	setDDSFreq (rf0);
	digitalWrite (goingUpLED, (++blinky & 0x80) ? HIGH : LOW);
    }
    digitalWrite (goingUpLED, LOW);

    // turn off DDS
    setDDSFreq (0UL);

    // store and confirm
    int16_t fo = FREQ_OFFSET;
    writeEEPROM ();			// write FREQ_OFFSET to EEPROM
    FREQ_OFFSET = 12345;		// any unlikely value
    readEEPROM();			// read FREQ_OFFSET from EEPROM
    if (fo != FREQ_OFFSET) {
	trace ("EEPROM error: wrote %d read %d\n", fo, FREQ_OFFSET);
	blinkErr(ERR_EEPROM);
    }
}


/* return a string that describes the current value of moving_up
 */
const char *
getDirName()
{
    return (moving_up ? "up" : "down");
}


/* process all pending messages from radio
 */
void
checkRadioMessages()
{
    uint8_t index;

    /* slow blink ComOkLED until we receive first radio freq.
     * N.B. don't disturb the LED if being used by other states.
     */
    if (current_state == STATE_STOPPED || current_state == STATE_MOVING) {
	if (!radio_freq)
	    digitalWrite (ComOkLED, (newfreq_time++ & 0x800) ? HIGH : LOW);
    }

    /* crack all pending messages
     */
    while (nextRXResponse(&index)) {

	// first two chars are always the response code
	char cmd1 = kx3rxchars[index];
	QADVANCE (index, RXCHARSMASK);
	char cmd2 = kx3rxchars[index];
	QADVANCE (index, RXCHARSMASK);

	// which response?
	if (cmd1 == 'I' && cmd2 == 'F') {

	    // IF: reporting frequency and modes

	    // first 11 chars are freq in Hz
	    uint32_t new_freq = 0UL;
	    bool freq_ok = true;
	    uint8_t c;
	    for (uint8_t i = 0; i < 11; i++) {

		c = kx3rxchars[index];
		QADVANCE (index, RXCHARSMASK);
		if (isdigit(c)) {
		    c -= '0';		// ASCII to numeric value
		    new_freq = 10UL*new_freq + c;
		} else {
		    freq_ok = false;	// note bad char
		    break;
		}

	    }

	    if (freq_ok) {

		// skip 15 chars to reach TR mode
		for (uint8_t i = 0; i < 15; i++)
		    QADVANCE (index, RXCHARSMASK);
		c = kx3rxchars[index];
		switch (c) {
		case '0':
		    transmitting_now = false;
		    break;
		case '1':
		    transmitting_now = true;
		    break;
		default:
		    freq_ok = false;
		    break;
		}

		if (freq_ok) {

		    // indicate if radio is now out of antenna range
		    // N.B. beware recursion
		    if (new_freq < LO_HZ)
			blinkErr(ERR_TOOLO);
		    else if (new_freq > HI_HZ)
			blinkErr(ERR_TOOHI);

		    // record time of _new_ radio frequency
		    if (new_freq != radio_freq) {

			newfreq_time = now;
			radio_freq = new_freq;

		    }

		    // blink ComOK to indicate receipt of good IF command
		    digitalWrite (ComOkLED, HIGH);
		    delay(1);
		    digitalWrite (ComOkLED, LOW);

		}
	    }

	}

	// ignore all other messages
    }
}


/* if there is a complete radio message ready:
 *   return true and the index in kx3rxchars[] where the message starts
 * else:
 *   return false
 */
bool
nextRXResponse (uint8_t *msg_start)
{
    // disable interrupts while we use the queue
    char cSREG = SREG;
    cli();

    // get start index of next command, if any
    bool cmd_ready = (rxcm_tail != rxcm_head);
    if (cmd_ready) {
        *msg_start = kx3rxcmds[rxcm_tail];
        QADVANCE (rxcm_tail, RXCMDSMASK);
    }

    // restore interrupts and return
    SREG = cSREG;
    return (cmd_ready);
}


/* interrupt called when new inbound char is available from radio.
 * add to kx3rxchars[] and add to kx3rxcmds[] if it's a ';'
 */
ISR(USART_RX_vect)
{
    char c = UDR0;                              // read new char from radio
    kx3rxchars[rxch_head] = c;                  // store at char q head
    QADVANCE (rxch_head, RXCHARSMASK);         // advance q
    if (c == ';') {				// if char is command terminator
        QADVANCE (rxcm_head, RXCMDSMASK);      //   advance cmd q head
        kx3rxcmds[rxcm_head] = rxch_head;       //   store location after ';'
    }
}


/* send the given printf-style message to the serial output
 * N.B. final message length is limited by size of msg[].
 * N.B. we wait here if necessary for output from previous call to drain.
 */
void
trace (const char *fmt, ...)
{
    // format message into msg, noting length. static to reduce stack requirement.
    static char msg[NTXCHARS];
    va_list args;
    va_start (args, fmt);
    uint8_t l = vsnprintf (msg, sizeof(msg), fmt, args);
    va_end (args);

    // wait for q to drain
    while (txch_head != txch_tail)
	delay(1);

    // disable interrupts while we modify the tx queue
    char cSREG = SREG;
    cli();

    // add message of length l to txchars q
    for (uint8_t i = 0; i < l; i++) {
	txchars[txch_head] = msg[i];
	QADVANCE (txch_head, TXCHARSMASK);
    }

    // insure we will get the UDRE interrupt when transmit buffer is ready for next character
    UCSR0B |= (1<<UDRIE0);

    // resume interrupts
    SREG = cSREG;
}


/* interrupt called when ok to send next outbound char from txchars[], if any
 */
ISR(USART_UDRE_vect)
{
    if (txch_head == txch_tail) {		// if q is empty
	UCSR0B &= ~(1<<UDRIE0);			//   turn off tx interrupt
    } else {
	UDR0 = txchars[txch_tail];		// send next char
	QADVANCE (txch_tail, TXCHARSMASK);	// advance queue
    }
}
