         
;**GPS-PLL Software Version 3.01********************************************
;*** Master/Slave selected by PC************************

; In Master Mode, the lock status is determined using the RMS deviation of the 
; Phase from its mean. 
; Segment 2 starts when lock is achieved. 
;Also, the phase detector gain is changed when segment 2 is entered.


	include  <p16f876a.inc>
	errorlevel 0, -202, -302, -306

; **Set Configuration Bits************************************************

	__Config H'3F3A'


;********************Constants***********************************
; Select Simulate = 0 for no simulation
;        Simulate = 1 for simulation in master mode
;        Simulate = 2 for simulation in slave mode, input voltage packet from PC
;        Simulate = 3 for simulation in slave mode, input mode packet from PC
; 		 Simualte = 4 for simulation in slave mode, new phase measurement
;		 Simulate = 5 to simulation reception of a voltage initialization packet

#define	Simulate	0

#define	DebugI_B	205
#define DebugI_C	120
#define	DebugI_S	0x31

B0	equ	0
B1	equ	1
B2	equ	2
B3	equ	3
B4	equ	4
B5	equ	5
B6	equ	6
B7	equ	7

MSB	equ	7
LSB	equ	0




;*****************CONSTANTS & MEMORY ALLOCATIONS****************************
;

;---Define Various Constants----------------------------------------------

NumSamplesPerMeas	equ		120				; Num of samples per measurement

InitialVolts		equ		1764			; Initial control voltage = 2.153 
PD_Intercept		equ		0x895C78		; Intercept = 1763.75 = 2.153 V * 4096/5
PD_Gain_Seg1		equ		0x84449C		; Gain = 49.152 = 0.06 V * 4096/(5 V)
PD_Gain_Seg2		equ		0x821D49		; Gain = 9.8304 = 0.012 V * 4096/(5 V)
	 
TotalNumSpp		equ		11				; Total # of measurements in Spp calculation

SppCut			equ		0x7F0000		; Pcut = 0.316 so SppCut = 1.	
; SppCut = Pcut^2 * (TotalNumSpp-1) where 
;      Pcut = Cut on Sum((MeasPhase - Mean(MeasPhase))^2/(TotalNumSpp-1))
  
T_NumPassingSppCut equ	1					; Number of passed Spp cuts to achieve lock 

MinPhaseSample		equ		30				; Minimum acceptable phase sample 
MaxPhaseSample		equ		210				; Maximum acceptable phass sample


; Define Segment 1 duration and digital filter parameters
; B = Exp(-NumSamplesPerMeas/TimeConstant)
; A = (1 - B)/2

; Segment 1 time constant = 300 s

Seg1Filter_A		equ		0x7C28CC		; Segment 1 A = 0.16484
Seg1Filter_B		equ		0x7E2B9A		; Segment 1 B = 0.67032

; Segment 2 time constant = 1500 s
Seg2Filter_A		equ		0x7A1D78		; Segment 2 A = .038445
Seg2Filter_B		equ		0x7E6C51		; Segment 2 B = .92311



;  First make general register assignments
 cblock	0x20
; Next group of variables used by MicroChip floating point routines.
	  exp:0, aexp:0
	  aarg,aargb0, aargb1, aargb2
	  aargb3, aargb4, aargb7
	  sign
	  fpflags
	  bexp:0, barg,bargb0,bargb1,bargb2
	  Temp, TempB0, TempB1

; More variables
	NumNewSamples 
	PhaseSample
	PhaseAccum:2
	MeasPhase:3
	Temp1:3
	SerialTextPtr
	Counter2
	DAC_Counter
	TableTemp
	ASCII_Num:8
	Flags
	Volts:3
	PrevInputVolts:3
	PrevOutputVolts:3
	intVolts:2
	TargetPhase:3
	NumMeas
	NumSpp
	NumPassingSppCut
	Spp:3
	fpNumSamplesPerMeas:3
	Debug_B
	Debug_C
	TextAdr
	InputPktFlags
	Flags2
	PktSum:2
	Debug_S
	fpTotalNumSpp:3
	Sp:3

 endc
   
#define	NewSampleFlag		Flags,0
#define ReadyFlag			Flags,1			; 1 = PhaseSample in acceptable range
#define	SeekLockFlag		Flags,2			; 1 means SW1 in lock position (negative logic)
#define	LockedFlag			Flags,3			; 1 = Lock achieved (LED2 on)
#define	Seg2Flag			Flags,4			; 1 = In Segment 2, 0 = In Segment 1
#define SecondSampleFlag	Flags,5			; 0 = First sample
#define	NegativeFlag		Flags,7


#define	MasterMode		Flags2,0
#define	SlaveMode		Flags2,1
#define	RunModeFlag		Flags2,2
#define	ResetFlag		Flags2,3

#define	NewPkt				InputPktFlags,3
#define	PktTyped			InputPktFlags,4
#define	VoltPkt				InputPktFlags,5
#define	ModePkt				InputPktFlags,6
#define	InitializationPkt	Flags2,7
#define	PktReceived			InputPktFlags,7


	cblock	0x7D

	wtemp, statustemp, PCLATHtemp

	endc


; Constants used for selecting pages

psel0		equ		3
psel1		equ		4


	#define	_C		STATUS,0
	#define	_Z		STATUS,2

; Define symbols for inputs and outputs

	#define	Counter1to7	PortB
	#define Counter0	PortA,0
    #define	WaitSW			PortA,2
	#define	FreeRunSW			PortA,1
	#define	CounterReset	PortC,0
	#define	DAC_Data	PortC,1
	#define	DAC_Clock	PortC,2
	#define	DAC_Load	PortC,3
	#define	LED1		PortC,5
	#define	LED2		PortC,4
	#define LED1Mask	0x20
	#define	LED2Mask	0x10

;
;       FLOATING POINT SPECIFIC DEFINITIONS
;
;       literal constants
;
EXPBIAS         equ     D'127'
;
;       floating point library exception flags
;

IOV	equ     0       ; bit0 = integer overflow flag
FOV     equ     1       ; bit1 = floating point overflow flag
FUN     equ     2       ; bit2 = floating point underflow flag
FDZ     equ     3       ; bit3 = floating point divide by zero flag
NAN	equ	4       ; bit4 = not-a-number exception flag
DOM	equ	5	; bit5 = domain error exception flag
RND     equ     6       ; bit6 = floating point rounding flag, 0 = truncation
SAT     equ     7       ; bit7 = floating point saturate flag, 0 = terminate on
                                ; exception without saturation, 1 = terminate on
                                ; exception with saturation to appropriate value

;*************MACRO DEFINITIONS*************************************

; Changes page bits in PCLATH. FromPage and ToPage can be 0, 1, 2, or 3 depending
; on page number

ChangePage	macro	FromPage,ToPage
	if (0x1 & FromPage) != (0x1 & ToPage) 
		if (0x1 & ToPage) != 0
			bsf		PCLATH,3
		else 
			bcf		PCLATH,3
		endif
	endif

	if (0x2 & FromPage) != (0x2 & ToPage) 
		if (0x2 & ToPage) != 0
			bsf		PCLATH,4
		else 
			bcf		PCLATH,4
		endif
	endif
 endm


; Changes register bank. FromBank and ToBank can be 0, 1,2, or 3 depending on the 
; respective bank number. 

ChangeBank	macro	FromBank,ToBank
	if	(0x1 & FromBank) != (0x1 & ToBank)
		if (0x1 & ToBank) != 0
			bsf		Status,rp0
		else
			bcf		Status,rp0
		endif
	endif

	if	(0x2 & FromBank) != (0x2 & ToBank)
		if (0x2 & ToBank) != 0
			bsf		Status,rp1
		else
			bcf		Status,rp1
		endif
	endif
 endm


; This macro moves nbytes from AA, located in any bank, to BB, also located in any bank

MoveBytes	macro	AA,BB,nbytes

	local	AA_rp0 = (0x80 & AA) >> 7
	local	AA_rp1 = (0x100 & AA) >> 8	
	local	BB_rp0 = (0x80 & BB) >> 7
	local	BB_rp1 = (0x100 & BB) >> 8	

		if AA_rp0 == BB_rp0
			if AA_rp0 == 1
				bsf		status,rp0
			else
				bcf		status,rp0
			endif
		endif
		
		if AA_rp1 == BB_rp1
			if AA_rp1 == 1
				bsf		status,rp1
			else
				bcf		status,rp1
			endif
		endif

     local i = 0			 
	 while i < nbytes		
		if AA_rp0 != BB_rp0
			if AA_rp0 == 1
				bsf		status,rp0
			else
				bcf		status,rp0
			endif
		endif
		if AA_rp1 != BB_rp1
			if AA_rp1 == 1
				bsf		status,rp1
			else
				bcf		status,rp1
			endif
		endif
	  movf	AA + i,w
		if AA_rp0 != BB_rp0
			if BB_rp0 == 1
				bsf		status,rp0
			else
				bcf		status,rp0
			endif
		endif
		if AA_rp1 != BB_rp1
			if BB_rp1 == 1
				bsf		status,rp1
			else
				bcf		status,rp1
			endif
		endif
	  movwf		BB + i
i		set	i + 1
	endw
 endm


; This macro moves nbytes # of bytes from AA to BB, both located in bank 0

MoveBytes0  macro  AA, BB, nbytes

     local i = 0			; a through a + nbytes - 1 to
	 while i < nbytes		; b through b + nbytes - 1
		movf	AA + i,w
		movwf	BB + i
i		  set	i + 1
	  endw
    endm



; This macro stores a single- or multi-byte literal at the location Address. 
; nbytes is the number of bytes in the literal and can be between 1 and 4. 
; The register bank number must be set prior to using this macro

StoreLiteral	macro	Literal, Address,nbytes
	local i = 0
	while i < nbytes
		movlw	((0xFF << 8*(nbytes - 1 - i)) & Literal) >> 8*(nbytes - 1 - i)
		movwf	Address + i
i		set i + 1
	endw
	endm



;This routine outputs the text string through the serial port.
; Text is a string containing the text to be output. 
; Uadr must be an unique address, not used anywhere else in the program. 


Serial_OutputText	macro Text,Uadr
	movlw	HIGH($+6)
	movwf	PCLATH
	movlw	LOW($+4)
	movwf	TextAdr
	call	OutputText
	goto	Uadr
	dt		Text,0
Uadr
	endm




;********* CODE FOR NORMAL MODE *************************************************************

	org   	0x0
    	goto	StartUp

; This interrupt driven routine is the inner sampling loop. 

	org		0x04
ISR
 if Simulate == 0
	btfsc	PIR1,RCIF		; Has a character been received on the serial interface?
	goto	ReceivePkt		; Yes
	btfss	INTCON,INTF		; Check for RB0 interupt
	retfie        ; No interrupt so return.
 endif
 if Simulate == 2 || Simulate == 3 || Simulate == 5 
 	goto	ReceivePkt
 endif


; Throw away the very first sample
	btfsc	SecondSampleFlag
	goto	$ + 3
	bsf		SecondSampleFlag
	goto	LeavePhaseMeas	

	btfss	WaitSW				; Process measurement only if in Run mode
	goto	ISR001				; Switch is in Run Mode position
	btfss	RunModeFlag			; Switch not in Run Mode position, but was it previously?
	goto	LeavePhaseMeas				; No, never previously in Run Mode

	bsf		ResetFlag			; Was previously in Run Mode so reset everything
	goto	LeavePhaseMeas	

ISR001
	bsf		RunModeFlag
	; RB0 interrupt. New phase measurement ready for processin
	movlw	LED1Mask		; Toggle LED1
	xorwf	PortC,f

	movf	Counter1to7,w	; Read Phase Counter
	movwf	PhaseSample
	bcf		PhaseSample,0
	btfsc	Counter0
	bsf		PhaseSample,0

 if Simulate ==1 || Simulate == 4
	movf	Debug_B,w
	movwf	PhaseSample
	decfsz	Debug_C,f
	goto	$+4
	decf	Debug_B,f
	movlw	DebugI_C
	movwf	Debug_C
 endif
	
	btfss	MasterMode				; Are we in master mode?
	goto	ISR100					; No. 

; In Master mode, so process phase measurement

	movf	PhaseSample,w		; Add latest PhaseSample to PhaseAccum
	addwf	PhaseAccum+1,f
	btfsc	_C
	incf	PhaseAccum,f

	incf	NumNewSamples,f		; Keep track of number of samples in Phase Accumulator
	bsf		NewSampleFlag		; Set New Sample Flag		
	goto	LeavePhaseMeas		; Done. 

ISR100							; In slave mode, so send PhaseSample to PC
	movlw	"@"
	call	Serial_TxChar
	movf	PhaseSample,w		; Convert Phase Sample to ASCII
	movwf	aargb0+1
	clrf	aargb0
	call	Int16ToASCII
	movlw	"P"
	call	Serial_TxChar
	movf	ASCII_Num,w
	call	Serial_TxChar
	movf	ASCII_Num+1,w
	call	Serial_TxChar
	movf	ASCII_Num+2,w
	call	Serial_TxChar
	movf	ASCII_Num+3,w
	call	Serial_TxChar
	movf	ASCII_Num+4,w
	call	Serial_TxChar
	movlw	"$"
	call	Serial_TxChar

LeavePhaseMeas
	bsf		CounterReset		; Reset phase counter to 0
	bcf		CounterReset
	bcf		INTCON,INTF
	retfie

ReceivePkt
	btfsc	NewPkt				; Is this the beginning of a new packet?
	goto	ISR601				; No
	movf	RCREG,w
 if Simulate==2 || Simulate==3 || Simulate == 5
	movlw	"@"
 endif				; Yes. Character should be "@"
	sublw	"@"
	btfsc	_Z
	bsf		NewPkt				; Yes, is start of new packet
	goto	LeaveReceivePkt

ISR601							; second or later character of packet
	btfsc	PktTyped			; Has second character already been read? 
	goto	ISR620
	movf	RCREG,w
 if Simulate==2
	movlw	"V"
 endif
	sublw	"V"					; Is this a voltage packet?
	btfss	_Z					; Skip if yes
	goto	ISR602				; Otherwise go on to see if a mode packet
	bsf		PktTyped
	bsf		VoltPkt
	goto	LeaveReceivePkt

ISR602							; Now see if packet is a mode type
	movf	RCREG,w
 if Simulate==3
	movlw	"M"
 endif
	sublw	"M"
	btfss	_Z
	goto	ISR603				; not mode packet so see if an voltage initialization packet 
	bsf		PktTyped			; Is a Mode packet
	bsf		ModePkt
	goto	LeaveReceivePkt

ISR603							; Is packet an initialization packet?
	movf	RCREG,w
 if Simulate == 5
	movlw	"I"
 endif
	sublw	"I"
	btfss	_Z
	goto	ISR650
	bsf		PktTyped
	bsf		InitializationPkt
	goto	LeaveReceivePkt

ISR620				; Packet has been identified, so next read data bytes
	movf	InputPktFlags,w
	incf	InputPktFlags,f	; Keep track of # data bytes read
	andlw	0x07	; Mask off all but bottom 3 bits
	btfsc	_Z		; Is count zero?
	goto	ISR621	; Yes, so skip ahead
	sublw	4		; Is count > 5, then all data bytes read
	btfss	_C
	goto	ISR630	; All data bytes read, so go on to last byte

	bcf		_C
	rlf		PktSum+1,f		; Multiply by 2 (one left shift)
	rlf		PktSum,f	
	MoveBytes0	PktSum,aargb0,2
	bcf		_C
	rlf		aargb0+1,f		; Multiply by 4 (two left shifts)
	rlf		aargb0,f
	rlf		aargb0+1,f
	rlf		aargb0,f
	movf	aargb0+1,w
	addwf	PktSum+1,f
	btfsc	_C
	incf	PktSum,f
	movf	aargb0,w
	addwf	PktSum,f
ISR621
	movlw	0x30
 if Simulate ==0
	subwf	RCREG,w				; Convert next data byte from ASCII
 else
	subwf	Debug_S,w
	incf	Debug_S,f
 endif
	addwf	PktSum+1,f
	btfsc	_C
	incf	PktSum,f
	goto	LeaveReceivePkt

ISR630							; Come here is last character of packet read
	movf	RCREG,w				; Last character should be "$"
 if Simulate > 0
	movlw	"$"
 endif
	sublw	"$"
	btfss	_Z
	goto	ISR650				; Not "$" so packet invalid. Discard
	btfss	ModePkt			; Is this a mode packet? 
	goto	ISR640			; no
	btfss	PktSum+1,0		; If PktSum = 1 then enter Master Mode
	goto	ISR632
	bsf		MasterMode		; Yes, so set master mode
	bcf		SlaveMode
	goto	ISR650

ISR632						; Not Master Mode so must be Slave Mode
	bcf		MasterMode
	bsf		SlaveMode
	goto	ISR650

ISR640					; Come here if voltage packet
	btfss	VoltPkt		; Is it really a voltage packet:
	goto	ISR645		; No so discard packet
	movf	PktSum,w	; Is voltage > 4095? 
	andlw	0xF0
	btfsc	_Z
	goto	ISR641		; OK, voltage <= 4095
	movlw	0x0F		; Voltage > 4095, so set = 4095
	movwf	intVolts
	movlw	0xFF
	movwf	intVolts+1
	goto	ISR642

ISR641					; Voltage <= 4095
	MoveBytes0	PktSum,intVolts,2

ISR642
	call	DAC_OutputVolts
	goto	ISR650

ISR645					; Is this an initialization packet?
	btfss	InitializationPkt
	goto	ISR650		; Unknown type of packet, so abort
	MoveBytes0	PktSum,intVolts,2
	call	InitializeVoltage


ISR650
	clrf	InputPktFlags		; Initialize in preparation for next packet
	bcf		InitializationPkt
	clrf	PktSum
	clrf	PktSum+1

LeaveReceivePkt
	bcf		PIR1,RCIF			; Prepare serial receive interrupt for next request
	retfie
	


StartUp	

 if Simulate > 0
  	movlw	DebugI_B
	movwf	Debug_B
	movlw	DebugI_C
	movwf	Debug_C
	movlw	DebugI_S
	movwf	Debug_S
 endif


; Initialize I/O ports

	ChangeBank 0,1
	
	movlw	0x06
	movwf	ADCON1
	movlw	0x3F		; PORTA: all 6 bits inputs
	movwf	TRISA

	movlw	0xFF		; PORTB: all 8 bits inputs normally
	movwf	TRISB

	movlw	0x80		; PORTC: Bits 0-6 outputs, bit 7 serial input
	movwf	TRISC	

	ChangeBank 1,0

; Initialize floating point routines by selecting "rounding" and "saturation"

	movlw	0xC0
	movwf	fpflags	

; Calculate fpNumSamplesPerMeas
	movlw	NumSamplesPerMeas
	movwf	aargb1
	clrf	aargb0
	ChangePage 0,1
	call	FLO1624
	ChangePage 1, 0
	MoveBytes0	aarg,fpNumSamplesPerMeas,3	

; Calculate fpTotalNumSpp

	movlw	TotalNumSpp
	movwf	aargb0+1
	clrf	aargb0
	ChangePage 0,1
	call	FLO1624
	ChangePage 1,0
	MoveBytes0	aarg,fpTotalNumSpp,3	

; Setup USART port for asynchronous I/O 9600 baud, no parity, 1 stop bit

	ChangeBank 0,1		; Select bank 1
	movlw	25			; Select for 9600 baud, no parity, 1 stop bit
	movwf	SPBRG
	bsf		TXSTA,BRGH	; Choose "high" baud rate generator
	ChangeBank 1,0	; Select bank 0
	bsf		RCSTA,SPEN	; Enable serial port
	ChangeBank 0,1	; bank 1
	bsf		TXSTA,TXEN	; Enable serial port transmission

	ChangeBank	1,0
	bsf	RCSTA,CREN		
	ChangeBank	0,1

	bcf		OPTION_REG,INTEDG		; Set RB0 interrupt to occur on falling edge
	ChangeBank 1,0
	bsf		INTCON,INTE	; unmask RB0 Interrupt

	ChangeBank	0,1
	bsf		PIE1,RCIE			; Unmask RB0 Interrupt
	ChangeBank	1,0
	bsf		INTCON, PEIE		;Enable peripheral interrupts

; Set initial control voltage for Rb oscillator
	movlw	HIGH(InitialVolts)
	movwf	intVolts
	movlw	LOW(InitialVolts)
	movwf	intVolts+1
	call	InitializeVoltage


; Initialize variables	
	clrf	PhaseAccum
	clrf	PhaseAccum+1
	clrf	NumNewSamples
	clrf	Flags
	clrf	NumPassingSppCut
	clrf	PhaseSample
	clrf	NumSpp
	clrf	Spp
	clrf	Spp + 1
	clrf	Spp + 2
	clrf	Sp
	clrf	Sp+1
	clrf	Sp+2
	clrf	NumMeas
	bcf		LED1
	bcf		LED2

	clrf	Flags2
 if Simulate == 0 || Simulate == 1 || Simulate == 5
    bsf		MasterMode
 else
	bsf		SlaveMode
 endif
	clrf	InputPktFlags
	clrf	PktSum
	clrf	PktSum+1

	bsf		INTCON,GIE		; Turn on (arm) all unmasked interrupts

;	Waiting loop. See if enough phase samples taken 
Main

 if Simulate > 0
	call	ISR
 endif
	btfsc	ResetFlag
	goto	StartUp
 	btfss	NewSampleFlag	; Is there a new sample ready for processing? 
	goto	Main			; No. 
	
	bcf		NewSampleFlag	; Clear the flag to make ready for next sample.
; Now test to see if PhaseSample satisfies MinPhaseSample <= PhaseSample <= MaxPhaseSample
	btfsc	SeekLockFlag		; Are we already in SeekLock mode?
	goto	Main050		; Yes, go on to next step
	
	movlw	MinPhaseSample	; Is PhaseSample >= MinPhaseSample?
	subwf	PhaseSample,w
	btfsc	_C			
	goto	Main020			; Yes, go to next test
	clrf	intVolts
	clrf	intVolts+1
	call	DAC_OutputVolts
	bsf		LED1			; Turn both LED's on
	bsf		LED2	
	bcf		ReadyFlag
	goto	Main050	

Main020
	movlw	MaxPhaseSample	; Is PhaseSample <= MaxPhaseSample?
	subwf	PhaseSample,w
	btfss	_C
	goto	Main040			; Yes, go on
	movlw	0x0F			; No, PhaseSample too large. Set voltage = 5.0 V
	movwf	intVolts
	movlw	0xFF
	movwf	intVolts+1
	call	DAC_OutputVolts
	bsf		LED1
	bsf		LED2
	bcf		ReadyFlag
	goto	Main050

Main040						; Phase is in acceptable range
	btfsc	ReadyFlag		; Is ready flag already set?
	goto	Main050			; Yes, go on
	bsf		ReadyFlag		; No
	bcf		LED2

	movlw	HIGH(InitialVolts)		; Set voltage to initial value
	movwf	intVolts
	movlw	LOW(InitialVolts)
	movwf	intVolts+1
	call	InitializeVoltage

Main050
	movlw	NumSamplesPerMeas			; Have we accumulated enough samples?
	subwf	NumNewSamples,w
	btfss	_C
	goto	Main						; No, keep waiting

	clrf	NumNewSamples							; Yes, make ready for next sample
	MoveBytes0	PhaseAccum,aargb0,2
	
	clrf	PhaseAccum
	clrf	PhaseAccum+1				; Now ready for more samples

	ChangePage	0,1
  	call	FLO1624			; Convert PhaseAccum to floating point
	MoveBytes0 fpNumSamplesPerMeas,barg,3	; Computer mean of phase samples
	call	FPD24
	MoveBytes0	aarg,MeasPhase,3
	ChangePage	1,0

	movlw	"P"
	call	SendPhaseToPC	; Send MeasPhase to PC over Serial Link

	btfss	ReadyFlag		; Is ready flag set? If not, bypass all locking code
	goto	Main201

	btfsc	FreeRunSW					; Is lock switch thrown, i.e., line low
	goto	Main201			; If not, bypass all locking code

	btfsc	SeekLockFlag		; Is SeekLockFlag already set? 
	goto	Main102			; SeekLockFlag already set. skip ahead

	bsf		SeekLockFlag		; Set SeekLockFlag
	
;  Set TargetPhase to last MeasPhase rounded to an integer

	MoveBytes0	MeasPhase,TargetPhase,3
	MoveBytes0	MeasPhase,aarg,3
	
	movlw	"K"						; Send target phase to PC just once
	call	SendPhaseToPC
	
	Serial_OutputText	  "@R 3.01$",Uadr22	; Output Firmware Version Number
	
Main102

	incf	NumMeas,f

	call 	PhaseDetector		; Generate voltage from MeasPhas

Main103
	call	Filter				; Low-pass the voltage from the phase detector
	
	MoveBytes0	Volts,aarg,3	; Next convert to integer and send to Rb source
	ChangePage	0,1
	StoreLiteral	0x7E0000,barg,3		; Add 0.5 before integer tranncation
	call	FPA24
	call	INT2416
	ChangePage	1,0
	MoveBytes0	aargb0,intVolts,2
	
	call DAC_OutputVolts			; Set new voltage	
	
Main201
	call	SendVoltsToPC		; Send voltage to PC

; If not in ready mode, skip remainder of code

	btfss	ReadyFlag
	goto	Main

	btfss	SeekLockFlag		; Skip if in FreeRun mode
	goto	Main

; The next section determines if the system has reached the lock state
; The lock variable is Spp = Spp - Sp^2/TotalNumSpp, where p = (MeasPhase - TargetPhase)

	MoveBytes0	MeasPhase,aarg,3		; Compute (MeasPhase - TargetPhase)^2
	MoveBytes0	TargetPhase,barg,3
	ChangePage	0,1
	call	FPS24
	MoveBytes0	aarg,Temp1,3				; Save difference for later Sp calculation
	MoveBytes0 	aarg,barg,3				; Now square difference
	call	FPM24
	MoveBytes0	Spp,barg,3				; Now add to Spp
	call	FPA24
	MoveBytes0	aarg,Spp,3

	MoveBytes0	Temp1,aarg,3			; Now calculate Sp = Sum(MeasPhase - TargetPhase)
	MoveBytes0	Sp,barg,3
	call	FPA24
	MoveBytes0	aarg,Sp,3
	ChangePage	1,0

	incf	NumSpp,f					; Keep trace of number of terms in Spp
	
	movlw	TotalNumSpp		; Have we accumulated all terms in Spp?
	subwf	NumSpp,w	
	btfss	_C
	goto	Main				; No, so wait for next phase sample

Main310
	MoveBytes0	Sp,aarg,3		; Complete calculation of Spp where
	MoveBytes0	Sp,barg,3		; Spp = Spp - Sp^2/NumSpp
	ChangePage 0,1				; First Square Sp
	call	FPM24
	MoveBytes0	fpTotalNumSpp,barg,3	; Next calculate Sp^2/NumSpp
	call	FPD24
	MoveBytes0	Spp,barg,3		; Now calculation Sp^2/SumSpp - Spp
	call	FPS24
	ChangePage 1,0
	btfsc	aargb0,7			; Result should be negative. If not set to 0
	goto	Main311				; Result is negative
	clrf	aarg				; Set Spp = 0
	clrf	aarg+1
	clrf	aarg+2
	goto	Main312
Main311
	bcf		aargb0,7			; Make result postive
Main312
	MoveBytes0	aarg,Spp,3

	call	SendSppToPC				; Send Spp to PC	
	MoveBytes0	Spp,aarg,3
	StoreLiteral	SppCut, barg,3			; Now see if Spp <= SppCut
	ChangePage	0,1
	call	TALEB24	
	ChangePage	1,0
	andlw	0xFF
	btfss	_Z
	goto	Main330							; Passed cut

Main325						; Failed cut
	clrf	NumPassingSppCut 
	bcf		LED2
	bcf		LockedFlag
	goto	Main350

Main330						; Passed Spp cut
	btfsc	Seg2Flag		; If not already in segment 2, enter it. 
	goto	Main335
	bsf		Seg2Flag
	Serial_OutputText	"@G     $",Uadr357	; Send notice to PC that segment 2 entered

Main335	
	incf	NumPassingSppCut,f	
	movlw	T_NumPassingSppCut			; Now check to see if enough passes
	subwf	NumPassingSppCut,w
	btfss	_C
	goto	Main350						; Number of passes still not enough
	
	; Enough passes--Rb now locked to GPS

	btfsc	LockedFlag				; Is LockedFlag already set?
	goto	Main350					; Yes, already set
	
	bsf		LockedFlag				; No. Set LockedFlag and turn on LED2				
	bsf		LED2
	Serial_OutputText	"@L     $",Uadr25	; Output message that lock just achievied. 
	
Main350
	clrf	Spp
	clrf	Spp+1
	clrf	Spp+2
	clrf	Sp
	clrf	Sp+1
	clrf	Sp+2
	clrf	NumSpp			; Initialize preparing for next Spp accumulation

	goto	Main



;*****************PhaseDetector*************************************
;
;  


PhaseDetector
	MoveBytes0	MeasPhase,aarg,3		; Compute deviation between phase and target phase
	MoveBytes0	TargetPhase,barg,3
	ChangePage	0,1
	call	FPS24
	ChangePage 1,0
	btfsc	Seg2Flag					; Gain depends on segment
	goto	PD001
	StoreLiteral	PD_Gain_Seg1,barg,3		; Gain for segment 1
	goto	PD002
PD001
	StoreLiteral	PD_Gain_Seg2,barg,3		; Gain for segment 2
PD002
	ChangePage	0,1
	call	FPM24							; Multiply phase error by gain
	StoreLiteral	PD_Intercept,barg,3		; Add detector intercept
	call	FPA24							; Result: 4095 = 5 V
  
; Make sure result is <= 4095
	StoreLiteral	0x8B0000,barg,3		; Load 4096
	call 	TALTB24
	andlw	0xFF
	ChangePage 1,0						; Non zero if true
	btfss	_Z
	goto	PD101						; True: voltage <= 4095	
	StoreLiteral	0x8A7FF0,Volts,3		; Set result = 4095, largest possible voltage
	return

PD101			; Now make sure voltage >= 0
	clrf	barg
	clrf	barg+1
	clrf	barg+2
	ChangePage 0,1
	call	TAGEB24
	ChangePage 1,0
	andlw	0xFF
	btfsc	_Z							; _Z = 0 if volt >= 0
	goto	PD102
	MoveBytes0	aarg,Volts,3
	return
PD102
	clrf	Volts						; Volts < 0, so set Volts = 0	
	clrf	Volts + 1
	clrf	Volts + 2
	return


;****************Digital Filter****************************************
;
;   Assume input voltage in Voltage. Result returned in voltage

Filter
	MoveBytes0	PrevInputVolts,aarg,3			; Calculate V + Vprevious
	MoveBytes0	Volts,PrevInputVolts,3
	MoveBytes0	Volts,barg,3
	ChangePage	0,1
	call		FPA24
	ChangePage	1,0
	btfsc		Seg2Flag					; Next multiply by A, for segement 1 or 2
	goto		Filter101
	StoreLiteral	Seg1Filter_A,barg,3
	goto		Filter102
Filter101
	StoreLiteral	Seg2Filter_A,barg,3
Filter102
	ChangePage	0,1
	call		FPM24
	ChangePage	1,0
	MoveBytes0	aarg,Volts,3
	MoveBytes0	PrevOutputVolts,aarg,3			; Now calculate B*Fprevious
	btfsc		Seg2Flag
	goto		Filter103
	StoreLiteral	Seg1Filter_B,barg,3
	goto		Filter104
Filter103
	StoreLiteral	Seg2Filter_B,barg,3
Filter104
	ChangePage	0,1
	call		FPM24
	MoveBytes0	Volts,barg,3				; Finally get A*(V + Vprevious) + B*Fprevious
	call		FPA24
	ChangePage	1,0
	MoveBytes0	aarg,Volts,3
	MoveBytes0	aarg,PrevOutputVolts,3
	return



	
;**********************************************************************
; This routine outputs MeasPhase to PC, in either a type "P" or "K" packet
; On entry, the type of packet is in the w regisgter and
; the phase to be sent is in aarg
; The format of the sent string is as follows:
;		Character 1:    "@"
;		Character 2:	"P" or "K"
;		Characters 3-7:	FLOOR(100*Phase + 1/2) in BCD
;		Character8:		"$" 

SendPhaseToPC
	movwf	Temp1
	ChangePage	0,1
	StoreLiteral	0x854800,barg,3			; Multiply MeasPhase by 100
	call	FPM24
	StoreLiteral	0x7E0000,barg,3				; Add 0.5
	call	FPA24
	call	INT2424							; Convert 100*MeasPhase to 24 bit integer
	ChangePage	1,0
	movf	Temp1,w
	call	SendPacket
	return



;***********************************************************************
; This routine sends the voltage to the PC
; The format of the sent string is as follows:
;		Character 1:    "@"
;		Character 2:	"V"
;		Characters 3-7:	intVoltage in BCD format
;		Character8:		"$" 

SendVoltsToPC

	MoveBytes0	intVolts,aargb0+1,2
	clrf		aargb0
	movlw	"V"
	call	SendPacket		
	return

 

;***********************************************************************
; This routine sends 30*Spp to the PC
; On entry, Spp in registers Spp, Spp+1, Spp++2
; The format of the sent string is as follows:
;		Character 1:    "@"
;		Character 2:	"S"
;		Characters 3-7:	30*Spp, maximum Spp = 3333.3
;		Character8:		"$" 

SendSppToPC
	
	MoveBytes0	Spp,aarg,3			; Multiply Spp by 30
	StoreLiteral	0x837000,barg,3	
	ChangePage	0,1
	call	FPM24				
	StoreLiteral	0x7E0000,barg,3	; Add 0.5 
	call	FPA24
	call	INT2424			; Convert to Integer
	ChangePage 1,0
	movlw	"S"
	call	SendPacket
	return

;**********************************************************************
; This routine outputs a 8-byte packet via the serial port. 
; The first character is always "@", and the second character is in w at entry. 
; A five digit number follows the second character. This number can be between 
; 0 and 99,999. Any number larger is truncated to 99,999. 
; The packet always ends with "$"

SendPacket
	movwf	Temp1						; Save second character of packet
	movlw	"@"							; Send phase as 5 digit integer
	call	Serial_TxChar
	
	StoreLiteral	0x01869F,bargb0,3		; Store 99,999
	call	INT24AGTB					; Is aargb0 > 99,999?
	andlw	0xFF
	btfsc	_Z
	goto	SP01
	MoveBytes0	bargb0,aargb0,3			; Yes, so load 99,999, most max value
SP01
	movf	Temp1,w						; Now output second character. 
	call	Serial_TxChar
	
	call 	Int24ToASCII
	
	movf	ASCII_Num+3,w				; Output five digit number
	call	Serial_TxChar
	movf	ASCII_Num+4,w
	call	Serial_TxChar
	movf	ASCII_Num+5,w
	call	Serial_TxChar
	movf	ASCII_Num+6,w
	call	Serial_TxChar
	movf	ASCII_Num+7,w
	call	Serial_TxChar
	movlw	"$"
	call	Serial_TxChar
	return



;****************************************************************************
; INT24AGTB
;    This routine determines whether an 24-bit integer in aargb0 is > than a
;    24-bit integer in bargb0

INT24AGTB
	movf	aargb0,w
	subwf	bargb0,w
	btfsc	_Z
	goto	IAB01			; Most significant bytes are equal, so check next bytes
	btfsc	_C				; C = 0 means A > B
	retlw	0
	retlw	1   

IAB01
	movf	aargb0+1,w
	subwf	bargb0+1,w
	btfsc	_Z
	goto	IAB02			; Middle bytes are equal, so check next low bytes
	btfsc	_C				; C = 0 means A > B
	retlw	0
	retlw	1

IAB02
	movf	aargb0+2,w
	subwf	bargb0+2,w
	btfsc	_C				; C = 0 means A > B
	retlw	0
	retlw	1




;*******************************************************************************
;   This routine outputs the character in w to the serial port

Serial_TxChar			; Output character in w to serial port
 if Simulate > 0
	return
 endif
	btfss	PIR1,TXIF		; Is serial port transmitter ready to receive character?
	goto	$-1
	movwf	TXREG			; Yes, write character to transmitt register
	return	


	
;****************************************************************************
; This subroutine outputs the contents of intVolts, intVolts+1 to the 
; digitial-to-analog converter which sets the control voltage for the Rb oscillator
; On entry the voltage to be output is in intVolts. At exit, the voltage is still
; in intVolts

DAC_OutputVolts				; Output contents of intVolts to external DAC
	movlw	4
	movwf	DAC_Counter
	rlf		intVolts+1,f
	rlf		intVolts,f
	decfsz	DAC_Counter,f
	goto	$-3

	movlw	12				; 12 bits to output
	movwf	DAC_Counter
	bcf		DAC_Clock		; Initialize control lines to DAC
	bsf		DAC_Load

DOA11
	rlf		intVolts+1,f		; Rotate next bit into c
	rlf		intVolts,f
	btfss	_C
	goto	$+3
	bsf		DAC_Data
	goto	$+2
	bcf		DAC_Data
	bsf		DAC_Clock
	bcf		DAC_Clock
	decfsz	DAC_Counter,f	
	goto	DOA11
	
	bcf		DAC_Load		; Now cycle load line to DAC to load shift register
	bsf		DAC_Load		; into DAC
	rlf		intVolts+1,f
	rlf		intVolts,f
	return


	


;*******************************************************************
;*************************************************************
; This routine takes a 16 bit integer (always positive) located in
; aargb0 (most significant 8 bits) and aargb1 and produces a five character
; ASCII string that expresses the number in decimal characters. The result is placed
; in ASCII_Num, ASCII_Num+1, ... , ASCII_Num+4. 
;

Int16ToASCII
	bcf	status,rp0 		; Data in bank 0
	movlw	0x27			; Load integer 10000 into bargb0, 1
	movwf	bargb0
	movlw	0x10
	movwf	bargb1
	call	DivideDown16Bits		; Subtract 10,000 repeatedly until
	movwf	ASCII_Num		; total less than 10,000. # of subtractions
					; is 10000's digit. STore in ASCII_num

	movlw	0x03			; load 1000
	movwf	bargb0
	movlw	0xe8
	movwf	bargb1
	call	DivideDown16Bits
	movwf	ASCII_Num + 1

	movlw	0x00			; load 100
	movwf	bargb0
	movlw	0x64
	movwf	bargb1
	call	DivideDown16Bits
	movwf	ASCII_Num + 2

	movlw	0x00			; load 10
	movwf	bargb0
	movlw	0x0a
	movwf	bargb1
	call	DivideDown16Bits
	movwf	ASCII_Num + 3

	movf	aargb1,w		; Remainder in aargb1 is units digit
	addlw	0x30			; Add 0x30 to convert to ASCII
	movwf	ASCII_Num + 4
	return

;*******************************************************************
; This routine repeatedly subtracts bargb0:b1 from aargb0:b1 until result is no longer positive

DivideDown16Bits
	clrf	Counter2			; Counts # of succesful subtractions
r11	call	Int16Sub		; subtract bargb0:b1 from aargb0:b1
	andlw	0xff			; Check is result < 0
	btfss	status,z
	goto	d1			; Yes, result was negative
	incf	Counter2,f		; No, result still positive
	goto	r11			; Subtract again
d1	movf	Counter2,w		; Last subtraction resulted in - result
					; aargb0:b1 restored to last + value
	addlw	0x30			; Add 0x30 to count to convert to ASCII
	return

;*******************************************************************
; This routine does a 16 bit integer subtraction if the result is positive

Int16Sub
	MoveBytes0 aargb0,Temp1,2	; Save aargb0:b1 in case result of subtraction is -
	movf	bargb1,w		; Subtract least significant 8 bits
	subwf	aargb1,f		; Result goes in aargb1
	btfsc	status,c		; Was a borrow necessary?
	goto	r1			; No, jump to r1
	movlw	1			; Yes. Subtract 1 from aargb0
	subwf	aargb0,f		; Place result back in aargb0
	btfsc	status,c		; If borrow necessary, overall result negative.
	goto 	r1			; No borrow
	MoveBytes0 Temp1,aargb0,2	; Borrow necessary-result negative-restore aargb0:b1
	retlw	0xff			; Return with w=0xff: result was negative
r1	movf	bargb0,w		; Subtract most significant 8 bits
	subwf	aargb0,f		; Place result back in aargb0
	btfsc	status,c		; Was borrow necessary?
	retlw	0x00			; No: result was positive, return
	MoveBytes0 Temp1,aargb0,2	; Yes, restore aargb0:b1 to previous value
	retlw	0xff			; REturn with w=0xff; result was negative



;*************************************************************
; This routine takes a 24 bit integer (always positive) located in
; aargb0 (most significant 8 bits), aargb1, and aargb2 and produces a eight character
; ASCII string that expresses the number in decimal characters. The result is placed
; in ASCII_Num, ASCII_Num+1, ... , ASCII_Num+7. 
;

Int24ToASCII
	StoreLiteral	0x989680,bargb0,3		; Load 1E7
	call	DivideDown24Bits		; Subtract 1E7 repeatedly until
	movwf	ASCII_Num		; total less than 1E7. # of subtractions
					; is seventh (most significant) digit. Store in ASCII_num

	StoreLiteral	0x0F4240,bargb0,3		; Load 1E6
	call	DivideDown24Bits		
	movwf	ASCII_Num+1		
					
	StoreLiteral	0x0186A0,bargb0,3		; Load 1E5
	call	DivideDown24Bits		
	movwf	ASCII_Num+2		
					
	clrf	bargb0			
	StoreLiteral	0x2710,bargb0+1,2		; Load 1E4
	call	DivideDown24Bits		
	movwf	ASCII_Num+3		

	StoreLiteral	0x03E8,bargb0+1,2		; Load 1E3
	call	DivideDown24Bits		
	movwf	ASCII_Num+4		

	clrf	bargb0+1
	StoreLiteral	0x64,bargb0+2,1		; Load 1E2
	call	DivideDown24Bits		
	movwf	ASCII_Num+5		

	StoreLiteral	0x0A,bargb0+2,1		; Load 1E1
	call	DivideDown24Bits		
	movwf	ASCII_Num+6		

	movf	aargb0+2,w		; Remainder in aargb1 is units digit
	addlw	0x30			; Add 0x30 to convert to ASCII
	movwf	ASCII_Num + 7
	return

;*******************************************************************
; This routine repeatedly subtracts bargb0:b2 from aargb0:b2 until result is no longer positive
; The remainder is returned in aargb0:b2

DivideDown24Bits
	clrf	Counter2			; Counts # of succesful subtractions
DD24B01
	call	Int24Sub		; subtract bargb0:b2 from aargb0:b2
	andlw	0xff			; Check is result < 0
	btfss	status,z
	goto	DD24B02			; Yes, result was negative
	incf	Counter2,f		; No, result still positive
	goto	DD24B01			; Subtract again
DD24B02
	movf	Counter2,w		; Last subtraction resulted in - result
							; aargb0:b2 restored to last + value
	addlw	0x30			; Add 0x30 to count to convert to ASCII
	return

;*******************************************************************
; This routine does a 16 bit integer subtraction if the result is positive

Int24Sub
	MoveBytes0 aargb0,Temp1,3	; Save aargb0:b2 in case result of subtraction is -
	movf	bargb0+2,w			; Subtract LSB of bargb0 from aargb0
	subwf	aargb0+2,f
	btfsc	_C					; Was borrow necessary?
	goto	I24S01				; No
	movlw	1					; Yes, so borrow 1 from next most significant byte
	subwf	aargb0+1,f
	btfsc	_C					; Was borrow again necessary?
	goto	I24S01				; No
	subwf	aargb0,f			; Yes, so subtract 1 from most significant byte
	btfss	_C					; Was borrow again necessary?
	goto	I24S10				; Yes, so overall result is negative. Go to negative handler
I24S01					
	movf	bargb0+1,w			; Subtract middle byte of bargb0 from aargb0
	subwf	aargb0+1,f
	btfsc	_C					; Was borrow necessary?
	goto	I24S02				; No
	movlw	1					; Yes, so borrow 1 from next most significant byte
	subwf	aargb0,f
	btfss	_C					; Was borrow again necessary?
	goto	I24S10			; Yes, so overall result is negative. Go to negative handler
I24S02
	movf	bargb0,w			; Subtract left byte of bargb0 from aargb0
	subwf	aargb0,f
	btfsc	_C					; Was borrow necessary?
	retlw	0x0					; Result of subtraction positive so set w = 0 with
								; remainder is aargb0:b2
I24S10
	MoveBytes0	Temp1,aargb0,3	; Result of subtraction was negative so restore positive
	retlw	0xFF				; remainder to aargb0:b2 and set w != 0
	




;*************OutputText************************************************
; This routine is used in conjuction with the macro Serial_OutputText

OutputText
	call	OutText01
	andlw	0xFF
	btfsc	_Z
	return
	call	Serial_TxChar
	incf	TextAdr,f
	btfsc	_Z
	incf	PCLATH,f
	goto	OutputText
OutText01
	movf	TextAdr,w
	movwf	PCL




;******************************************************************************
; This routine intializes the control voltage as well as the voltage variables
; in the subroutine Filter. 
; On entry, the 16-bit integer voltage is in intVolts


InitializeVoltage
	Call	DAC_OutputVolts
	MoveBytes0	intVolts,aargb0,2
	ChangePage 0,1
	call	FLO1624					; Convert InitalVolts to floating point
	ChangePage 1,0
	MoveBytes0	aarg,PrevInputVolts,3	; Initial variables used by digital filter
	MoveBytes0	aarg,PrevOutputVolts,3
	return



;*********PIC 24 bit floating point library***************************************
; Place in page 1

	org		0x800

;**********************************************************************************************
;**************************************************************************

;       PIC16 24 BIT FLOATING POINT LIBRARY
;
;       Place library at beginning of Page 1 of memory



;**********************************************************************************************

;       Integer to float conversion
;       Input:  16 bit 2's complement integer right justified in AARGB0, AARGB1
;       Use:    CALL    FLO1624 or      CALL    FLO24
;       Output: 24 bit floating point number in AEXP, AARGB0, AARGB1
;       Result: AARG  <--  FLOAT( AARG )
;       Max Timing:     11+72 = 83 clks         SAT = 0
;                       11+77 = 88 clks         SAT = 1

;       Min Timing:     7+14 = 21 clks                  AARG = 0
;                       7+18 = 25 clks

;       PM: 11+26 = 37                                  DM: 6

;----------------------------------------------------------------------------------------------

FLO1624

FLO24           MOVLW           D'15'+EXPBIAS		; initialize exponent and add bias
                MOVWF           EXP
                 MOVF            AARGB0,W
                MOVWF           SIGN
                BTFSS           AARGB0,MSB		; test sign
                GOTO            NRM2424
                COMF            AARGB1,F		; if < 0, negate and set MSB in SIGN
                COMF            AARGB0,F
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F

;**********************************************************************************************

;       Normalization routine

;       Input:  24 bit unnormalized floating point number in AEXP, AARGB0, AARGB1,
;               with sign in SIGN,MSB and other bits zero.

;       Use:    CALL    NRM2424 or      CALL    NRM24

;       Output: 24 bit normalized floating point number in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  NORMALIZE( AARG )

;       Max Timing:     10+6+7*7+7 = 72 clks            SAT = 0
;                       10+6+7*7+1+11 = 77 clks SAT = 1

;       Min Timing:     14 clks                         AARG = 0
;                       5+9+4 = 18 clks

;       PM: 26                                          DM: 6

;----------------------------------------------------------------------------------------------

NRM2424
NRM24
		CLRF            TEMP			; clear exponent decrement
                MOVF            AARGB0,W		; test if highbyte=0
                BTFSS           _Z
                GOTO            NORM2424
                MOVF            AARGB1,W		; if so, shift 8 bits by move
                MOVWF           AARGB0
                BTFSC           _Z			; if highbyte=0, result=0
                GOTO            RES024
                CLRF            AARGB1
                BSF             TEMP,3

NORM2424        MOVF            TEMP,W
                SUBWF           EXP,F
                BTFSS           _Z
                BTFSS           _C
                GOTO            SETFUN24

                BCF             _C			; clear carry bit

NORM2424A       BTFSC           AARGB0,MSB		; if MSB=1, normalization done
                GOTO            FIXSIGN24
                RLF             AARGB1,F		; otherwise, shift left and 
                RLF             AARGB0,F		; decrement EXP
                DECFSZ          EXP,F
                GOTO            NORM2424A

                GOTO            SETFUN24                ; underflow if EXP=0

FIXSIGN24       BTFSS           SIGN,MSB
                BCF             AARGB0,MSB		; clear explicit MSB if positive
                RETLW           0

RES024          CLRF            AARGB0			; result equals zero
                CLRF            AARGB1
		CLRF		AARGB2			; clear extended byte
                CLRF            EXP
                RETLW           0


;**********************************************************************************************
;**********************************************************************************************

;       Integer to float conversion

;       Input:  24 bit 2's complement integer right justified in AARGB0, AARGB1, AARGB2

;       Use:    CALL    FLO2424

;       Output: 24 bit floating point number in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  FLOAT( AARG )

;       Max Timing:     14+94 = 108 clks                RND = 0
;                       14+103 = 117 clks               RND = 1, SAT = 0
;                       14+109 = 123 clks               RND = 1, SAT = 1

;       Min Timing:     6+28 = 34 clks                  AARG = 0
;                       6+22 = 28 clks

;       PM: 14+51 = 65                                  DM: 7

;----------------------------------------------------------------------------------------------

FLO2424         MOVLW           D'23'+EXPBIAS		; initialize exponent and add bias
                MOVWF           EXP
                CLRF            SIGN
                BTFSS           AARGB0,MSB		; test sign
                GOTO            NRM3224
                COMF            AARGB2,F		; if < 0, negate and set MSB in SIGN
                COMF            AARGB1,F
                COMF            AARGB0,F
                INCF            AARGB2,F
                BTFSC           _Z
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F
                BSF             SIGN,MSB

;**********************************************************************************************

;       Normalization routine

;       Input:  32 bit unnormalized floating point number in AEXP, AARGB0, AARGB1,
;               AARGB2, with sign in SIGN,MSB

;       Use:    CALL    NRM3224

;       Output: 24 bit normalized floating point number in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  NORMALIZE( AARG )

;       Max Timing:     21+6+7*8+7+4 = 94 clks  RND = 0
;                       21+6+7*8+20+4 = 103 clks        RND = 1, SAT = 0
;                       21+6+7*8+19+11 = 109 clks       RND = 1, SAT = 1

;       Min Timing:     22+6 = 28 clks                  AARG = 0
;                       5+9+4+4 = 22 clks

;       PM: 51                                          DM: 7

;----------------------------------------------------------------------------------------------

NRM3224         CLRF            TEMP                    ; clear exponent decrement
                MOVF            AARGB0,W		; test if highbyte=0
                BTFSS           _Z
                GOTO            NORM3224
                MOVF            AARGB1,W		; if so, shift 8 bits by move
                MOVWF           AARGB0
                MOVF            AARGB2,W
                MOVWF           AARGB1
                CLRF            AARGB2
                BSF             TEMP,3                  ; increase decrement by 8

                MOVF            AARGB0,W		; test if highbyte=0
                BTFSS           _Z
                GOTO            NORM3224
                MOVF            AARGB1,W		; if so, shift 8 bits by move
                MOVWF           AARGB0
                CLRF            AARGB1
                BCF             TEMP,3                  ; increase decrement by 8
                BSF             TEMP,4

                MOVF            AARGB0,W		; if highbyte=0, result=0
                BTFSC           _Z
                GOTO            RES024

NORM3224        MOVF            TEMP,W
                SUBWF           EXP,F
                BTFSS           _Z
                BTFSS           _C
                GOTO            SETFUN24

                BCF             _C                      ; clear carry bit

NORM3224A       BTFSC           AARGB0,MSB		; if MSB=1, normalization done
                GOTO            NRMRND3224
                RLF             AARGB2,F		; otherwise, shift left and
                RLF             AARGB1,F		; decrement EXP
                RLF             AARGB0,F
                DECFSZ          EXP,F
                GOTO            NORM3224A
                GOTO            SETFUN24                ; underflow if EXP=0

NRMRND3224      BTFSC           FPFLAGS,RND
                BTFSS           AARGB1,LSB
                GOTO            FIXSIGN24
		BTFSS		AARGB2,MSB		; round if next bit is set
                GOTO            FIXSIGN24
		INCF		AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F

                BTFSS           _Z                     ; has rounding caused carryout?
                GOTO            FIXSIGN24
                RRF             AARGB0,F		; if so, right shift
                RRF             AARGB1,F
                INCF            EXP,F
                BTFSC           _Z                     ; check for overflow
                GOTO            SETFOV24
                GOTO            FIXSIGN24

;**********************************************************************************************
;**********************************************************************************************

;       Float to integer conversion

;       Input:  24 bit floating point number in AEXP, AARGB0, AARGB1

;       Use:    CALL    INT2416         or      CALL    INT24

;       Output: 16 bit 2's complement integer right justified in AARGB0, AARGB1

;       Result: AARG  <--  INT( AARG )

;       Max Timing:     29+6*6+5+13 = 83 clks           RND = 0
;                       29+6*6+5+19 = 89 clks           RND = 1, SAT = 0
;                       29+6*6+5+22 = 92 clks           RND = 1, SAT = 1

;       Min Timing:     18+5+7 = 30 clks

;       PM: 63                                  DM: 6

;----------------------------------------------------------------------------------------------

INT2416
INT24
		MOVF		EXP,W			; test for zero argument
		BTFSC		_Z
		RETLW		0x00

		MOVF            AARGB0,W		; save sign in SIGN
                MOVWF           SIGN
                BSF             AARGB0,MSB		; make MSB explicit

                MOVLW           EXPBIAS+D'15'		; remove bias from EXP
                SUBWF           EXP,F
		BTFSS		EXP,MSB
                GOTO            SETIOV16
		COMF		EXP,F
		INCF		EXP,F

                MOVLW           8                       ; do byte shift if EXP >= 8
                SUBWF           EXP,W
                BTFSS           _C
                GOTO            TSHIFT2416
                MOVWF           EXP
                RLF             AARGB1,F		; rotate next bit for rounding
                MOVF            AARGB0,W
                MOVWF           AARGB1
                CLRF            AARGB0

                MOVLW           8                       ; do byte shift if EXP >= 8
                SUBWF           EXP,W
                BTFSS           _C
                GOTO            TSHIFT2416
                MOVWF           EXP
                RLF             AARGB1,F		; rotate next bit for rounding
                CLRF            AARGB1
		MOVF		EXP,W
		BTFSS		_Z
		BCF		_C
		GOTO		SHIFT2416OK

TSHIFT2416      MOVF            EXP,W                   ; shift completed if EXP = 0
                BTFSC           _Z
                GOTO            SHIFT2416OK

SHIFT2416       BCF             _C
                RRF             AARGB0,F		; right shift by EXP
                RRF             AARGB1,F
                DECFSZ          EXP,F
                GOTO            SHIFT2416

SHIFT2416OK     BTFSC           FPFLAGS,RND
                BTFSS           AARGB1,LSB
                GOTO            INT2416OK
		BTFSS		_C			; round if next bit is set
                GOTO            INT2416OK
		INCF		AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F

                BTFSC           AARGB0,MSB		; test for overflow
                GOTO            SETIOV16

INT2416OK       BTFSS           SIGN,MSB                ; if sign bit set, negate
                RETLW           0
                COMF            AARGB1,F
                COMF            AARGB0,F
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F
                RETLW           0

SETIOV16        BSF             FPFLAGS,IOV             ; set integer overflow flag
                BTFSS           FPFLAGS,SAT             ; test for saturation
                RETLW           0xFF                    ; return error code in WREG

                CLRF            AARGB0			; saturate to largest two's
                BTFSS           SIGN,MSB                ; complement 16 bit integer
                MOVLW           0xFF
                MOVWF           AARGB0			; SIGN = 0, 0x 7F FF
                MOVWF           AARGB1			; SIGN = 1, 0x 80 00
                RLF             SIGN,F
                RRF             AARGB0,F
                RETLW           0xFF                    ; return error code in WREG



;**********************************************************************************************
;**********************************************************************************************

;       Float to integer conversion

;       Input:  24 bit floating point number in AEXP, AARGB0, AARGB1

;       Use:    CALL    INT2424

;       Output: 24 bit 2's complement integer right justified in AARGB0, AARGB1, AARGB2

;       Result: AARG  <--  INT( AARG )

;       Max Timing:     41+6*7+6+16 = 105 clks		RND = 0
;                       41+6*7+6+24 = 113 clks		RND = 1, SAT = 0
;                       41+6*7+6+26 = 115 clks  	RND = 1, SAT = 1

;       Min Timing:     5 clks

;       PM: 82                                          DM: 6

;----------------------------------------------------------------------------------------------

INT2424
                CLRF            AARGB2
		MOVF		EXP,W			; test for zero argument
		BTFSC		_Z
		RETLW		0x00

		MOVF            AARGB0,W		; save sign in SIGN
                MOVWF           SIGN
                BSF             AARGB0,MSB		; make MSB explicit

                MOVLW           EXPBIAS+D'23'		; remove bias from EXP
                SUBWF           EXP,F
                BTFSS           EXP,MSB
                GOTO            SETIOV24        
		COMF		EXP,F
		INCF		EXP,F

                MOVLW           8                       ; do byte shift if EXP >= 8
                SUBWF           EXP,W
                BTFSS           _C
                GOTO            TSHIFT2424
                MOVWF           EXP
                RLF             AARGB2,F		; rotate next bit for rounding
                MOVF            AARGB1,W
                MOVWF           AARGB2
                MOVF            AARGB0,W
                MOVWF           AARGB1
                CLRF            AARGB0

                MOVLW           8                       ; do another byte shift if EXP >= 8
                SUBWF           EXP,W
                BTFSS           _C
                GOTO            TSHIFT2424
                MOVWF           EXP
                RLF             AARGB2,F		; rotate next bit for rounding
                MOVF            AARGB1,W
                MOVWF           AARGB2
                CLRF            AARGB1

                MOVLW           8                       ; do another byte shift if EXP >= 8
                SUBWF           EXP,W
                BTFSS           _C
                GOTO            TSHIFT2424
                MOVWF           EXP
                RLF             AARGB2,F		; rotate next bit for rounding
                CLRF            AARGB2
		MOVF		EXP,W
		BTFSS		_Z
		BCF		_C
		GOTO		SHIFT2424OK

TSHIFT2424      MOVF            EXP,W                   ; shift completed if EXP = 0
                BTFSC           _Z
                GOTO            SHIFT2424OK

SHIFT2424       BCF             _C
                RRF             AARGB0,F		; right shift by EXP
                RRF             AARGB1,F
                RRF             AARGB2,F
                DECFSZ          EXP,F
                GOTO            SHIFT2424

SHIFT2424OK     BTFSC           FPFLAGS,RND
                BTFSS           AARGB2,LSB
                GOTO            INT2424OK
                BTFSS           _C
                GOTO            INT2424OK
                INCF            AARGB2,F
                BTFSC           _Z
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F
                BTFSC           AARGB0,MSB		; test for overflow
                GOTO            SETIOV24

INT2424OK       BTFSS           SIGN,MSB                ; if sign bit set, negate               
                RETLW           0
                COMF            AARGB0,F
                COMF            AARGB1,F
                COMF            AARGB2,F
                INCF            AARGB2,F
                BTFSC           _Z
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F
                RETLW           0

IRES024         CLRF            AARGB0			; integer result equals zero
                CLRF            AARGB1
                CLRF            AARGB2
                RETLW           0

SETIOV24        BSF             FPFLAGS,IOV             ; set integer overflow flag
                BTFSS           FPFLAGS,SAT             ; test for saturation
                RETLW           0xFF                    ; return error code in WREG

                CLRF            AARGB0			; saturate to largest two's
                BTFSS           SIGN,MSB                ; complement 24 bit integer
                MOVLW           0xFF
                MOVWF           AARGB0			; SIGN = 0, 0x 7F FF FF
                MOVWF           AARGB1			; SIGN = 1, 0x 80 00 00
                MOVWF           AARGB2
                RLF             SIGN,F
                RRF             AARGB0,F
                RETLW           0xFF                    ; return error code in WREG



;**********************************************************************************************

;       Floating Point Multiply

;       Input:  24 bit floating point number in AEXP, AARGB0, AARGB1
;               24 bit floating point number in BEXP, BARGB0, BARGB1

;       Use:    CALL    FPM24

;       Output: 24 bit floating point product in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  AARG * BARG

;       Max Timing:     25+15*16+15+18 = 298 clks       RND = 0
;                       25+15*16+15+29 = 309 clks       RND = 1, SAT = 0
;                       25+15*16+15+33 = 313 clks       RND = 1, SAT = 1

;       Min Timing:     6+5 = 11 clks                   AARG * BARG = 0
;                       24+15*11+14+15 = 218 clks

;       PM: 80                                  DM: 11

;----------------------------------------------------------------------------------------------

FPM24           MOVF            AEXP,W                  ; test for zero arguments
                BTFSS           _Z
                MOVF            BEXP,W
                BTFSC           _Z
                GOTO            RES024

M24BNE0         MOVF            AARGB0,W
                XORWF           BARGB0,W
                MOVWF           SIGN                    ; save sign in SIGN

                MOVF            BEXP,W
                ADDWF           EXP,F
                MOVLW           EXPBIAS-1
                BTFSS           _C
                GOTO            MTUN24

                SUBWF           EXP,F
                BTFSC           _C
                GOTO            SETFOV24                ; set multiply overflow flag
                GOTO            MOK24

MTUN24          SUBWF           EXP,F
                BTFSS           _C
                GOTO            SETFUN24

MOK24
                MOVF            AARGB0,W
                MOVWF           AARGB2			; move result to AARG
                MOVF            AARGB1,W
                MOVWF           AARGB3
		BSF             AARGB2,MSB		; make argument MSB's explicit
                BSF             BARGB0,MSB
                BCF             _C
                CLRF            AARGB0			; clear initial partial product
                CLRF            AARGB1
                MOVLW           D'16'
                MOVWF           TEMP                    ; initialize counter

MLOOP24         BTFSS           AARGB3,LSB		; test next bit
                GOTO            MNOADD24

MADD24          MOVF            BARGB1,W
                ADDWF           AARGB1,F
                MOVF            BARGB0,W
                BTFSC           _C
                INCFSZ          BARGB0,W
                ADDWF           AARGB0,F

MNOADD24        RRF             AARGB0,F
                RRF             AARGB1,F
                RRF             AARGB2,F
                RRF             AARGB3,F
                BCF             _C
                DECFSZ          TEMP,F
                GOTO            MLOOP24

                BTFSC           AARGB0,MSB		; check for postnormalization
                GOTO            MROUND24
                RLF             AARGB2,F
                RLF             AARGB1,F
                RLF             AARGB0,F
                DECF            EXP,F

MROUND24        BTFSC           FPFLAGS,RND
                BTFSS           AARGB1,LSB
                GOTO            MUL24OK
		BTFSS		AARGB2,MSB		; round if next bit is set
                GOTO            MUL24OK
		INCF		AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F

                BTFSS           _Z                      ; has rounding caused carryout?
                GOTO            MUL24OK
                RRF             AARGB0,F		; if so, right shift
                RRF             AARGB1,F
                INCF            EXP,F
                BTFSC           _Z                      ; check for overflow
                GOTO            SETFOV24

MUL24OK         BTFSS           SIGN,MSB
                BCF             AARGB0,MSB		; clear explicit MSB if positive

                RETLW           0

SETFOV24        BSF             FPFLAGS,FOV             ; set floating point underflag
                BTFSS           FPFLAGS,SAT             ; test for saturation
                RETLW           0xFF                    ; return error code in WREG

                MOVLW           0xFF
                MOVWF           AEXP                    ; saturate to largest floating
                MOVWF           AARGB0			; point number = 0x FF 7F FF
                MOVWF           AARGB1			; modulo the appropriate sign bit
                RLF             SIGN,F
                RRF             AARGB0,F
                RETLW           0xFF                    ; return error code in WREG

;**********************************************************************************************
;**********************************************************************************************

;       Floating Point Divide

;       Input:  24 bit floating point dividend in AEXP, AARGB0, AARGB1
;               24 bit floating point divisor in BEXP, BARGB0, BARGB1

;       Use:    CALL    FPD24

;       Output: 24 bit floating point quotient in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  AARG / BARG

;       Max Timing:     32+13+15*26+25+12 = 472 clks    RND = 0
;                       32+13+15*26+25+34 = 494 clks    RND = 1, SAT = 0
;                       32+13+15*26+25+38 = 498 clks    RND = 1, SAT = 1

;       Min Timing:     7+5 = 12 clks

;       PM: 120                                 DM: 11

;----------------------------------------------------------------------------------------------

FPD24           MOVF            BEXP,W                  ; test for divide by zero
                BTFSC           _Z
                GOTO            SETFDZ24

                MOVF            AEXP,W
                BTFSC           _Z
                GOTO            RES024

D24BNE0         MOVF            AARGB0,W
                XORWF           BARGB0,W
                MOVWF           SIGN                    ; save sign in SIGN
                BSF             AARGB0,MSB		; make argument MSB's explicit
                BSF             BARGB0,MSB

TALIGN24        CLRF            TEMP                    ; clear align increment
                MOVF            AARGB0,W
                MOVWF           AARGB2			; test for alignment
                MOVF            AARGB1,W
                MOVWF           AARGB3

                MOVF            BARGB1,W
                SUBWF           AARGB3, f
                MOVF            BARGB0,W
                BTFSS           _C
                INCFSZ          BARGB0,W
                SUBWF           AARGB2, f

                CLRF            AARGB2
                CLRF            AARGB3

                BTFSS           _C
                GOTO            DALIGN24OK

                BCF             _C                      ; align if necessary
                RRF             AARGB0,F
                RRF             AARGB1,F
                RRF             AARGB2,F
                MOVLW           0x01
                MOVWF           TEMP                    ; save align increment          

DALIGN24OK      MOVF            BEXP,W                  ; compare AEXP and BEXP
                SUBWF           EXP,F
                BTFSS           _C
                GOTO            ALTB24
        
AGEB24          MOVLW           EXPBIAS-1
                ADDWF           TEMP,W
                ADDWF           EXP,F
                BTFSC           _C
                GOTO            SETFOV24
                GOTO            DARGOK24                ; set overflow flag

ALTB24          MOVLW           EXPBIAS-1
                ADDWF           TEMP,W
                ADDWF           EXP,F
                BTFSS           _C
                GOTO            SETFUN24                ; set underflow flag

DARGOK24        MOVLW           D'16'			; initialize counter
                MOVWF           TEMPB1

DLOOP24         RLF             AARGB3,F		; left shift
                RLF             AARGB2,F
                RLF             AARGB1,F
                RLF             AARGB0,F
                RLF             TEMP,F

                MOVF            BARGB1,W		; subtract
                SUBWF           AARGB1,F
                MOVF            BARGB0,W
                BTFSS           _C
                INCFSZ          BARGB0,W
                SUBWF           AARGB0,F

                RLF             BARGB0,W
                IORWF           TEMP,F
                
                BTFSS           TEMP,LSB                ; test for restore
                GOTO            DREST24

                BSF             AARGB3,LSB
                GOTO            DOK24

DREST24         MOVF            BARGB1,W		; restore if necessary
                ADDWF           AARGB1,F
                MOVF            BARGB0,W
                BTFSC           _C
                INCF            BARGB0,W
                ADDWF           AARGB0,F
                BCF             AARGB3,LSB

DOK24           DECFSZ          TEMPB1,F
                GOTO            DLOOP24

DROUND24        BTFSC           FPFLAGS,RND
                BTFSS           AARGB3,LSB
                GOTO            DIV24OK
                BCF             _C
                RLF             AARGB1,F		; compute next significant bit
                RLF             AARGB0,F		; for rounding
                RLF             TEMP,F

                MOVF            BARGB1,W		; subtract
                SUBWF           AARGB1,F
                MOVF            BARGB0,W
                BTFSS           _C
                INCFSZ          BARGB0,W
                SUBWF           AARGB0,F

                RLF             BARGB0,W
                IORWF           TEMP,W
                ANDLW           0x01            

                ADDWF           AARGB3,F
                BTFSC           _C
                INCF            AARGB2,F

                BTFSS           _Z                      ; test if rounding caused carryout
                GOTO            DIV24OK
                RRF             AARGB2,F
                RRF             AARGB3,F
                INCF            EXP,F
                BTFSC           _Z                      ; test for overflow
                GOTO            SETFOV24

DIV24OK         BTFSS           SIGN,MSB
                BCF             AARGB2,MSB		; clear explicit MSB if positive

                MOVF            AARGB2,W
                MOVWF           AARGB0			; move result to AARG
                MOVF            AARGB3,W
                MOVWF           AARGB1

                RETLW           0

SETFUN24        BSF             FPFLAGS,FUN             ; set floating point underflag
                BTFSS           FPFLAGS,SAT             ; test for saturation
                RETLW           0xFF                    ; return error code in WREG

                MOVLW           0x01                    ; saturate to smallest floating
                MOVWF           AEXP                    ; point number = 0x 01 00 00
                CLRF            AARGB0			; modulo the appropriate sign bit
                CLRF            AARGB1
                RLF             SIGN,F
                RRF             AARGB0,F
                RETLW           0xFF                    ; return error code in WREG

SETFDZ24        BSF             FPFLAGS,FDZ             ; set divide by zero flag
                RETLW           0xFF

;**********************************************************************************************
;**********************************************************************************************

;       Floating Point Subtract

;       Input:  24 bit floating point number in AEXP, AARGB0, AARGB1
;               24 bit floating point number in BEXP, BARGB0, BARGB1

;       Use:    CALL FPS24

;       Output: 24 bit floating point sum in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  AARG - BARG

;       Max Timing:     2+197 = 199 clks                RND = 0
;                       2+208 = 210 clks                RND = 1, SAT = 0
;                       2+213 = 215 clks                RND = 1, SAT = 1

;       Min Timing:     2+12 = 14 clks

;       PM: 2+112 = 114                         DM: 11

;----------------------------------------------------------------------------------------------

FPS24           MOVLW           0x80
                XORWF           BARGB0,F

;**********************************************************************************************

;       Floating Point Add

;       Input:  24 bit floating point number in AEXP, AARGB0, AARGB1
;               24 bit floating point number in BEXP, BARGB0, BARGB1

;       Use:    CALL FPA24

;       Output: 24 bit floating point sum in AEXP, AARGB0, AARGB1

;       Result: AARG  <--  AARG - BARG

;       Max Timing:     25+28+6*6+5+31+72 = 197 clks            RND = 0
;                       25+28+6*6+5+42+72 = 208 clks            RND = 1, SAT = 0
;                       25+28+6*6+5+42+77 = 213 clks            RND = 1, SAT = 1

;       Min Timing:     8+4 = 12 clks

;       PM: 112                                                 DM: 11

;----------------------------------------------------------------------------------------------

FPA24           MOVF            AARGB0,W		; exclusive or of signs in TEMP
                XORWF           BARGB0,W
                MOVWF           TEMP

		CLRF		AARGB2			; clear extended byte
		CLRF		BARGB2

                MOVF            AEXP,W                  ; use AARG if AEXP >= BEXP
                SUBWF           BEXP,W
                BTFSS           _C
                GOTO            USEA24

                MOVF            BEXP,W                  ; use BARG if AEXP < BEXP
                MOVWF           AARGB4			; therefore, swap AARG and BARG
                MOVF            AEXP,W
                MOVWF           BEXP
                MOVF            AARGB4,W
                MOVWF           AEXP

                MOVF            BARGB0,W
                MOVWF           AARGB4
                MOVF            AARGB0,W
                MOVWF           BARGB0
                MOVF            AARGB4,W
                MOVWF           AARGB0

                MOVF            BARGB1,W
                MOVWF           AARGB4
                MOVF            AARGB1,W
                MOVWF           BARGB1
                MOVF            AARGB4,W
                MOVWF           AARGB1

USEA24          MOVF            BEXP,W                  ; return AARG if BARG = 0
                BTFSC           _Z
                RETLW           0x00

                MOVF            AARGB0,W
                MOVWF           SIGN                    ; save sign in SIGN
                BSF             AARGB0,MSB		; make MSB's explicit
                BSF             BARGB0,MSB

                MOVF            BEXP,W                  ; compute shift count in BEXP
                SUBWF           AEXP,W
                MOVWF           BEXP
                BTFSC           _Z
                GOTO            ALIGNED24

                MOVLW           8
                SUBWF           BEXP,W
                BTFSS           _C                      ; if BEXP >= 8, do byte shift
                GOTO            ALIGNB24
                MOVWF           BEXP
                MOVF            BARGB1,W		; keep for postnormalization
		MOVWF		BARGB2
                MOVF            BARGB0,W
		MOVWF		BARGB1
                CLRF            BARGB0

                MOVLW           8
                SUBWF           BEXP,W
                BTFSS           _C                      ; if BEXP >= 8, BARG = 0 relative to AARG
                GOTO            ALIGNB24
                MOVF            SIGN,W
                MOVWF           AARGB0
                RETLW           0x00

ALIGNB24        MOVF            BEXP,W                  ; already aligned if BEXP = 0
                BTFSC           _Z
                GOTO            ALIGNED24

ALOOPB24        BCF             _C                      ; right shift by BEXP
                RRF             BARGB0,F
                RRF             BARGB1,F
		RRF		BARGB2,F
                DECFSZ          BEXP,F
                GOTO            ALOOPB24

ALIGNED24       BTFSS           TEMP,MSB                ; negate if signs opposite
                GOTO            AOK24
		COMF		BARGB2,F
                COMF            BARGB1,F
                COMF            BARGB0,F
                INCF            BARGB2,F
                BTFSC           _Z
                INCF            BARGB1,F
		BTFSC		_Z
		INCF		BARGB0,F

AOK24
                MOVF    	BARGB2,W
                ADDWF   	AARGB2,F
                MOVF            BARGB1,W
                BTFSC           _C
                INCFSZ          BARGB1,W
                ADDWF           AARGB1,F
                MOVF            BARGB0,W
                BTFSC           _C
                INCFSZ          BARGB0,W
                ADDWF           AARGB0,F

                BTFSC           TEMP,MSB
                GOTO            ACOMP24
                BTFSS           _C
                GOTO            NRMRND3224

                RRF             AARGB0,F		; shift right and increment EXP
                RRF             AARGB1,F
		RRF		AARGB2,F
                INCFSZ          AEXP,F
                GOTO            NRMRND3224
                GOTO            SETFOV24

ACOMP24         BTFSC           _C
                GOTO            NRM3224			; normalize and fix sign

		COMF		AARGB2,F
                COMF            AARGB1,F		; negate, toggle sign bit and
                COMF            AARGB0,F		; then normalize
                INCF            AARGB2,F
                BTFSC           _Z
                INCF            AARGB1,F
                BTFSC           _Z
                INCF            AARGB0,F

                MOVLW           0x80
                XORWF           SIGN,F
                GOTO            NRM24

;**********************************************************************************************
;**********************************************************************************************

;	Evaluate floor(x)

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1

;	Use:	CALL	FLOOR24

;	Output:	24 bit floating point number in AEXP, AARGB0, AARGB1

;	Result:	AARG  <--  FLOOR( AARG )

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	37	55	42.7	clks

;		min	max	mean	rms
;	Error:	0x00	0x00	0.0	0.0	nsb

;----------------------------------------------------------------------------------------------

;	floor(x) evaluates the largest integer, as a float, not greater than x.

FLOOR24
		CLRF		AARGB2			; test for zero argument
		MOVF		AEXP,W
		BTFSC		_Z
		RETLW		0x00

		MOVF		AARGB0,W
		MOVWF		AARGB3			; save mantissa
		MOVF		AARGB1,W
		MOVWF		AARGB4

		MOVLW		EXPBIAS			; computed unbiased exponent
		SUBWF		AEXP,W
		MOVWF		TEMPB1
		BTFSC		TEMPB1,MSB
		GOTO		FLOOR24ZERO

		SUBLW		0x10-1
		MOVWF		TEMPB0			; save number of zero bits in TEMPB0
		MOVWF		TEMPB1

		BTFSC		TEMPB1,LSB+3		; divide by eight
		GOTO		FLOOR24MASKH

FLOOR24MASKL
		MOVLW		0x07			; get remainder for mask pointer
		ANDWF		TEMPB0,F
		MOVLW		LOW FLOOR24MASKTABLE
		ADDWF		TEMPB0,F
		MOVLW		HIGH FLOOR24MASKTABLE
		BTFSC		_C
		ADDLW		0x01
		MOVWF		PCLATH
		INCF		TEMPB0,W

		CALL		FLOOR24MASKTABLE	; access table for mask

		ANDWF		AARGB1,F
		BTFSS		AARGB0,MSB		; if negative, round down
		RETLW		0x00

		MOVWF		AARGB7
		MOVF		AARGB4,W
		SUBWF		AARGB1,W
		BTFSS		_Z
		GOTO		FLOOR24RNDL
		RETLW		0x00

FLOOR24RNDL
		COMF		AARGB7,W
		MOVWF		TEMPB1
		INCF		TEMPB1,W
		ADDWF		AARGB1,F
                BTFSC           _Z
                INCF            AARGB0, F
		BTFSS		_Z			; has rounding caused carryout?
		RETLW		0x00
		RRF		AARGB0,F
		RRF		AARGB1,F
		INCFSZ		AEXP,F			; check for overflow
		RETLW		0x00
		GOTO		SETFOV24

FLOOR24MASKH
		MOVLW		0x07			; get remainder for mask pointer
		ANDWF		TEMPB0,F
		MOVLW		LOW FLOOR24MASKTABLE
		ADDWF		TEMPB0,F
		MOVLW		HIGH FLOOR24MASKTABLE
		BTFSC		_C
		ADDLW		0x01
		MOVWF		PCLATH
		INCF		TEMPB0,W

		CALL		FLOOR24MASKTABLE	; access table for mask

		ANDWF		AARGB0,F
		CLRF		AARGB1
		BTFSS		AARGB0,MSB		; if negative, round down
		RETLW		0x00

		MOVWF		AARGB7
		MOVF		AARGB4,W
		SUBWF		AARGB1,W
		BTFSS		_Z
		GOTO		FLOOR24RNDH
		MOVF		AARGB3,W
		SUBWF		AARGB0,W
		BTFSS		_Z
		GOTO		FLOOR24RNDH
		RETLW		0x00

FLOOR24RNDH
		COMF		AARGB7,W
		MOVWF		TEMPB1
		INCF		TEMPB1,W
		ADDWF		AARGB0,F
		BTFSS		_C			; has rounding caused carryout?
		RETLW		0x00
		RRF		AARGB0,F
		RRF		AARGB1,F
		INCFSZ		AEXP,F
		RETLW		0x00
		GOTO		SETFOV24		; check for overflow

FLOOR24ZERO
		BTFSC		AARGB0,MSB
		GOTO		FLOOR24MINUSONE
		CLRF		AEXP
		CLRF		AARGB0
		CLRF		AARGB1
		RETLW		0x00

FLOOR24MINUSONE
		MOVLW		0x7F
		MOVWF		AEXP
		MOVLW		0x80
		MOVWF		AARGB0
		CLRF		AARGB1
		RETLW		0x00

;----------------------------------------------------------------------------------------------

;	table for least significant byte requiring masking, using pointer from
;	the remainder of the number of zero bits divided by eight.

FLOOR24MASKTABLE
		MOVWF		PCL
		RETLW		0xFF
		RETLW		0xFE
		RETLW		0xFC
		RETLW		0xF8
		RETLW		0xF0
		RETLW		0xE0
		RETLW		0xC0
		RETLW		0x80
		RETLW		0x00

;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A < B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1
;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TALTB24

;	Output:	logical result in W

;	Result:	if A < B TRUE,	 W = 0x01
;		if A < B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	9	28	14.6	clks

TALTB24		MOVF		AARGB0,W		; test if signs opposite
		XORWF		BARGB0,W
		MOVWF		TEMPB0
		BTFSC		TEMPB0,MSB
		GOTO		TALTB24O

		BTFSC		AARGB0,MSB
		GOTO		TALTB24N

TALTB24P	MOVF		AEXP,W			; compare positive arguments
		SUBWF		BEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB0,W
		SUBWF		BARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB1,W
		SUBWF		BARGB1,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01
		RETLW		0x00

TALTB24N	MOVF		BEXP,W			; compare negative arguments
		SUBWF		AEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01
		RETLW		0x00

TALTB24O	BTFSS		BARGB0,MSB
		RETLW		0x01
		RETLW		0x00

;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A <= B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1
;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TALEB24

;	Output:	logical result in W

;	Result:	if A <= B TRUE,	 W = 0x01
;		if A <= B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	9	26	14.4	clks

TALEB24		MOVF		AARGB0,W		; test if signs opposite
		XORWF		BARGB0,W
		MOVWF		TEMPB0
		BTFSC		TEMPB0,MSB
		GOTO		TALEB24O

		BTFSC		AARGB0,MSB
		GOTO		TALEB24N

TALEB24P	MOVF		AEXP,W			; compare positive arguments
		SUBWF		BEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB0,W
		SUBWF		BARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB1,W
		SUBWF		BARGB1,W
		BTFSS		_C
		RETLW		0x00
		RETLW		0x01

TALEB24N	MOVF		BEXP,W			; compare negative arguments
		SUBWF		AEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_C
		RETLW		0x00
		RETLW		0x01

TALEB24O	BTFSS		BARGB0,MSB
		RETLW		0x01
		RETLW		0x00

;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A > B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1
;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TAGTB24

;	Output:	logical result in W

;	Result:	if A > B TRUE,	 W = 0x01
;		if A > B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	9	28	14.6	clks

TAGTB24		MOVF		BARGB0,W		; test if signs opposite
		XORWF		AARGB0,W
		MOVWF		TEMPB0
		BTFSC		TEMPB0,MSB
		GOTO		TAGTB24O

		BTFSC		BARGB0,MSB
		GOTO		TAGTB24N

TAGTB24P	MOVF		BEXP,W			; compare positive arguments
		SUBWF		AEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01
		RETLW		0x00

TAGTB24N	MOVF		AEXP,W			; compare negative arguments
		SUBWF		BEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB0,W
		SUBWF		BARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB1,W
		SUBWF		BARGB1,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01
		RETLW		0x00

TAGTB24O	BTFSS		AARGB0,MSB
		RETLW		0x01
		RETLW		0x00

;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A >= B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1
;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TAGEB24

;	Output:	logical result in W

;	Result:	if A >= B TRUE,	 W = 0x01
;		if A >= B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	9	26	14.4	clks

TAGEB24		MOVF		BARGB0,W		; test if signs opposite
		XORWF		AARGB0,W
		MOVWF		TEMPB0
		BTFSC		TEMPB0,MSB
		GOTO		TAGEB24O

		BTFSC		BARGB0,MSB
		GOTO		TAGEB24N

TAGEB24P	MOVF		BEXP,W			; compare positive arguments
		SUBWF		AEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_C
		RETLW		0x00
		RETLW		0x01

TAGEB24N	MOVF		AEXP,W			; compare negative arguments
		SUBWF		BEXP,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB0,W
		SUBWF		BARGB0,W
		BTFSS		_C
		RETLW		0x00
		BTFSS		_Z
		RETLW		0x01

		MOVF		AARGB1,W
		SUBWF		BARGB1,W
		BTFSS		_C
		RETLW		0x00
		RETLW		0x01

TAGEB24O	BTFSS		AARGB0,MSB
		RETLW		0x01
		RETLW		0x00

;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A == B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1
;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TAEQB24

;	Output:	logical result in W

;	Result:	if A == B TRUE,	 W = 0x01
;		if A == B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	5	14	6.9	clks

TAEQB24		MOVF		BEXP,W
		SUBWF		AEXP,W
		BTFSS		_Z
		RETLW		0x00

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_Z
		RETLW		0x00

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_Z
		RETLW		0x00
		RETLW		0x01


;**********************************************************************************************
;**********************************************************************************************

;	Floating Point Relation	A =! B

;	Input:	24 bit floating point number in AEXP, AARGB0, AARGB1

;		24 bit floating point number in BEXP, BARGB0, BARGB1

;	Use:	CALL	TANEB24

;	Output:	logical result in W

;	Result:	if A =! B TRUE,	 W = 0x01
;		if A =! B FALSE, W = 0x00

;	Testing on [-MAXNUM,MAXNUM] from 100000 trials:

;		min	max	mean
;	Timing:	5	14	6.9	clks

TANEB24		MOVF		BEXP,W
		SUBWF		AEXP,W
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB0,W
		SUBWF		AARGB0,W
		BTFSS		_Z
		RETLW		0x01

		MOVF		BARGB1,W
		SUBWF		AARGB1,W
		BTFSS		_Z
		RETLW		0x01
		RETLW		0x00



	end
