//////////////////////////////////////////////////////////////////////////////////
// Engineer:       James C. Ahlstrom
// 
// Create Date:    7 December 2009
// Design Name:    Digital receiver and transmitter
// Module Name:    Receiver
// Project Name:   Transceiver
// Target Devices: Cyclone 3
// Tool versions: 
// Description: 
//
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
// Additional Comments: 
//
//////////////////////////////////////////////////////////////////////////////////

module Receiver(
	input clock,
	input [13:0] adc_data,
	input eth_nRESET,
	input [31:0] rx_tune_phase,
	input [7:0] rx_ctrl,
	output reg [31:0] mem_data,
	output reg [9:0] mem_write_addr,
	output reg mem_wen,
	output reg [31:0] mem_checksum, 		// must be unsigned
	output reg mem_block		// Index of available block 0 or 1
	);
	
	`include "gparam.v"
	
	// Memory blocks for ADC data
	reg [31:0] mem_data3;		// data to write
	reg [15:0] mem_save;		// odd two bytes to save from 6-byte sample
	reg [31:0] checksum;		// must be unsigned

	reg [3:0] rx_state;		// state machine for receive
	parameter sRx		= 0;
	parameter sRx1		= 1;
	parameter sRx2		= 2;
	parameter sRx3		= 3;
	parameter sRx4		= 4;
	parameter sRx5		= 5;
	parameter sRx6		= 6;
	parameter sRx7		= 7;
	parameter sRx8		= 8;
	parameter sRx9		= 9;
	parameter sRx10		= 10;
	parameter sRx11		= 11;
	
	// Tune ADC data with CORDIC
	wire signed [17:0] cordic_real, cordic_imag;
	cordic2 rx_cordic (adc_data, clock, rx_tune_phase, cordic_real, cordic_imag);
	
	// Receive CIC filters followed by FIR filter
	wire decimA_avail, decimB_avail, decim_avail;
	wire signed [17:0] decimA_real, decimB_real;
	wire signed [17:0] decimA_imag, decimB_imag;
	wire signed [23:0] decim_real, decim_imag;
	CicDecimM5 #(.RRRR(8), .IBITS(18), .OBITS(18), .GBITS(15)) cic1 (
		clock, 1'd1, cordic_real, cordic_imag,
				decimA_avail, decimA_real, decimA_imag);
	CicVarDecimM5 cic2 (rx_ctrl[5:0],
		clock, decimA_avail, decimA_real, decimA_imag,
				decimB_avail, decimB_real, decimB_imag);
	firX8R8 fir2 (clock, decimB_avail, decimB_real, decimB_imag,
			decim_avail, decim_real, decim_imag);
	
	//reg [10:0] jim;	// Create an impulse 0, 0, 1, , 0, ...
	//always @(posedge decimB_avail)
	//	if (jim == 560)
	//		jim <= 0;
	//	else
	//		jim <= jim + 1'd1;
	//assign decimB_real = (jim == 0)? 18'd4096 : 18'd0;
	//assign decimB_imag = (jim == 0)? 18'd4096 : 18'd0;
	//assign decimB_real = 18'd4096;	// create constant x input to check sum of coef
	//assign decimB_imag = 18'd4096;

	wire [17:0] csum;	// unsigned for checksum
	assign csum = mem_data[15:0] + mem_data[31:16];

	always @(posedge clock)		// Rx state machine
	begin
		if (eth_nRESET == 0)	// we are in reset
		begin
			mem_write_addr <= 1'd0;
			mem_wen <= 0;
			rx_state <= sRx;
		end
		else case (rx_state)
		sRx:		// Wait for first sample of two samples
		begin
			if (decim_avail)
			begin		// write first dword, save last two bytes LITTLE-ENDIAN
				mem_data <= {decim_imag[7:0], decim_real};
				mem_save <= decim_imag[23:8];
				rx_state <= sRx1;
			end
		end
		sRx1:
		begin
			mem_wen <= 1'd1;
			rx_state <= sRx2;
		end
		sRx2:
		begin
			mem_wen <= 1'd0;
			checksum <= checksum + csum;
			mem_write_addr <= mem_write_addr + 1'd1;
			rx_state <= sRx3;		// wait for next sample
		end
		sRx3:		// Wait for second sample of two samples
		begin
			if (decim_avail)
			begin
				mem_data <= {decim_real[15:0], mem_save};
				mem_data3 <= {decim_imag, decim_real[23:16]};
				rx_state <= sRx4;
			end
		end
		sRx4:		// write second and third dword
		begin
			mem_wen <= 1'd1;
			rx_state <= sRx5;
		end
		sRx5:
		begin
			mem_wen <= 1'd0;
			checksum <= checksum + csum;
			rx_state <= sRx6;
		end
		sRx6:
		begin
			mem_data <= mem_data3;
			mem_write_addr <= mem_write_addr + 1'd1;
			rx_state <= sRx7;
		end
		sRx7:
		begin
			mem_wen <= 1'd1;
			rx_state <= sRx8;
		end
		sRx8:
		begin
			mem_wen <= 1'd0;
			checksum <= checksum + csum;
			rx_state <= sRx9;
		end
		sRx9:		// increment memory address
		begin
			if (mem_write_addr == MEM_DWORD_SIZE - 1)
			begin		// first block of memory is ready to be sent
				mem_write_addr <= mem_write_addr + 1'd1;
				mem_checksum <= checksum;
				checksum <= 1'd0;
				rx_state <= sRx10;
			end
			else if (mem_write_addr == MEM_DWORD_SIZE * 2 - 1)
			begin		// second block of memory is ready to be sent
				mem_write_addr <= 1'd0;
				mem_checksum <= checksum;
				checksum <= 1'd0;
				rx_state <= sRx11;
			end
			else
			begin
				mem_write_addr <= mem_write_addr + 1'd1;
				rx_state <= sRx;
			end
		end
		sRx10:
		begin
			mem_block <= 1'd0;
			rx_state <= sRx;
		end
		sRx11:
		begin
			mem_block <= 1'd1;
			rx_state <= sRx;
		end
		endcase
	end

endmodule
	