
# Create the module fircoef.v from the file coef*.dat.
# Create the memory initialization files cxx.mif from the file coef*.dat.

import sys

# WIDTH is the width of a data word in bits == 18
# DEPTH is the memory size in words
DEPTH = 512
# The FIR filter decimation
DECIM = 20
# MULTS is the number of multipliers to use
MULTS = 30

NUM = DEPTH  # Number of coefficients needed.  Ending coef may be zero.

MIF = """DEPTH = %d;
WIDTH = 18;
ADDRESS_RADIX = HEX;
DATA_RADIX = HEX;
CONTENT
BEGIN
"""

def get_coef(fname, prnt=0, scale = 1.0):
	fp = open(fname)
	lines = fp.readlines()
	fp.close()
	f = []
	l = []
	big = 0
	asum = 0
	sum = 0
	for line in lines:
		if line[0:2] != 'a[':
			continue
		x = line.split()[2]
		x = float(x)
		sum += x
		a = abs(x)
		if a > big:
			big = a
		asum += a
		f.append(x)
	#if len(f) != NUM:
	#	print "Have %d coef but need %d" % (len(f), NUM)
	#	sys.exit(1)
	NUM = len(f)
	# Scale coefficient to 18 bits
	factor = (2**17 - 1) / big * scale
	for i in range(NUM):
		x = f[i] * factor
		if x < 0:
			l.append(-int(-x + 0.5))
		else:
			l.append(int(x + 0.5))
	sym = 'SYM'
	for i in range (0, NUM / 2):
		if l[i] != l[NUM - 1 - i]:
			sym = 'NOT SYM'
			break
	if prnt:
		isum = 0
		for i in range (len(l)):
			print "%4d  %8d  %12.8f  %12.2f" % (i, l[i], f[i], f[i] * factor)
			isum += l[i]
		print 'sum', sum, 'big', big, 'asum', asum, 'sym', sym, 'isum', isum
		print "factor:  big %.2f  asum %.2f  asum/big %.2f" % (big * factor, asum * factor, asum/big)
	return l


def mif():
	l = get_coef("coef600.dat")
	for i in range(len(l)):
		v = l[i]
		if v < 0:
			l[i] = (v & 0x3FFFF)
	index = 0
	for n in range(MULTS-1, -1, -1):
		fp = open("c%02d.mif" % n, "w")
		fp.write(MIF % DEPTH)
		for addr in range(0, DEPTH):
			if addr < DECIM:
				v = l[index]
				index += 1
			else:	# addresses after DECIM are zero
				v = 0
			fp.write("%02X : %05X;\n" % (addr, v))
		fp.write("END;\n\n")
		fp.close()

def array():
	fp = open("fircoef.v", "w")
	fp.write("// MACHINE CREATED FILE (make_coef.py) DO NOT EDIT\n")
	fp.write('\n`include "header.v"\n\nmodule fircoef (clock, addr,\n\t')
	for n in range(MULTS-1):
		fp.write(" coef%02d," % n)
		if n > 0 and n % 7 == 0:
			fp.write("\n\t")
	fp.write(" coef%02d);\n\tinput clock;\n\tinput [`ADBITS-1:0] addr;" % (MULTS-1))
	fp.write("\n\toutput reg signed [`PMBITS-1:0]\n\t")
	for n in range(MULTS-1):
		fp.write(" coef%02d," % n)
		if n > 0 and n % 7 == 0:
			fp.write("\n\t")
	fp.write(" coef%02d;\n\n" % (MULTS-1))
	for n in range(MULTS):
		fp.write(
		"\treg signed [`PMBITS-1:0] c%02d [0:`RRRR - 1];\n"
		 % (n,))
	fp.write("\n\talways @(posedge clock)\n\tbegin\n")
	for n in range(MULTS):
		fp.write("\t\tcoef%02d <= c%02d[addr];\n" % (n, n))
	fp.write("\tend\n")
	fp.write("\n\tinitial\n\tbegin\n")
	coefs = get_coef("coef400.dat")
	index = 0
	for n in range (MULTS):
		for i in range(DECIM):
			fp.write("\t\tc%02d[%2d] = %d;\n" % (n, i, coefs[index]))
			index += 1
	fp.write("\tend\nendmodule\n")
	fp.close()

def regs():
	coefs = get_coef("coef40.dat")
	fp = open("fircoefs.v", "w")
	index = 0
	for i in range (MULTS-1, -1, -1):
		for j in range (0, DECIM):
			k = j + ord('A')
			fp.write("\tparameter signed [MBITS-1:0] coef%c%02d = %d;\n" %
				(chr(k), i, coefs[index]))
			index += 1

def linear(source):
	coefs = get_coef(source + ".dat")
	fp = open(source + ".mif", "w")
	fp.write(MIF % DEPTH)
	num = len(coefs)
	for addr in  range(DEPTH):
		if addr < num:
			v = coefs[addr]
			if v < 0:
				v = (v & 0x3FFFF)
			fp.write("%02X : %05X;\n" % (addr, v))
		else:
			fp.write("%02X : %05X;\n" % (addr, 0))
	fp.write("END;\n\n")
	fp.close()

def test_linear():
	coefs = get_coef("coef512.dat", 0)
	coefs = coefs + [0]
	count = 0
	start = 0
	all = []
	for i in range(20):
		all.append(i)
	for s in range(50000):
		count += 1
		if count > 512:
			count = 0
			start = 1
		if s % 20 == 0:
			if start:
				print
				if count in all:
					all.remove(count)
			print "%4d  %8d" % (count, coefs[count])
			start = 0
		if not all:
			break
def EvenOdd(source):
	coefs = get_coef(source + ".dat", 1)
	length = len(coefs)
	fpe = open(source + "e.mif", "w")
	fpo = open(source + "o.mif", "w")
	fpe.write(MIF % DEPTH)
	fpo.write(MIF % DEPTH)
	index = 0
	for addr in  range(0, length/2):
		v = coefs[index]
		if v < 0:
			v = (v & 0x3FFFF)
		fpe.write("%02X : %05X;\n" % (addr, v))
		index += 1
		v = coefs[index]
		if v < 0:
			v = (v & 0x3FFFF)
		fpo.write("%02X : %05X;\n" % (addr, v))
		index += 1
	for addr in  range(length/2, 512):
		fpe.write("%02X : %05X;\n" % (addr, 0))
		fpo.write("%02X : %05X;\n" % (addr, 0))
	fpe.write("END;\n\n")
	fpe.close()
	fpo.write("END;\n\n")
	fpo.close()

def mif8(source, scale):	# Make eight banks
	global DEPTH
	DEPTH = 256
	coefs = get_coef(source + ".dat", 1, scale)
	length = len(coefs)
	for n in range(8):
		fp = open("%s%c.mif" % (source, chr(ord('A') + n)), "w")
		fp.write(MIF % DEPTH)
		index = length - 8 + n
		addr = 0
		for i in  range(DEPTH):
			if index >= 0:
				v = coefs[index]
			else:
				v = 0
			if v < 0:
				v = (v & 0x3FFFF)
			fp.write("%02X : %05X;\n" % (addr, v))
			# if index >= 0: print "row %d  index %d" %  (n, index)
			index -= 8
			addr += 1
		fp.write("END;\n\n")
		fp.close()

# Currently using EvenOdd and linear and mif8
if __name__ == "__main__":
	#name = "coefR8"
	#EvenOdd(name)
	#linear(name)
	mif8("coefL8", 0.63)
