//////////////////////////////////////////////////////////////////////////////////
// Engineer:       James C. Ahlstrom
// 
// Create Date:    29 June 2009
// Design Name:    Ethernet interface for LAN9115 controller
// Module Name:    ethernet
// Project Name:   Transceiver
// Target Devices: Cyclone 3
// Tool versions: 
// Description: 
//
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
// Additional Comments: 
//
//////////////////////////////////////////////////////////////////////////////////

module ethernet(
	// Public interface
	input clock,
	input key_down,
	output reg [6:0] LED,
	output reg TestC,
	output reg [7:0] rx_ctrl,				// receive control
	output reg [31:0] rx_tune_phase,		// receive tune phase for CORDIC
	output reg [7:0] tx_ctrl,				// transmit control
	output reg [31:0] tx_tune_phase,		// transmit tune phase
	// access to ADC data in RAM
	output reg [9:0] rx_mem_read_addr,		// read address in dwords
	input [31:0] rx_mem_q,					// read data
	input [31:0] rx_mem_checksum,			// checksum of data in RAM
	input rx_mem_block,						// index of available block, 0 or 1
	// access to DAC data in RAM
	output signed [31:0] tx_sample,			// next sample to transmit
	input get_next_dac,						// toggle to get next sample
	// Ethernet ASIC bus interface
	output [6:0] eth_address_bus,
	inout  [15:0] eth_data_bus,
	output reg eth_nRD,
	output reg eth_nWR,
	output reg eth_nCS,
	output eth_FIFO_SEL,
	input eth_nRESET,
	input eth_IRQ,
	// Output level DAC
	output reg [7:0] level_data,		// Output level control
	output reg level_nCS,
	output reg level_nWR
	);
	
	// Our MAC address in network order
	parameter MAC1	= 8'h00;
	parameter MAC2	= 8'h0B;
	parameter MAC3	= 8'hB8;	
	parameter MAC4	= 8'h60;	
	parameter MAC5	= 8'h0C;	
	parameter MAC6	= 8'h99;
	// Our MAC address in Ethernet controller order
	parameter [31:0] MAC_ADDRESS_H	= {8'd0, 8'd0, MAC6, MAC5};
	parameter [31:0] MAC_ADDRESS_L	= {MAC4, MAC3, MAC2, MAC1};
	// Our MAC address in machine order
	parameter [47:0] MAC_ADDRESS	= {MAC6, MAC5, MAC4, MAC3, MAC2, MAC1};
		
	// Our IP address in network order
	parameter IP1	= 8'd192;
	parameter IP2	= 8'd168;
	parameter IP3	= 8'd2;
	parameter IP4	= 8'd196;
	// Our IP address in machine order
	parameter [31:0] IP_ADDRESS = {IP4, IP3, IP2, IP1};

	// Our UDP port numbers
	parameter UDP_PORT_ADC = 16'hBC77;			// Port for sending ADC samples
	parameter UDP_PORT_CTR = UDP_PORT_ADC + 16'h1;	// Port for control
	parameter UDP_PORT_DAC = UDP_PORT_ADC + 16'h2;	// Port for receiving transmit audio
	parameter UDP_PORT_KEY = 16'h553C;			// Port for sending key state
	
	`include "gparam.v"
	
	// Register addresses for LAN9115 Ethernet controller ASIC
	parameter IRQ_CFG		= 8'h54;
	parameter HW_CFG		= 8'h74;
	parameter GPIO_CFG		= 8'h88;
	parameter INT_EN		= 8'h5C;
	parameter MAC_CSR_CMD	= 8'hA4;
	parameter MAC_CSR_DATA	= 8'hA8;
	parameter BYTE_TEST		= 8'h64;
	parameter PMT_CTRL		= 8'h84;
	parameter RX_STATUS		= 8'h40;
	parameter RX_DATA		= 8'h00;
	parameter RX_FIFO_INF	= 8'h7C;
	parameter INT_STS		= 8'h58;
	parameter TX_STATUS		= 8'h48;
	parameter TX_DATA		= 8'h20;
	parameter TX_FIFO_INF	= 8'h80;
	parameter TX_CFG		= 8'h70;
	parameter AFC_CFG		= 8'hAC;
	
	// States for the Ethernet FSM
//(* syn_encoding = "one-hot" *) reg [7:0] state;		// State of Ethernet FSM
	reg [7:0] state;					// State of Ethernet FSM
	reg [7:0] return_state;				// Where to return after a branch in state
	reg [7:0] bus_return;				// Where to return from a bus read/write
	parameter sMacWrite		= 8'd22;	// Proceedure to write to a MAC register
	parameter sMacRead		= 8'd23;	// Proceedure to read from a MAC register
	parameter sMacA			= 8'd24;
	parameter sMacB			= 8'd25;
	parameter sMacC			= 8'd26;
	parameter sIdle			= 8'd30;	// Idle state; wait for something to do
	parameter sIdleA		= 8'd31;
	parameter sIdleRxA		= 8'd35;
	parameter sIdleRxB		= 8'd36;
	parameter sRxPacket		= 8'd40;	// Receive a packet from Ethernet
	parameter sRxPacketA	= 8'd41;
	parameter sRxPacketB	= 8'd42;
	parameter sRxPacketC	= 8'd43;
	parameter sRxPacketD	= 8'd44;
	parameter sRxPacketE	= 8'd45;
	parameter sRxIp			= 8'd48;	// Handle IP packet
	parameter sRxIpA		= 8'd49;	
	parameter sRxSkip		= 8'd52;	// Skip remainder of current packet
	parameter sRxArp		= 8'd54;	// Handle ARP request
	parameter sRxArpA		= 8'd55;
	parameter sRxIcmp		= 8'd56;	// Handle ICMP request
	parameter sRxIcmpA		= 8'd57;
	parameter sRxUdp		= 8'd60;	// Handle UDP packet
	parameter sRxUdpA		= 8'd61;
	parameter sRxUdpB		= 8'd62;
	parameter sRxUdpC		= 8'd63;
	parameter sRxUdpD		= 8'd64;
	parameter sRxUdpE		= 8'd65;
	parameter sRxUdpF		= 8'd66;
	parameter sSendIp		= 8'd70;	// Send IP packet
	parameter sSendIpA		= 8'd71;	
	parameter sSendEth		= 8'd75;	// Send Ethernet packet
	parameter sSendEthA		= 8'd76;
	parameter sSendEthB		= 8'd77;	
	parameter sSendEthC		= 8'd78;
	parameter sSendEthD		= 8'd79;	
	parameter sBufChecksum	= 8'd80;	// Calculate checksum of eth_buf_data
	parameter sBufChecksumA	= 8'd81;
	parameter sBufChecksumB	= 8'd82;
	parameter sBufChecksumC	= 8'd83;
	parameter sBufChecksumD	= 8'd84;
	parameter sSendAdc		= 8'd85;	// Send ADC data back to the PC
	parameter sSendAdcData	= 8'd86;
	parameter sSendAdcDataA	= 8'd87;
	parameter sSendKeyState	= 8'd90;	// Send key up/down state to the PC
	parameter sSendUdp		= 8'd95;	// Send UDP packet
	parameter sSendUdpA		= 8'd96;
	parameter sSendUdpB		= 8'd97;
	parameter sSendUdpC		= 8'd98;
	parameter sEtherBus		= 8'd105;	// Read or write the Ethernet bus
	parameter sEtherBusA	= 8'd106;
	parameter sEtherBusB	= 8'd107;	
	parameter sEtherBusC	= 8'd108;	
	parameter sEtherBusD	= 8'd109;	
	parameter sEtherBusE	= 8'd110;
	parameter sEtherBusF	= 8'd111;
	parameter sEtherBusG	= 8'd112;
	parameter sEtherBusH	= 8'd113;
	parameter sDelay		= 8'd115;	// Generate a delay
	parameter sDelayA		= 8'd116;
	parameter sTxAudio		= 8'd120;	// Receive the transmit audio
	parameter sTxAudioA		= 8'd121;
	parameter sTxAudioB		= 8'd122;
	parameter sTxAudioC		= 8'd123;
	parameter sTxAudioD		= 8'd124;
	parameter sTxAudioE		= 8'd125;
	parameter sTxAudioF		= 8'd126;
	
	// Keep the Ethernet headers in a register data buffer.
	// The first byte is eth_buf_data[7:0], the second is eth_buf_data[15:8], and
	// the byte at index is eth_buf_data[8 * index += 8].
	parameter ETH_BUF_SIZE	= 80;				// size in bytes of the buffer; must be even by 4
	parameter ETH_BUF_ABITS	= 7;				// number of bits in buffer address
	reg [ETH_BUF_ABITS-1:0] eth_buf_addr;		// buffer read/write address in bytes
	reg [ETH_BUF_ABITS-1:0] eth_buf_addr2;		// second address
	reg [8*ETH_BUF_SIZE+32-1:0] eth_buf_data;	// buffer for read/write data

/*Here is the layout of an Ethernet Packet:
;*
;*  The Ethernet header (14 bytes) is:
;*		0 - 5	Ethernet destination
;*		6 - 11	Ethernet source
;*		12 - 13	Ethernet length/type (IP is 0x0800); the remainder is IP data:
;*
;*	The IP header (20 bytes) is:
;*		14		Version == 0x45
;*		15		Type of service
;*		16 - 17	Length of IP header and data
;*		18 - 19	Identification
;*		20 - 21	Flags and fragment offset
;*		22		Time to Live
;*		23		Protocol: 0x11==UDP
;*		24 - 25	Header checksum
;*		26 - 29	Source IP address
;*		30 - 33	Destination address
;*		34      Start of IP data
;*
;*	The UDP header (8 bytes) is:
;*		34 - 35	Source port
;*		36 - 37	Destination port
;*		38 - 39	UDP length
;*		40 - 41	UDP checksum
;*		42		Start of UDP data	*/
	// Ethernet header items.  Some quantities need byte reversal to machine order.
	`define EthMacDest		eth_buf_data[8 * 0  +: 48]
	`define EthMacSource	eth_buf_data[8 * 6  +: 48]
	`define EthType			{eth_buf_data[8 * 12 +:  8], eth_buf_data[8 * 13 +:  8]}
	`define EthIpVer		eth_buf_data[8 * 14 +:  8]
	`define EthIpType		eth_buf_data[8 * 15 +:  8]
	`define EthIpLength		{eth_buf_data[8 * 16 +:  8], eth_buf_data[8 * 17 +:  8]}
	`define EthIpSeq		{eth_buf_data[8 * 18 +: 8], eth_buf_data[8 * 19 +: 8]}
	`define EthIpFlags		eth_buf_data[8 * 20 +: 16]
	`define EthIpTtl		eth_buf_data[8 * 22 +:  8]
	`define EthIpProtocol	eth_buf_data[8 * 23 +:  8]
	`define EthIpChecksum	eth_buf_data[8 * 24 +: 16]
	`define EthIpSource		eth_buf_data[8 * 26 +: 32]
	`define EthIpDest		eth_buf_data[8 * 30 +: 32]
	`define EthUdpSourcePort	{eth_buf_data[8 * 34 +: 8], eth_buf_data[8 * 35 +: 8]}
	`define EthUdpDestPort		{eth_buf_data[8 * 36 +: 8], eth_buf_data[8 * 37 +: 8]}	
	`define EthUdpLength		{eth_buf_data[8 * 38 +: 8], eth_buf_data[8 * 39 +: 8]}	
	`define EthUdpChecksum		eth_buf_data[8 * 40 +: 16]
	wire [15:0] EthUdpData2Bytes	=	eth_buf_data[8 * 42 +: 16];

	reg [31:0] rx_status;		// status of current received packet
	reg [13:0] rx_count;		// number of bytes read
	reg [31:0] checksum;		// checksum calculation
	reg [15:0] tempword;		// temporary word of 16 bits
	reg [15:0] ip_sequence;		// sequence number for IP packets sent
	reg [10:0] send_size;			// number of bytes to transmit
	reg [15:0] ip_end_byte;		// end byte of IP packet = IP length + 14
	reg [15:0] udp_end_byte;	// end byte of UDP packet = UDP length + 34
	reg ip_in_buffer;			// IP packet fits into buffer
	reg send_is_adc;				// Are we transmitting ADC data?
	//reg old_key_state;			// Last key state sent to the PC

	// These are the target address for returning ADC samples
	reg [47:0] adc_eth_addr;
	reg [31:0] adc_ip_addr;
	reg [15:0] adc_udp_port;
	
	reg [7:0] mem_sequence;		// packet sequence 0, 1, 2, ...
	reg current_mem_block;		// index of the last block we sent
	
	// data for the Ethernet bus
	reg [5:0] address;			// Ethernet bus address
	reg [31:0] write_data;		// Ethernet data to write
	reg [31:0] read_data;		// Ethernet data read from bus
	reg do_write;				// Request a write
	reg do_read;				// Request a read
	reg [3:0] mac_index;		// Index of MAC register to write
	wire [15:0] bus_timer;		// Timer for Ethernet bus transactions
	reg bus_timer_clear;		// Clear the bus timer to zero
	reg second;					// Indicates second 16-bit read/write
	wire [15:0] data_16;		// 16-bit data to write to the controller data bus
	
	assign eth_address_bus[6:1] = address;
	assign eth_address_bus[0] = second;
	assign data_16 = second ? write_data[31:16] : write_data[15:0];
	assign eth_data_bus = do_read ? 16'bz : data_16;
	assign eth_FIFO_SEL = 0;
	
	// This is a timer for Ethernet bus transactions
	eth_bus_timer ebustime (clock, bus_timer_clear, bus_timer);
	
	// Memory for transmit
	reg signed [31:0] tx_mem_data;		// data to write
	reg [11:0] tx_mem_read_addr;		// read address
	reg [11:0] tx_mem_write_addr;		// write address
	reg tx_mem_wen;						// write enable
	wire signed [31:0] tx_mem_q;
	eth_ram2 dac_ram (clock, tx_mem_data, tx_mem_read_addr, tx_mem_write_addr, tx_mem_wen, tx_mem_q);
	
	task eth_read (input [7:0] addr);
	begin
		address <= addr[7:2];
		do_read <= 1'd1;
	end
	endtask
	
	task eth_write (input [7:0] addr, input [31:0] data);
	begin
		address <= addr[7:2];
		write_data <= data;
		do_write <= 1'd1;
	end
	endtask

	function [15:0] nsec;	// return the number of clock ticks for input nanoseconds
		input integer nsecs;
		integer x;
		x = ((eth_clock_freq / 100000) * nsecs) / 10000 + 1;
		nsec = x[15:0];
	endfunction

	// Transmitter
	reg [2:0] tx_clock_sync;		// synchronize to tx clock
	reg [1:0] tx_state;				// State machine for transmit
	parameter sTxFill	= 2'b00;	// Wait for memory to become half full
	parameter sTxRun	= 2'b01;	// Return memory samples to transmitter
	parameter sTxUnFlow	= 2'b10;	// Underflow while running
	reg [11:0] tx_mem_count;		// Number of samples remaining in memory
	assign tx_sample = (tx_state == sTxRun) ? tx_mem_q : 32'd0;
	always @(posedge clock)
	begin		// Provide the next transmit sample to the DAC
		if (key_down)
		begin
			tx_mem_count <= tx_mem_write_addr - tx_mem_read_addr;
			tx_clock_sync[0] <= get_next_dac;
			tx_clock_sync[1] <= tx_clock_sync[0];
			tx_clock_sync[2] <= tx_clock_sync[1];
			case (tx_state)
			sTxFill:
			begin
				if (tx_mem_count[11])		// memory is half full
					tx_state <= sTxRun;
			end
			sTxRun:
			begin
				if (tx_mem_count[11:4] == 8'hFF)
				begin
					LED[6] <= 1'd1;		// Tx near overflow
				end
				if (tx_mem_count == 12'd0)
				begin
					tx_state <= sTxUnFlow;
				end
				else if (tx_clock_sync[2] != tx_clock_sync[1])
				begin
					tx_mem_read_addr <= tx_mem_read_addr + 1'd1;
				end
			end
			sTxUnFlow:
			begin
				tx_state <= sTxFill;
				LED[5] <= 1'd1;		// Tx underflow error
			end
			endcase
		end
		else
		begin
			tx_state <= sTxFill;
			tx_mem_read_addr <= 1'd0;
			tx_mem_count <= 12'd0;
			LED[5] <= 1'd0;
			LED[6] <= 1'd0;
		end
	end
			
	always @(posedge clock)
	begin
		if (eth_nRESET == 1'd0)		// We are in reset; wait.
		begin
			state <= 0;
			do_write <= 0;
			do_read <= 0;
			ip_sequence <= 0;
			LED[4:0] <= 1'd0;
			TestC <= 1'd0;
			eth_nRD	<= 1'd1;
			eth_nWR <= 1'd1;
			eth_nCS <= 1'd1;
			bus_timer_clear <= 1'd1;
			send_is_adc <= 1'd0;
			rx_tune_phase <= 32'd251658240;	// 7.2 MHz for 122.88 MHz clock
			tx_tune_phase <= 32'd251658240;
			adc_udp_port <= 1'd0;
			mem_sequence <= 1'd0;
			level_nCS <= 1'd1;
			level_nWR <= 1'd1;
			tx_ctrl <= 1'd0;
			tx_mem_wen <= 1'd0;
		end
		else
		begin
		// *********************************
		// Finite state machine for Ethernet.
			case (state)
			0:	// read ready bit that indicates reset complete
			begin
				eth_read(PMT_CTRL);
				bus_return <= 1;
				state <= sEtherBus;
			end
			1:	// wait for READY bit
			begin		
				if (read_data[0] == 0)
					state <= 0;	// not complete, go back and re-read
				else
					state <= 2;	// complete	
			end
			// Reset was released; write configuration registers.
			// Write PHY register 4 to advertize full duplex.
			// For PHY registers we use MAC registers MII_ACC and MII_DATA, and check bit 0 for completion.			
			2:
			begin
				mac_index <= 4'd7;		// write to MII_DATA
				eth_write(MAC_CSR_DATA, 16'b111100001);	// data is for PHY register 4
				bus_return <= sMacWrite;
				state <= sEtherBus;
				return_state <= 3;
			end
			3:
			begin
				mac_index <= 4'd6;		// write to MII_ACC
				eth_write(MAC_CSR_DATA, 16'b1_00100_000011);	//write MII_DATA to PHY register 4
				bus_return <= sMacWrite;
				state <= sEtherBus;
				return_state <= 4;
			end
			4:		// read MII_ACC at index 6
			begin
				state <= sMacRead;
				return_state <= 5;	
			end
			5:		// value of MII_ACC is in MAC_CSR_DATA
			begin
				eth_read (MAC_CSR_DATA);
				bus_return <= 6;
				state <= sEtherBus;
			end
			6:
			begin			
				if (read_data[0] != 0)
					state <= 4;		// not complete
				else
					state <= 10;	// complete
			end			
			10:
			begin
				//eth_write(IRQ_CFG, 32'b1_0001_0001);
				eth_write(IRQ_CFG, 32'b0_0001_0001);	// disable eth_IRQ
				bus_return <= 11;
				state <= sEtherBus;
			end
			11:
			begin
				eth_write(HW_CFG, 32'b10110_0000_0000_0000_0000);
				bus_return <= 12;
				state <= sEtherBus;
			end
			12:
			begin
				eth_write(GPIO_CFG, 32'h70_00_00_00);
				bus_return <= 13;
				state <= sEtherBus;
			end
			13:
			begin
				eth_write(INT_EN, 32'h0008);
				bus_return <= 14;
				state <= sEtherBus;
			end					
			14:			// write MAC register ADDRH, index 2
			begin
				mac_index <= 4'd2;		// index 2
				eth_write(MAC_CSR_DATA, MAC_ADDRESS_H);	// data is MAC address high
				bus_return <= sMacWrite;
				state <= sEtherBus;
				return_state <= 15;		
			end
			15:			// write MAC register ADDRL, index 3
			begin
				mac_index <= 4'd3;
				eth_write(MAC_CSR_DATA, MAC_ADDRESS_L);
				bus_return <= sMacWrite;
				state <= sEtherBus;
				return_state <= 16;		
			end
			16:			// write MAC register MAC_CR, index 1			
			begin
				mac_index <= 4'd1;
				eth_write(MAC_CSR_DATA, 1<<8 | 1<<3 | 1<<2);
				bus_return <= sMacWrite;
				state <= sEtherBus;	
				return_state <= 17;		
			end
			17:			// write MAC register FLOW, index 8		
			begin
				mac_index <= 4'd8;
				eth_write(MAC_CSR_DATA, 32'h044402);
				bus_return <= sMacWrite;
				state <= sEtherBus;
				return_state <= 18;		
			end			
			18:
			begin
				eth_write(AFC_CFG, 32'h006E37D1);
				bus_return <= 19;
				state <= sEtherBus;	
			end
			19:
			begin
				eth_write(TX_CFG, 32'h06);
				bus_return <= sIdle;
				state <= sEtherBus;	
			end
			// For MAC registers, we use MAC_CSR and check MAC_CSR_CMD bit 31 for completion.			
			sMacWrite:		// MAC write: data was already written to register MAC_CSR_DATA
			begin		
				// write MAC index to register MAC_CSR_CMD, clear bit 30 for write
				eth_write(MAC_CSR_CMD, 32'h80000000 | mac_index);
				bus_return <= sMacA;
				state <= sEtherBus;		
			end
			sMacRead:		// MAC read
			begin
				// write MAC index to register MAC_CSR_CMD, set bit 30 for read
				eth_write(MAC_CSR_CMD, 32'hC0000000 | mac_index);
				bus_return <= sMacA;
				state <= sEtherBus;			
			end
			sMacA:		// required delay
			begin
				eth_read(BYTE_TEST);
				bus_return <= sMacB;
				state <= sEtherBus;		
			end
			sMacB:		// read status
			begin
				eth_read (MAC_CSR_CMD);
				bus_return <= sMacC;
				state <= sEtherBus;		
			end
			sMacC:	// check bit 31 to see if the operation is complete
			begin			
				if (read_data[31] != 0)
					state <= sMacA;			// not complete
				else
					state <= return_state;	// complete
			end
			// Idle state; wait for packet
			sIdle:	//***************************
			// My old logic based on eth_IRQ failed.
			begin
				if ( ! key_down)
					tx_mem_write_addr <= 1'd0;
				if (rx_mem_block != current_mem_block && adc_udp_port != 0)
				begin		// is there a new block?
					state <= sSendAdc;		// there is ADC data ready to be sent
					current_mem_block <= rx_mem_block;
				end
				else
				begin
					state <= sIdleA;
				end
			end
			sIdleA:
			begin		// Poll to see if we have received data.
				eth_read (RX_FIFO_INF);
				bus_return <= sIdleRxA;
				state <= sEtherBus;
			end
			sIdleRxA:
			begin	// size of RX_STATUS data available is in bits [23:16]
				if (read_data[23:16] != 0)
				begin			// status data is available, read the packet
					eth_read(RX_STATUS);
					bus_return <= sRxPacket;	
					state <= sEtherBus;
				end
				else
				begin		// no packet available, read error status
					eth_read(INT_STS);
					bus_return <= sIdleRxB;
					state <= sEtherBus;		
				end
			end
			sIdleRxB:
			begin
				LED[0] <= read_data[10];	// Ethernet controller errors
				LED[1] <= read_data[25] | read_data[13] | read_data[10];		// Tx errors
				LED[2] <= read_data[24] | read_data[15] | read_data[14] |
					read_data[6] | read_data[4];								// Rx errors
				LED[3] <= read_data[13];
				bus_return <= sIdle;
				state <= sDelay;		
			end
			// Receive an Ethernet packet
			sRxPacket:	//***********************
			begin
				rx_count <= 1'd0;
				rx_status <= read_data;
				if (read_data[29:16] == 0)	// zero packet length should not happen
				begin
					LED[4] <= 1'd1;		// Rx error
					bus_return <= sIdle;
					state <= sDelay;
				end
				else
					state <= sRxPacketB;
			end
			sRxPacketB:	// LOOP: read the Ethernet packet into our buffer
			begin
				eth_read (RX_DATA);
				rx_count <= rx_count + 4'd4;	// add four bytes to address
				bus_return <= sRxPacketC;
				state <= sEtherBus;
			end
			sRxPacketC:
			begin
				eth_buf_data[rx_count * 8 - 1 -:32] <= read_data;	// save data		
				if ((rx_count < rx_status[29:16]) && (rx_count < ETH_BUF_SIZE))
					state <= sRxPacketB;	// skip to start of loop
				else
					state <= sRxPacketD;	// skip to end of loop
			end
			sRxPacketD:	// END LOOP; examine packet in buffer
			begin
				return_state <= sIdle;
				send_is_adc <= 1'd0;		
				if (rx_status[15] == 1)		// error bit
					state <= sRxSkip;
				else if (rx_status[13] == 1)		// broadcast packet
				begin
					if (`EthType == 16'h0806)			// type 0x0806 is ARP
						state <= sRxArp;
					else
						state <= sRxSkip;
				end
				else if (`EthType == 16'h0800 &&		// type 0x0800 is IP
						`EthIpDest == IP_ADDRESS &&
						`EthIpVer == 8'h45)
					state <= sRxIp;
				else
					state <= sRxSkip;
			end
			sRxSkip:	// Read and discard the remaining bytes of the current packet
			begin
				if (rx_count < rx_status[29:16])	// packet length
				begin
					eth_read (RX_DATA);
					bus_return <= sRxSkip;
					state <= sEtherBus;
					rx_count <= rx_count + 4'd4;	// add four bytes to address					
				end
				else
					state <= return_state;
			end
			sRxIp:	// We have an IP packet; calculate the IP header checksum
			begin
				ip_end_byte <= `EthIpLength + 8'd14;
				udp_end_byte <= `EthUdpLength + 8'd34;
				checksum <= 1'd0;
				eth_buf_addr <= 14;
				eth_buf_addr2 <= 34;
				return_state <= sRxIpA;
				state <= sBufChecksum;
			end
			sRxIpA:
			begin
				ip_in_buffer <= (ip_end_byte <= ETH_BUF_SIZE);
				return_state <= sIdle;			
				if (checksum[15:0] == 16'hFFFF)		// checksum succeeds
				begin
					if (`EthIpProtocol == 8'h01)		// IP protocol 0x01 is ICMP
						state <= sRxIcmp;
					else if (`EthIpProtocol == 8'h11)	// IP protocol 0x11 is UDP
						state <= sRxUdp;
					else
						state <= sRxSkip;
				end
				else
				begin
					state <= sRxSkip;
				end
			end			
/******************************************************************
;*	Perform ARP Response
;*   This routine supplies a requesting computer with the
;*   Ethernet modules's MAC (hardware) address.
;*
;*  ARP packets have the following byte offsets and fields:
;*		0 - 5	Ethernet destination
;*		6 - 11	Ethernet source
;*		12 - 13	Ethernet length/type (ARP is 0x0806); the remainder is ARP data:
;*		14 - 15	Hardware address type, 0x0001
;*		16 - 17	Protocol address type, 0x0800
;*		18		Hardware size, 6
;*		19		Protocol size, 4
;*		20 - 21	Operation; 0x0001==request, 0x0002==reply
;*		22 - 27	MAC source address
;*		28 - 31	IP source address
;*		32 - 37	MAC destination address
;*		38 - 41	IP destination address
;*		42 - 59	Zeros to fill out to minimum length
;********************************************************/		
			sRxArp:	// we have a type 0x0806 arp packet	
			begin				
				if (eth_buf_data[8 * 38 +: 32] != IP_ADDRESS ||	// target IP address
					eth_buf_data[8 * 14 +: 8] != 0 || 
					eth_buf_data[8 * 15 +: 8] != 1 ||
					eth_buf_data[8 * 16 +: 8] != 8 ||
					eth_buf_data[8 * 17 +: 8] != 0 ||
					eth_buf_data[8 * 18 +: 8] != 6 ||
					eth_buf_data[8 * 19 +: 8] != 4 ||
					eth_buf_data[8 * 20 +: 8] != 0 ||
					eth_buf_data[8 * 21 +: 8] != 1)
				begin
					return_state <= sIdle;				
					state <= sRxSkip;
				end
				else	// respond to this ARP request
				begin			
					// Copy the source Mac address to the destination
					eth_buf_data[8 * 32 +: 48] <= `EthMacSource;
					`EthMacDest <= `EthMacSource;
					// Write our MAC address to the source
					eth_buf_data[8 * 22 +: 48] <= MAC_ADDRESS;
					// Change type to reply
					eth_buf_data[8 * 21 +: 8] <= 8'd2;
					// Copy the IP source address to the desination address
					eth_buf_data[8 * 38 +: 32] <= eth_buf_data[8 * 28 +: 32];
					// Write our IP source address
					eth_buf_data[8 * 28 +: 32] <= IP_ADDRESS;
					send_size <= 42;
					state <= sRxArpA;									
				end
			end
			sRxArpA:	// Allow time for copies
			begin
				return_state <= sSendEth;
				state <= sRxSkip;									
			end
/*	The ICMP ping request (8 bytes plus data) is:
;*		34		ICMP type: 0x08==echo request, 0x00==echo response
;*		35		ICMP code
;*		36 - 37	ICMP checksum
;*		38 - 39	ICMP identifier
;*		40 - 41	ICMP sequence number
;*		42		start of ICMP data
;****************************************************/			
			sRxIcmp:
			begin
				if (eth_buf_data[8 * 34 +: 8] != 8'h08)		// echo request
				begin
					return_state <= sIdle;			
					state <= sRxSkip;
				end
				else
				begin		// send echo reply
					eth_buf_data[8 * 34 +: 8] <= 8'h00;
					eth_buf_data[8 * 35 +: 8] <= 8'h00;					
					`EthMacDest <= `EthMacSource;	// reverse MAC and IP addresses
					`EthIpDest <= `EthIpSource;
					eth_buf_data[8 * 36 +: 16] <= 16'h00;	// zero ICMP checksum					
					checksum <= 1'd0;
					eth_buf_addr <= 6'd34;
					if (ip_in_buffer)	// Send back same data
					begin
						eth_buf_addr2 <= ip_end_byte[ETH_BUF_ABITS-1:0];
						send_size       <= ip_end_byte[ETH_BUF_ABITS-1:0];
					end
					else				// send back available data
					begin
						eth_buf_addr2 <= ETH_BUF_SIZE[ETH_BUF_ABITS-1:0];
						send_size       <= ETH_BUF_SIZE[ETH_BUF_ABITS-1:0];									
					end
					return_state <= sRxIcmpA;
					state <= sBufChecksum;																
				end
			end
			sRxIcmpA:
			begin
				eth_buf_data[8 * 36 +: 16] <= ~ checksum[15:0];
				return_state <= sSendIp;
				state <= sRxSkip;					
			end
			sRxUdp:		// We have a UDP packet ending at udp_end_byte
			begin
				if (ip_in_buffer)	// packet fits in buffer
				begin		// calculate checksum
					checksum <= 16'h1100 + eth_buf_data[8 * 38 +: 16];
					eth_buf_addr <= 26;				
					eth_buf_addr2 <= udp_end_byte[ETH_BUF_ABITS-1:0];
					return_state <= sRxUdpA;					
					state <= sBufChecksum;		
				end
				else if (udp_end_byte >= rx_status[29:16])	// packet length ????
				begin
					return_state <= sIdle;					
					state <= sRxSkip;
				end
				else	// packet is too big to fit in buffer
				begin
					if (`EthUdpDestPort == UDP_PORT_DAC && key_down)	// check UDP port
					begin	// packet is transmit audio data
						state <= sTxAudio;
						eth_buf_addr <= 6'd44;
					end
					else
					begin
						return_state <= sIdle;					
						state <= sRxSkip;
					end						
				end
			end
			sRxUdpA:
			begin
				if (checksum[15:0] == 16'hFFFF || checksum[15:0] == 16'h0000)	// checksum succeeds
					state <= sRxUdpB;			
				else				
					state <= sIdle;			
			end
			sRxUdpB:	// A short UDP packet has a valid checksum.
			begin		// Examine the port to see what to do with it.
				case (`EthUdpDestPort)
					UDP_PORT_ADC:				// ADC audio port
						state <= sRxUdpC;
					UDP_PORT_CTR:				// Control port
						state <= sRxUdpD;
					default:
						state <= sIdle;
				endcase
			end
			sRxUdpC:	// Control for sending ADC data to the PC
			begin
				state <= sIdle;
				if (udp_end_byte == 44)
				begin
					if (EthUdpData2Bytes == 16'h7272)
					begin	// "rr"  Address for sending ADC samples
						adc_eth_addr <= `EthMacSource;
						adc_ip_addr  <= `EthIpSource;
						adc_udp_port <= `EthUdpSourcePort;
					end
					if (EthUdpData2Bytes == 16'h7373)
					begin	// "ss"  Stop sending samples
						adc_udp_port <= 1'd0;
					end
				end
			end
			sRxUdpD:	// Rx and Tx control
			begin
				if (udp_end_byte == 56 && EthUdpData2Bytes == 16'h7453)
				begin	// "St" - BYTE REVERSAL
					rx_tune_phase	<= eth_buf_data[8 * 44 +: 32];
					tx_tune_phase	<= eth_buf_data[8 * 48 +: 32];
					level_data		<= eth_buf_data[8 * 52 +: 8];	// Output level
					tx_ctrl			<= eth_buf_data[8 * 53 +: 8];	// Tx control bits
					rx_ctrl			<= eth_buf_data[8 * 54 +: 8];	// Rx control
					level_nCS <= 1'd0;		// write level to the bus
					level_nWR <= 1'd0;
					bus_timer_clear <= 1'd0;
					state <= sRxUdpE;
				end
				else
				begin
					state <= sIdle;
				end
			end
			sRxUdpE:
			begin	// Generate a short delay
				if (bus_timer >= nsec(40))
				begin
					state <= sRxUdpF;
					bus_timer_clear <= 1'd1;
					level_nCS <= 1'd1;	// bus transaction finished
					level_nWR <= 1'd1;
					`EthUdpDestPort <= `EthUdpSourcePort;
					`EthMacDest <= `EthMacSource;	// reverse MAC and IP addresses
					`EthIpDest <= `EthIpSource;
				end
			end
			sRxUdpF:		// Echo the Tx/Rx control packet
			begin
				`EthUdpSourcePort <= UDP_PORT_CTR;
				send_size <= udp_end_byte[ETH_BUF_ABITS-1:0];
				state <= sSendUdp;
			end
			sSendAdc:	// Send ADC samples back to the host PC.
			begin
				if (current_mem_block == 0)
					rx_mem_read_addr <= 0;		// first block dword
				else
					rx_mem_read_addr <= MEM_DWORD_SIZE;	// second block dword
				send_is_adc <= 1'd1;
				`EthUdpDestPort <= adc_udp_port;
				`EthMacDest <= adc_eth_addr;
				`EthIpDest <= adc_ip_addr;
				eth_buf_data[8 * 42 +: 8] <= mem_sequence;	// sequence number
				mem_sequence <= mem_sequence + 1'd1;	// increment sequence number
				eth_buf_data[8 * 43 +: 8] <= key_down;	// status byte
				`EthUdpSourcePort <= UDP_PORT_ADC;
				// Size of ADC samples plus headers + sequence number in bytes
				send_size <= MEM_BYTE_SIZE + 6'd42 + 6'd2;
				state <= sSendUdp;
			end
			sSendAdcData:
			begin		// LOOP: write ADC data
				eth_write(TX_DATA, rx_mem_q);		// read ADC data from memory
				rx_mem_read_addr <= rx_mem_read_addr + 1'd1;	// add one dword
				bus_return <= sSendAdcDataA;
				state <= sEtherBus;
			end
			sSendAdcDataA:
			begin
				if (rx_mem_read_addr == MEM_DWORD_SIZE || rx_mem_read_addr == MEM_DWORD_SIZE*2)
				begin
					state <= sIdle;
				end
				else
					state <= sSendAdcData;
			end
			//sSendKeyState:	// Send key up/down state back to the host PC.
			//begin
			//	old_key_state <= key_down;
			//	`EthMacDest <= adc_eth_addr;
			//	`EthIpDest <= adc_ip_addr;
			//	`EthUdpSourcePort <= UDP_PORT_KEY;
			//	`EthUdpDestPort <= UDP_PORT_KEY;
			//	eth_buf_data[8 * 42 +: 8] <= key_down;
			//	send_size <= 6'd44;
			//	state <= sSendUdp;
			//end
			sSendIp:	// Fill in IP headers, calculate IP checksum and send packet
			begin
				`EthType <= 16'h0800;
				`EthIpVer <= 8'h45;
				`EthIpType <= 8'h0;
				`EthIpLength <= send_size - 4'd14;	// send_size must already be set
				`EthIpSeq <= ip_sequence;
				ip_sequence <= ip_sequence + 1'd1;
				`EthIpFlags <= 1'd0;
				`EthIpTtl <= 8'd20;
				// EthIpProtocol must already be set
				`EthIpChecksum <= 16'h00;	// zero IP checksum	
				`EthIpSource <= IP_ADDRESS;
				// EthIpDest must already be set
				checksum <= 1'd0;
				eth_buf_addr <= 14;
				eth_buf_addr2 <= 34;
				return_state <= sSendIpA;
				state <= sBufChecksum;																												
			end
			sSendIpA:
			begin
				`EthIpChecksum <= ~ checksum[15:0];
				state <= sSendEth;							
			end
			sSendEth:
			begin
				`EthMacSource <= MAC_ADDRESS;	// Set source MAC address
				// MAC destination and type must already be set
				eth_read(TX_FIFO_INF);
				bus_return <= sSendEthA;
				state <= sEtherBus;
				eth_buf_addr <= 4'd4;
				tempword <= send_size + 16'd8;	// TX space needed in TX buffer
			end
			sSendEthA:
			begin
				if (read_data[15:0] < tempword)
				begin		// No space; discard packet
					state <= sIdle;
					LED[4] <= 1'd1;
				end
				else	// transmit one segment
				begin	// write TX command A
					eth_write(TX_DATA, send_size | (1<<13) | (1<<12));
					bus_return <= sSendEthB;
					state <= sEtherBus;
				end
			end
			sSendEthB:
			begin		// write TX command B
				eth_write(TX_DATA, send_size);
				if (send_is_adc)
					bus_return <= sSendEthD;
				else
					bus_return <= sSendEthC;
				state <= sEtherBus;
			end
			sSendEthC:
			begin		// LOOP: write all data from eth_buf_data
				eth_write(TX_DATA, eth_buf_data[8 * eth_buf_addr - 1 -: 32]);
				state <= sEtherBus;
				if (eth_buf_addr >= send_size)
				begin
					bus_return <= sIdle;
				end
				else
				begin
					eth_buf_addr <= eth_buf_addr + 4'd4;
					bus_return <= sSendEthC;
				end
			end
			sSendEthD:
			begin		// LOOP: write headers and sequence number
				eth_write(TX_DATA, eth_buf_data[8 * eth_buf_addr - 1 -: 32]);
				state <= sEtherBus;
				if (eth_buf_addr >= 6'd44)
				begin
					bus_return <= sSendAdcData;
					send_is_adc <= 1'd0;
				end
				else
				begin
					eth_buf_addr <= eth_buf_addr + 4'd4;
					bus_return <= sSendEthD;
				end
			end
			sBufChecksum:	// Calculate the checksum of the data in eth_buf_data
			begin			// from address eth_buf_addr up to address eth_buf_addr2
				state <= sBufChecksumA;
				tempword <= eth_buf_data[8 * eth_buf_addr +: 16];
				if (eth_buf_addr2[0])	// odd byte address
					eth_buf_data[8 * eth_buf_addr2 +: 8] <= 8'd0;	// add a zero byte to end
			end
			sBufChecksumA:
			begin
				checksum <= checksum + tempword;
				eth_buf_addr <= eth_buf_addr + 4'd2;
				state <= sBufChecksumB;
			end
			sBufChecksumB:
			begin
				if (eth_buf_addr < eth_buf_addr2)
				begin	
					tempword <= eth_buf_data[8 * eth_buf_addr +: 16];
					state <= sBufChecksumA;
				end
				else
					state <= sBufChecksumC;							
			end
			sBufChecksumC:
			begin	// add all carries back into the checksum
				tempword <= checksum[31:16];
				checksum <= checksum[15:0];
				state <= sBufChecksumD;			
			end
			sBufChecksumD:
			begin
				if (tempword != 0)
				begin
					checksum <= checksum + tempword;
					state <= sBufChecksumC;
				end
				else
					state <= return_state;						
			end
			sSendUdp:
			begin	// Send UDP packet
				`EthIpProtocol <= 8'h11;	// Set IP protocol to UDP 0x11
				`EthIpSource <= IP_ADDRESS;	// Needed for checksum
				// EthIpDest must already be set		
				// EthUdpSourcePort must already be set
				// Destination port must already be set
				`EthUdpLength <= (send_size - 8'd34);	// send_size must already be set
				eth_buf_addr <= 26;		
				if (send_is_adc)
				begin
					checksum <= 16'h1100 + rx_mem_checksum;	// include checksum of data
					eth_buf_addr2 <= 6'd44;
				end
				else
				begin
					checksum <= 16'h1100;
					eth_buf_addr2 <= send_size[ETH_BUF_ABITS-1:0];
				end
				state <= sSendUdpA;			
			end
			sSendUdpA:
			begin	// Calculate UDP checksum and send
				`EthUdpChecksum <= 1'd0;			
				checksum <= checksum + eth_buf_data[8 * 38 +: 16];
				return_state <= sSendUdpC;					
				state <= sBufChecksum;		
			end
			sSendUdpC:
			begin
				state <= sSendIp;
				if (checksum[15:0] == 16'hFFFF)
					`EthUdpChecksum <= 16'hFFFF;		// change zero checksum to 0xFFFF
				else
					`EthUdpChecksum <= ~ checksum[15:0];			
			end
			sEtherBus:
			begin	// Start an Ethernet bus transaction.  The following states occur in
					// order at the indicated time in nanoseconds.
				bus_timer_clear <= 1'd0;
				second <= 1'd0;
				eth_nCS <= 1'd0;
				state <= sEtherBusA;
			end
			sEtherBusA:
			begin
				if (bus_timer >= nsec(20))	// assert read or write (eth_nRD or eth_nWR)
				begin
					state <= sEtherBusB;
					if (do_read)
						eth_nRD <= 1'd0;
					if (do_write)
						eth_nWR <= 1'd0;
				end
			end
			sEtherBusB:
			begin
				if (bus_timer >= nsec(140))	// read data from bus
				begin
					state <= sEtherBusC;
					read_data[15:0] <= eth_data_bus;
				end
			end			
			sEtherBusC:
			begin
				if (bus_timer >= nsec(180))	// de-assert read or write
				begin
					state <= sEtherBusD;
					eth_nRD <= 1'd1;
					eth_nWR <= 1'd1;
				end
			end			
			sEtherBusD:
			begin
				if (bus_timer >= nsec(200))	// add one to address for next 16-bit transfer
				begin
					state <= sEtherBusE;
					second <= 1'd1;
				end
			end
			sEtherBusE:
			begin
				if (bus_timer >= nsec(220))	// assert read or write	
				begin
					state <= sEtherBusF;
					if (do_read)
						eth_nRD <= 1'd0;
					if (do_write)
						eth_nWR <= 1'd0;
				end
			end
			sEtherBusF:
			begin
				if (bus_timer >= nsec(340))	// read data from bus
				begin
					state <= sEtherBusG;
					read_data[31:16] <= eth_data_bus;
				end
			end
			sEtherBusG:
			begin
				if (bus_timer >= nsec(380))	// de-assert read or write	
				begin
					state <= sEtherBusH;
					eth_nRD <= 1'd1;
					eth_nWR <= 1'd1;
					eth_nCS <= 1'd1;
				end
			end
			sEtherBusH:
			begin
				if (bus_timer >= nsec(400))	// end of bus read/write transaction
				begin
					state <= bus_return;
					do_read <= 1'd0;
					do_write <= 1'd0;
					second <= 1'd0;
					bus_timer_clear <= 1'd1;
				end
			end
			sDelay:
			begin	// Generate a delay
				bus_timer_clear <= 1'd0;
				state <= sDelayA;
			end
			sDelayA:
			begin
				if (bus_timer >= nsec(100000))
				begin
					state <= bus_return;
					bus_timer_clear <= 1'd1;
				end
			end
			sTxAudio:	// Copy Tx audio data from buffer to Tx memory
			begin
				tx_mem_data <= eth_buf_data[8 * eth_buf_addr +: 32];
				state <= sTxAudioA;
			end
			sTxAudioA:
			begin
				tx_mem_wen <= 1'd1;
				eth_buf_addr <= eth_buf_addr + 4'd4;
				state <= sTxAudioB;
			end
			sTxAudioB:
			begin
				tx_mem_wen <= 1'd0;
				tx_mem_write_addr <= tx_mem_write_addr + 1'd1;
				if (eth_buf_addr == ETH_BUF_SIZE)
					state <= sTxAudioC;
				else
					state <= sTxAudio;
			end
			sTxAudioC:	// Copy remaining audio data from Ethernet to Tx memory
			begin
				if (rx_count < udp_end_byte)	// packet length
				begin
					eth_read (RX_DATA);
					bus_return <= sTxAudioD;
					state <= sEtherBus;
					rx_count <= rx_count + 4'd4;	// add four bytes to count
				end
				else
				begin
					return_state <= sIdle;
					state <= sRxSkip;
				end
			end
			sTxAudioD:
			begin
				tx_mem_data <= read_data;
				state <= sTxAudioE;
			end
			sTxAudioE:
			begin
				tx_mem_wen <= 1'd1;
				state <= sTxAudioF;
			end
			sTxAudioF:
			begin
				tx_mem_wen <= 1'd0;
				tx_mem_write_addr <= tx_mem_write_addr + 1'd1;
				state <= sTxAudioC;
			end
			endcase
		end
	end
endmodule

module eth_bus_timer(	// timer for Ethernet bus transactions and delays
	input clock,
	input bus_timer_clear,
	output reg [15:0] bus_timer);
	
	always @(posedge clock)
		if (bus_timer_clear)
			bus_timer <= 1'd0;
		else
			bus_timer <= bus_timer + 1'd1;
endmodule
