// Interpolating CIC filter, order 5.
// Produce an output when clock_en is true.  Output a strobe on req
// to request an input from the next filter.

module CicInterpM5(
	input clock,
	input clock_en,				// enable an output sample
	output reg req,				// request the next input sample
	input signed [IBITS-1:0] x_real,	// input samples
	input signed [IBITS-1:0] x_imag,
	output signed [OBITS-1:0] y_real,	// output samples
	output signed [OBITS-1:0] y_imag
	);
	
	parameter RRRR = 16;		// interpolation; limited by size of counter
	parameter IBITS = 20;		// input bits
	parameter OBITS = 20;		// output bits
	parameter GBITS = 16;		//log2(RRRR ** 4);	// growth bits: growth is R**M / R
	// Note: log2() rounds UP to the next integer - Not available in Verilog!
	localparam CBITS = IBITS + GBITS;	// calculation bits

	reg [7:0] counter;		// increase for higher maximum RRRR

	reg signed [CBITS-1:0] x0, x1, x2, x3, x4, x5, dx0, dx1, dx2, dx3, dx4;		// variables for comb, real
	reg signed [CBITS-1:0] y1, y2, y3, y4, y5;	// variables for integrator, real
	reg signed [CBITS-1:0] q0, q1, q2, q3, q4, q5, dq0, dq1, dq2, dq3, dq4;		// variables for comb, imag
	reg signed [CBITS-1:0] s1, s2, s3, s4, s5;	// variables for integrator, imag

	wire signed [CBITS-1:0] sxtxr, sxtxi;
	assign sxtxr = {{(CBITS - IBITS){x_real[IBITS-1]}}, x_real};	// sign extended
	assign sxtxi = {{(CBITS - IBITS){x_imag[IBITS-1]}}, x_imag};
	assign y_real = y5[CBITS-1 -:OBITS];		// output data
	assign y_imag = s5[CBITS-1 -:OBITS];
	
	initial
	begin
		counter = 0;
		req = 0;
	end

	always @(posedge clock)
	begin
		if (clock_en)
		begin
			// (x0, q0) -> comb -> (x5, q5) -> interpolate -> integrate -> (y5, s5)
			if (counter == RRRR - 1)
			begin	// Process the sample (x0, q0) to get (x5, q5)
				counter <= 1'd0;
				x0 <= sxtxr;
				q0 <= sxtxi;
				req <= 1'd1;
				// Comb for real data
				x1 <= x0 - dx0;
				x2 <= x1 - dx1;			
				x3 <= x2 - dx2;				
				x4 <= x3 - dx3;				
				x5 <= x4 - dx4;			
				dx0 <= x0;
				dx1 <= x1;
				dx2 <= x2;
				dx3 <= x3;
				dx4 <= x4;
				// Comb for imaginary data
				q1 <= q0 - dq0;
				q2 <= q1 - dq1;
				q3 <= q2 - dq2;
				q4 <= q3 - dq3;
				q5 <= q4 - dq4;
				dq0 <= q0;
				dq1 <= q1;
				dq2 <= q2;
				dq3 <= q3;
				dq4 <= q4;
			end
			else
			begin
				counter <= counter + 1'd1;
				x5 <= 0;	// stuff a zero for (x5, q5)
				q5 <= 0;
				req <= 1'd0;
			end
			// Integrate the sample (x5, q5) to get the output (y5, s5)
			// Integrator for real data; input is x5
			y1 <= y1 + x5;
			y2 <= y2 + y1;
			y3 <= y3 + y2;
			y4 <= y4 + y3;
			y5 <= y5 + y4;
			// Integrator for imaginary data; input is q5
			s1 <= s1 + q5;
			s2 <= s2 + s1;
			s3 <= s3 + s2;
			s4 <= s4 + s3;
			s5 <= s5 + s4;
		end
		else
		begin
			req <= 1'd0;
		end
	end
endmodule

// Decimating CIC filter, order 5.
// Strobe x_avail when an input is available.
// Strobe out_avail when an output is available.
module CicDecimM5(
	// parameterized CIC filter order M = 5 = delay; redefine RRRR etc.
	input clock,
	input x_avail,
	input signed [IBITS-1:0] x_real,
	input signed [IBITS-1:0] x_imag,
	output reg out_avail,
	output signed [OBITS-1:0] y_real,
	output signed [OBITS-1:0] y_imag);

	parameter RRRR = 16;		// decimation; limited by size of counter
	parameter IBITS = 18;		// input bits
	parameter OBITS = 18;		// output bits
	parameter GBITS = "None";	//log2(RRRR ** 5);	// growth bits: growth is R**M
	// Note: log2() rounds UP to the next integer - Not available in Verilog!
	localparam CBITS = IBITS + GBITS;	// calculation bits

	reg [7:0] counter;		// increase for higher maximum RRRR
	reg  signed [IBITS-1:0] xr, xi;	// registered input data
	wire signed [CBITS-1:0] sxtxr, sxtxi;	// sign extended registered input data
	reg  signed [CBITS-1:0] Ri0, Ri1, Ri2, Ri3, Ri4, Ii0, Ii1, Ii2, Ii3, Ii4;
	reg  signed [CBITS-1:0] Rc0, Rc1, Rc2, Rc3, Rc4, Rc5, Ic0, Ic1, Ic2, Ic3, Ic4, Ic5;
	reg  signed [CBITS-1:0] Rc0d, Rc1d, Rc2d, Rc3d, Rc4d, Ic0d, Ic1d, Ic2d, Ic3d, Ic4d;
	
	assign sxtxr = {{(CBITS - IBITS){xr[IBITS-1]}}, xr};
	assign sxtxi = {{(CBITS - IBITS){xi[IBITS-1]}}, xi};
	assign y_real = Rc5[CBITS-1 -:OBITS];		// output data
	assign y_imag = Ic5[CBITS-1 -:OBITS];

	initial
	begin
		counter = 0;
		out_avail = 0;
	end
	
	always @(posedge clock)
	begin
		if (x_avail)
		begin
			// Integrator: integrate every input sample
			xr <= x_real;
			Ri0 <= Ri0 + sxtxr;
			Ri1 <= Ri1 + Ri0;
			Ri2 <= Ri2 + Ri1;
			Ri3 <= Ri3 + Ri2;
			Ri4 <= Ri4 + Ri3;
			xi <= x_imag;
			Ii0 <= Ii0 + sxtxi;
			Ii1 <= Ii1 + Ii0;
			Ii2 <= Ii2 + Ii1;
			Ii3 <= Ii3 + Ii2;
			Ii4 <= Ii4 + Ii3;
			// increment counter
			if (counter == RRRR - 1)
			begin
				counter <= 0;
				out_avail <= 1'd1;
				// Comb: output every RRRR'th sample
				Rc0   <= Ri4;	// Ri4 from integrator
				Rc0d <= Rc0;
				Rc1   <= Rc0 - Rc0d;
				Rc1d <= Rc1;
				Rc2   <= Rc1 - Rc1d;
				Rc2d <= Rc2;
				Rc3   <= Rc2 - Rc2d;
				Rc3d <= Rc3;
				Rc4   <= Rc3 - Rc3d;
				Rc4d <= Rc4;
				Rc5   <= Rc4 - Rc4d;
				//Rc5d <= Rc5;
				Ic0   <= Ii4;	// Ii4 from integrator
				Ic0d <= Ic0;
				Ic1   <= Ic0 - Ic0d;
				Ic1d <= Ic1;
				Ic2   <= Ic1 - Ic1d;
				Ic2d <= Ic2;
				Ic3   <= Ic2 - Ic2d;
				Ic3d <= Ic3;
				Ic4   <= Ic3 - Ic3d;
				Ic4d <= Ic4;
				Ic5   <= Ic4 - Ic4d;
				//Ic5d <= Ic5;
			end
			else
			begin
				counter <= counter + 1'd1;
				out_avail <= 1'd0;
			end
		end
		else
			out_avail <= 1'd0;
	end
endmodule

// Decimating CIC filter, order 5.
// Strobe x_avail when an input is available.
// Strobe out_avail when an output is available.
// Decimation is from 2 through 40; decimation less 1 is Rminus1
module CicVarDecimM5(
	input [5:0] Rminus1,
	input clock,
	input x_avail,
	input signed [IBITS-1:0] x_real,
	input signed [IBITS-1:0] x_imag,
	output reg out_avail,
	output signed [OBITS-1:0] y_real,
	output signed [OBITS-1:0] y_imag);

	parameter IBITS = 18;		// input bits
	parameter OBITS = 18;		// output bits
	// growth bits: growth is R**M
	// Note: log2() rounds UP to the next integer - Not available in Verilog!
	localparam CBITS = 45;	// calculation bits

	reg [5:0] counter;
	reg signed [IBITS-1:0] xr, xi;	// registered input data
	reg signed [CBITS-1:0] xrr, xrrr, xii, xiii;
	reg signed [CBITS-1:0] Ri0, Ri1, Ri2, Ri3, Ri4, Ii0, Ii1, Ii2, Ii3, Ii4;
	reg signed [CBITS-1:0] Rc0, Rc1, Rc2, Rc3, Rc4, Rc5, Ic0, Ic1, Ic2, Ic3, Ic4, Ic5;
	reg signed [CBITS-1:0] Rc0d, Rc1d, Rc2d, Rc3d, Rc4d, Ic0d, Ic1d, Ic2d, Ic3d, Ic4d;
		
	assign y_real = Rc5[CBITS-1 -:OBITS];		// output data
	assign y_imag = Ic5[CBITS-1 -:OBITS];

	initial
	begin
		counter = 0;
		out_avail = 0;
	end
	
	reg [4:0] shift;	// number of bits to shift left
	always @(Rminus1)
		case (Rminus1)
		1:			shift = 5'd22;	// Decim  2, growth bits 5
		3:			shift = 5'd17;	// Decim  4, growth bits 10
		4:			shift = 5'd15;	// Decim  5, growth bits 12
		7:			shift = 5'd12;	// Decim  8, growth bits 15
		9:			shift = 5'd10;	// Decim  10, growth bits 17
		19:			shift = 5'd5;	// Decim  20, growth bits 22
		default:	shift = 5'd0;	// Decim  40, growth bits 27
		endcase

	always @(posedge clock)
	begin
		if (x_avail)
		begin
			// Integrator: integrate every input sample
			xr <= x_real;
			xrr <= {{(CBITS - IBITS){xr[IBITS-1]}}, xr};
			xrrr <= xrr << shift;
			Ri0 <= Ri0 + xrrr;
			Ri1 <= Ri1 + Ri0;
			Ri2 <= Ri2 + Ri1;
			Ri3 <= Ri3 + Ri2;
			Ri4 <= Ri4 + Ri3;
			xi <= x_imag;
			xii <= {{(CBITS - IBITS){xi[IBITS-1]}}, xi};
			xiii <= xii << shift;
			Ii0 <= Ii0 + xiii;
			Ii1 <= Ii1 + Ii0;
			Ii2 <= Ii2 + Ii1;
			Ii3 <= Ii3 + Ii2;
			Ii4 <= Ii4 + Ii3;
			// increment counter
			if (counter == Rminus1)
			begin
				counter <= 0;
				out_avail <= 1'd1;
				// Comb: output every RRRR'th sample
				Rc0   <= Ri4;	// Ri4 from integrator
				Rc0d <= Rc0;
				Rc1   <= Rc0 - Rc0d;
				Rc1d <= Rc1;
				Rc2   <= Rc1 - Rc1d;
				Rc2d <= Rc2;
				Rc3   <= Rc2 - Rc2d;
				Rc3d <= Rc3;
				Rc4   <= Rc3 - Rc3d;
				Rc4d <= Rc4;
				Rc5   <= Rc4 - Rc4d;
				//Rc5d <= Rc5;
				Ic0   <= Ii4;	// Ii4 from integrator
				Ic0d <= Ic0;
				Ic1   <= Ic0 - Ic0d;
				Ic1d <= Ic1;
				Ic2   <= Ic1 - Ic1d;
				Ic2d <= Ic2;
				Ic3   <= Ic2 - Ic2d;
				Ic3d <= Ic3;
				Ic4   <= Ic3 - Ic3d;
				Ic4d <= Ic4;
				Ic5   <= Ic4 - Ic4d;
				//Ic5d <= Ic5;
			end
			else
			begin
				counter <= counter + 1'd1;
				out_avail <= 1'd0;
			end
		end
		else
			out_avail <= 1'd0;
	end
endmodule
