;******************************************************************************************************
;   Battery_Cycler, By Bertrand Zauhar, VE2ZAZ
;   For more details, visit VE2ZAZ's website at:   
;        http://ve2zaz.net                                                                      
;******************************************************************************************************
;    Filename:      Bat_Cycler.asm                                            
;    Date:          15/12/2007                                                   
;    Software Version:  1                                                   
;
;    Designed to run on PICmicro PIC18F1220.
;                                                                     
;    Author: Bertrand Zauhar                                                          
;******************************************************************************************************
;    Files required:         P18F1220.INC                                     
;    Assembles using Microchip's MPASM assembler software
;******************************************************************************************************

    LIST P=18F1220        ;directive to define processor
    #include <P18F1220.INC>    ;processor specific variable definitions

;******************************************************************************************************
;Configuration bits
; The CONFIG directive defines configuration data within the .ASM file.
; The labels following the directive are defined in the P18F1220.INC file.
; The PIC18F1220/1320 Data Sheet explains the functions of the configuration bits.

;CONFIG1H
    CONFIG OSC = INTIO2, FSCM = OFF, IESO = OFF
;CONFIG2L
    CONFIG PWRT = ON, BOR = ON, BORV = 27 
;CONFIG2H
    CONFIG WDT = OFF
;CONFIG3H
    CONFIG MCLRE = ON
;CONFIG4L
    CONFIG DEBUG = OFF, LVP = OFF, STVR = OFF 
;CONFIG5L
    CONFIG CP0 = OFF, CP1 = OFF
;CONFIG5H
    CONFIG CPB = OFF, CPD = OFF
;CONFIG6L
    CONFIG WRT0 = OFF, WRT1 = OFF
;CONFIG6H
    CONFIG WRTB = OFF, WRTC = OFF, WRTD = OFF 
;CONFIG7L
    CONFIG EBTR0 = OFF, EBTR1 = OFF 
;CONFIG7H
    CONFIG EBTRB = OFF

;******************************************************************************************************
;Bank 0 has 256 locations available.

    CBLOCK    0x008         ;All of these variables are located in the RAM Bank 0.
                            ;Starts at 0x008 since debugger shows conflict of memory labels from 0x000 to 0x007

    WREG_TEMP               ;0_variable used for context saving when interrupt occurs
    STATUS_TEMP             ;1_variable used for context saving when interrupt occurs
    BSR_TEMP                ;2_variable used for context saving when interrupt occurs

                            ;These are the variables that hold the ADC values for subsequent analysis
    RES_VOLT_HI             ;3_Corresponds to ADC Channel 0, high byte
    RES_VOLT_LO             ;4_Corresponds to ADC Channel 0, low byte

    BAT_VOLT_HI             ;5_Corresponds to ADC Channel 1, high byte
    BAT_VOLT_LO             ;6_Corresponds to ADC Channel 1, low byte

    PUSHBUTT_STAT           ;7_This register is used to treat the various states and actions the pushbutton may lead to.
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_Push button debounced state flag. 1 = released, 0 = activated.
                            ; 4_Not used
                            ; 3_Not used
                            ; 2_System in action delay following a release of the push button (3-second delay).
                            ; 1_Push button old debounced state. Used to detect new depress of pushbutton for action update
                            ; 0_Push button in debouncing stage flag.

	AD_STAT					;8_This register is used to control the ADC process
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_Not used
                            ; 4_Not used
                            ; 3_Not used
                            ; 2_Not used
                            ; 1_Not used
                            ; 0_New ADC results available flag.

    PWM_STAT             	;9_Register used to control the PWM generated charge or discharge current
                            ; 7_Not used
                            ; 6_decrease charge/discharge current flag
                            ; 5_increase charge/discharge current flag
                            ; 4_Not used 
                            ; 3_Not used
                            ; 2_Not used
                            ; 1_Not used
                            ; 0_pwm railed at max value flag

    STAGE_REQ_STAT			;10_This register is used to flag various stage requests (while LED is flashing)
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_Idle stage request
                            ; 4_Battery #2 charge request
                            ; 3_Battery #2 discharge request
                            ; 2_Battery #1 charge request
                            ; 1_Battery #1 discharge request
                            ; 0_Entire Cycle OFF request

    CLOCK_REG_U             ;11_This is th clock register. 24 bits are required to count up to a month or even more
    CLOCK_REG_H             ;12 "
    CLOCK_REG_L             ;13_ "

    SAVED_CLOCK_VAL_U       ;14_This is a 24 bit register used to save the clock value when a new charge or 
    SAVED_CLOCK_VAL_H       ;15 discharge stage begins. 
    SAVED_CLOCK_VAL_L       ;16_

    BUTT_DEBOUNCING_CTR     ;17_Counter used for decbouncing the push button

	ACTION_DELAY_CTR		;18_Counter used to check for action delay expiry.

    LED_FLASH_CTR           ;19_Counter used to make LEDs flash

    CONDIT_CYCLE_CTR        ;20_Counter used to limit the number of conditioning cycles performed

	PAUSE_OUTS_OFF_CTR		;21_Counter used between two stages to pause with the outputs off

    IDLE_VOLT_CHK_CTR       ;22_Used to let the output settle down before making an ADC measurement during Idle stage.

    SERIAL_STAT             ;23_This is the USART status flag register    
                            ; 7-Not used
                            ; 6-send_current battery voltage and current values
                            ; 5-Send current clock request
                            ; 4-Send Log info request
                            ; 3-tx cycle being performed.
                            ; 2-Request to transmit parameter information
                            ; 1-LF character received (end of user command entry)
                            ; 0-New character transmission cycle required

    PWM_DUTY_L              ;24_PWM duty cycle register, lower 8 bits            
    PWM_DUTY_H              ;25_PWM duty cycle output register, higher 2 bits

    TEMP1_UP                ;26_Temporary 24-bit variable available throughout code execution.
    TEMP1_HI                ;27 "
    TEMP1_LO                ;28 "

    TEMP2_HI                ;29_Temporary 16-bit variable available throughout code execution.
    TEMP2_LO                ;30_ "

    TEMP3_HI                ;31_Temporary 16-bit variable available throughout code execution.
    TEMP3_LO                ;32_ "

    WRT_EEPROM_POS          ;33_EEPROM Writing Counter. Indicates what parameter to write in EEPROM

    SERIAL_TX_INT_FSR0L_SAVE;34_Register that allow to save FSR0L value after a TX interrupt was treated

    SERIAL_RX_INT_FSR0L_SAVE;35_Register that allow to save FSR0L value after a RX interrupt was treated

    EEPROM_WRITE_FSR1L_SAVE ;36_Register that allow to save FSR0L value while the EEPROM routine is waiting for a write to complete

    GENERAL1_STAT           ;37_Melting pot register
                            ; 7-Not used
                            ; 6-Not used
                            ; 5-Not used
                            ; 4-Not used
                            ; 3-Not used
                            ; 2-Flag to check battery inversion during charge cycle.
                            ; 1-LED flashing alternate. Used to track if the LEDs are on or off due to flashing
                            ; 0-New EEprom write cycle is required request

	M1_H					;38_Temporary variables used to execute the 16bit x 8 bit multiply routine
	M1_L					;39_ "
	M2_L					;40_ "
	R1_U					;41_ "
	R1_H					;42_ "
	R1_L					;43_ "

    CYCLE_CLOCK_LIM_U       ;44_This is the complete cycle time limit as sent by the user. 24 bits. Typically corresponds to a couple of weeks.
    CYCLE_CLOCK_LIM_H       ;45_ "
    CYCLE_CLOCK_LIM_L       ;46_ "

    DISCHRG_LIM1_HI         ;47_This is the voltage discharge limit to be compared to the ADC reading when dischaging.
    DISCHRG_LIM1_LO         ;48_ "
	DISCHRG_LIM2_HI         ;49_ "
    DISCHRG_LIM2_LO         ;50_ "
                            
    TARGET_CHRG_DROP1_H		;51_Voltage drop in 10 ohm resistor corresponding to the optimum C/10 charging 
    TARGET_CHRG_DROP1_L     ;52_ current (RES_VOLT - BAT_VOLT)
    TARGET_CHRG_DROP2_H		;53_ "
    TARGET_CHRG_DROP2_L    	;54_ "

    TARGET_DISCHRG_DROP1_H	;55_Voltage drop in 10 ohm resistor corresponding to the requested discharging 
    TARGET_DISCHRG_DROP1_L  ;56; current (BAT_VOLT - RES_VOLT)
    TARGET_DISCHRG_DROP2_H	;57_ "
    TARGET_DISCHRG_DROP2_L  ;58_ "

    CLOCK_CHARGE_LIM1_H     ;59_This is the Charge time limit as sent by the user. 16 bits. Typically corresponds to 12 hours.
    CLOCK_CHARGE_LIM1_L     ;60_ "
    CLOCK_CHARGE_LIM2_H     ;61_ "
    CLOCK_CHARGE_LIM2_L     ;62_ "

    CLOCK_DISCHG_LIM1_H     ;63_This is the Charge time limit as sent by the user. 16 bits. Typically corresponds to 12 hours.
    CLOCK_DISCHG_LIM1_L     ;64_ "
    CLOCK_DISCHG_LIM2_H     ;65_ "
    CLOCK_DISCHG_LIM2_L     ;66_ "

    LOG_BUFF_FSR0L_SAVE 	;67_Register that allow to save/restore FSR0L when working in the Interrupt routine.

;=  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
;This is a group of consecutive registers that get uploaded/downloaded via the serial port

	
                                ;Leave CYCLE_STAT stat as the first parameter of the list. It is referenced to elsewhere in the code
    CYCLE_STAT				;68_This register is used to flag various conditions
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_battery #1 or #2 being checked for voltage in Idle stage. 0=bat #1, 1=bat #2.
                            ; 4_idle voltage monitoring routine is currently executing measurements
                            ; 3_120 second flag to tell idle monitoring routine to check for battery voltage
                            ; 2_Pause insertion between two stages with outputs off is active
                            ; 1_Number of batteries connected: 0 = one battery, 1 = two batteries
                            ; 0_Battery Conditioning mode: 0 = off, 1 = on.

   ALARM_STAT1              ;69_This first alarm latching register flags any abnormal condition detected in the cycles.
                            ; 7_Battery #2: undervoltage (failed to reach minimum charged-up voltage at end of charge stage)
                            ; 6_Battery #1: undervoltage (failed to reach minimum charged-up voltage at end of charge stage)
                            ; 5_Battery #2: no charge current or too small charge current detected
                            ; 4_Battery #1: no charge current or too small charge current detected
                            ; 3_Battery #2: discharge timeout (too much time taken to discharge)
                            ; 2_Battery #1: discharge timeout (too much time taken to discharge)
                            ; 1_Battery #2: no discharge current or too small discharge current detected (could be caused by completely dead battery)
                            ; 0_Battery #1: no discharge current or too small discharge current detected (could be caused by completely dead battery)

	CYCLE_DURATION			;70_This variable corresponds to the number of days of duration of the whole cycle

	BAT1_CHRG_30M_SLICE_INDEX	;71_These variables correnspond to the number of 30 minute slices the charging cycles last for.
	BAT2_CHRG_30M_SLICE_INDEX	;72_ "

	BAT1_CHRG_CURR_INDEX	;73_This is the index used in the lookup table to figure the target charging current for the batteries
	BAT2_CHRG_CURR_INDEX	;74 "

	BAT1_DISCHRG_CURR_INDEX	;75_This is the index used in the lookup table to figure the target discharging current for the batteries
	BAT2_DISCHRG_CURR_INDEX	;76_ "

	BAT1_DISCHRG_VOLT_INDEX ;77_This is the index used in the lookup table to figure the threshold discharged voltage for the batteries
	BAT2_DISCHRG_VOLT_INDEX	;78_ "

    MAX_CONDIT_CYCLES       ;79_This is the number of conditioning cycles  to be performed (provided by the user).

    BAT1_CAPACITY           ;80_This is used to store the Battery Capacity (mAh). Not used in firmware but used to store and recall values in the Windows Program
    BAT2_CAPACITY           ;81_ " 

    BAT1_NUM_CELLS          ;82_This is the number of cells in the battery. Not used in firmware but used to store and recall values in the Windows Program
    BAT2_NUM_CELLS          ;83_ "

	DEFAULTS_CHECKED		;84_This is used to store the fact that the Windows tool has its default parameters checkmark enabled or not
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_Not used 
                            ; 4_Not used 
                            ; 3_Not used
                            ; 2_Not used
                            ; 1_Battery #2 Default checkmark checked (1) or unchecked (0).
                            ; 0_Battery #1 Default checkmark checked (1) or unchecked (0).

    BAT1_MAX_DISCH_HOUR_LIM ;85_These variables represent the maximum number of hours that the discharge stage should last before aborting the stage.
    BAT2_MAX_DISCH_HOUR_LIM ;86_ "

    STAGE_STAT				;87_This register is used to flag various stages the cycle is in. This parameter does not get sent or updated over Serial Port
                            ; 7_Not used
                            ; 6_Not used
                            ; 5_Idle stage 
                            ; 4_Battery #2 charge 
                            ; 3_Battery #2 discharge 
                            ; 2_Battery #1 charge 
                            ; 1_Battery #1 discharge 
                            ; 0_Entire Cycle OFF 

                            ;Attention! RAM commence a 0x008 a cause du bug de MPLAB IDE. +8 au chiffre plus haut!

;=  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
            
    ENDC                    ;########### Maximum 112 variables in the block above! ############.

    CBLOCK    0x068
    ALARM_BUFF              ;Alarm Buffer. Entries match the Log entries
    ENDC

    CBLOCK    0x070
    LOG_BUFF                ;Activity Log Buffer. 48 positions reserved. Cannot be made bigger since Serial Tx Buffer is also 48 bytes long.
    ENDC

    CBLOCK    0x0A0
    SERIAL_RX_BUFF          ;Serial USART Receive Buffer memory. 48 positions reserved.
    ENDC

    CBLOCK    0x0D0
    SERIAL_TX_BUFF          ;Serial USART Transmit Buffer memory. 48 positions reserved
    ENDC


;******************************************************************************************************
;Constant definitions

    ;Bit Positions
lf_eoc_received             EQU .1                  ;position of the new cycle of character reception is required
send_parameters             EQU .2                  ;position of the bit to request sending settings information
tx_cycle_on                 EQU .3                  ;position of the tx cycle being performed flag
send_log_req                EQU .4                  ;bit position of SERIAL_STAT to flag a request for Log transmission
send_current_clock			EQU .5					;bit position of SERIAL_STAT to flag a request for current clock transmission 
send_current_v_i  			EQU .6					;bit position of SERIAL_STAT to flag a request for current battery voltage and current transmission 
write_cycle_req             EQU .0                  ;bit position of GENERAL1_STAT for flagging when a parameter Write in EEPROM is req'd
new_adc_results				EQU .0					;bit position of AD_STAT to flag when a new series of ADC readings is available.
inc_current					EQU .5					;bit position of CHARGE_STAT to flag that an increase of charge current is required					
dec_current					EQU .6					;bit position of CHARGE_STAT to flag that a decrease of charge current is required
push_butt_action_delay      EQU .2                  ;bit position of PUSHBUTT_STAT to flag that a push button action delay is counting after a button release
pushbutt_debounced          EQU .5                  ;bit position of PUSHBUTT_STAT to indicate the debounced state of the pushbutton
pushbutt_in_debouncing      EQU .0                  ;bit position of PUSHBUTT_STAT to indicate that the pushbutton is being debounced
pushbutt_old_debounced		EQU .1					;bit position of PUSHBUTT_STAT to indicate the old pushbutton state for transition detection purpose
num_of_batteries			EQU .1					;bit position of CYCLE_STAT to indicate how many batteries, 1 or 2, are provisioned on the system			
conditioning_mode			EQU .0					;bit position of CYCLE_STAT to indicate that the conditioning mode is enabled on the system			
cycle_off_bit   			EQU .0                  ;bit position of STAGE_REQ_STAT to indicate that the off stage is requested.
bat1_disch_bit  			EQU .1					;bit position of STAGE_REQ_STAT to indicate that a battery #1 discharge cycle is requested.
bat1_chrg_bit   			EQU .2					;bit position of STAGE_REQ_STAT to indicate that a battery #1 charge cycle is requested.				
bat2_disch_bit  			EQU .3					;bit position of STAGE_REQ_STAT to indicate that a battery #2 discharge cycle is requested.
bat2_chrg_bit   			EQU .4					;bit position of STAGE_REQ_STAT to indicate that a battery #2 charge cycle is requested.
idle_bit				    EQU .5					;bit position of STAGE_REQ_STAT to indicate that an idle stage is requested
led_red_anode               EQU .7                  ;bit position on PORTA and PORTB for both LEDs red anode
led_green_anode             EQU .6                  ;bit position on PORTA and PORTB for both LEDs green anode
pause_off_active			EQU	.2					;bit position of CYCLE_STAT that flags that a pause with all outputs off is currently active
check_idle_volt_req         EQU .3                  ;bit position of CYCLE_STAT that flags when a check of the batteries' idle voltage is requested
checking_idle_volt          EQU .4                  ;bit position of CYCLE_STAT that indicates that the idle voltage check process is happening.
check_idle_volt_bat_num     EQU .5                  ;bit position of CYCLE_STAT that indicates which battery is being checked for idle voltage.
pwm_rail_max                EQU .0                  ;bit position of PWM_STAT that flags when the PWM has reached its maximum value
leds_flashing               EQU .1                  ;bit position of GENERAL1_STAT that flags when the LEDs are in flashing (off) state (1) or normal lit state (0)
check_bat_invert_now        EQU .2                  ;bit position of GENERAL1_STAT that flags when to check for an inverted battery during the charge cycle.
bat1_mosfet_pin             EQU .3                  ;pin position of TRISA to test if Battery #1 Mosfet is on.
bat2_mosfet_pin             EQU .0                  ;pin position of TRISB to test if Battery #2 Mosfet is on.

    ;Bit Masks
mask_duty_1con              EQU b'00110000'         ;mask used to isolate the two MSb's of the PWM duty cycle
mask_duty_l                 EQU b'00000011'         ;mask used to isolate the two MSB's of the PWM duty cycle in the PWM_DUTY_H register
new_tx_cycle                EQU b'00000001'         ;mask to test if new cycle of character transmission is required
pushbutt_pin_pos            EQU b'00100000'         ;mask to locate the push button on port B
pushbutt_pin_mask           EQU b'11011111'
stage_request_bits_mask		EQU b'00111111'			;mask to test if the rollover of the STAGE_REQ_STAT register is required.
led_red_pin_mask            EQU b'10000000'         ;mask used on PORTA and PORTB to isolate the status of the LEDs red anode
discharge_bit_mask          EQU b'00001010'         ;This mask is used to test for any discharge stage
charge_bit_mask             EQU b'00010100'         ;This mask is used to test for any charge stage
bat1_chg_dischg_mask        EQU b'00000110'         ;This mask is used to test if battery #1 is being charged or discharged
bat1_alm_mask               EQU b'00001111'         ;This mask is used to check if battery #1 has an alarm raised
bat2_alm_mask               EQU b'11110000'         ;This mask is used to check if battery #2 has an alarm raised
bat1_bat2_chg_disch_mask    EQU b'00011110'         ;This mask is used to test if any of the batteries are being charged or discharged

    ;Initialization Values
;W                          EQU .0                  ;Is the working Register flag in various commands (defined in .INC file)
F                           EQU .1                  ;File register flag in various commands
ccp1_init                   EQU b'00001100'         ;used to initialize CPP1 in PWM mode
init_pwm_duty_h             EQU 0x40                ;used to initialize the PWM to 25% of full range to save time.
pwm_period_init             EQU 0xFF                ;used to set PWM period
eecon1_data_eeprom_access   EQU b'00000000'         ;will initialize EECON1 register for a Data EEPROM access.
eecon1_prog_ee_access       EQU b'00000000'         ;will initialize EECON1 register for a Data EEPROM access.
ad_delay_timer_val_h        EQU 0xFB                ;used to load Timer3 for a 300usec delay between 2 ADC readings
ad_delay_timer_val_l        EQU 0x00                ; "
t0_adjust_val_h             EQU 0x1B                ;This value is added to T0 after rollover so that the rollover happens every 120 seconds
t0_adjust_val_l             EQU 0x1E                ; "
t1_adjust_val_h             EQU 0xE7                ;This value is added to T1 after rollover so that the rollover happens every 50 milliseconds
t1_adjust_val_l             EQU 0x96                ; "
butt_debouncing_limit       EQU 0x02                ;push button debouncing limit 0.05s per increment 
action_delay_limit			EQU .60					;3 Second delay used for pushbutton action delay counting
led_flash_limit             EQU .4                  ;200ms LED flashing rate
led_alm_flash_limit         EQU .2                  ;100ms LED Flashing rate
pause_outs_limit			EQU .40					;2 second pause with PWM outputs off between stages
pause_outs_all_off_limit	EQU .5                  ;250ms pause with all outputs (PWM, Solid State relays, etc) off.
clock_discharge_limit_h     EQU 0x01                ;Must set this value to something like 12 hours...
clock_discharge_limit_l     EQU 0x68                ; "
firmware_version            EQU .1                  ;Also update the Flash table to correspond to the firmware version
one_day_equiv_coeff_h		EQU 0x02				;This is the 16-bit value corresponding to the clock counting for one day
one_day_equiv_coeff_l		EQU 0xD0
one_hour_equiv_coeff        EQU .30
thirty_min_equiv_coeff		EQU .15

number_param_to_stor        EQU .20                 ;This is the number of parameters that are saved/restored in Data EEPROM, incl. addr. 0.                                                    
number_param_to_xfer        EQU .19                 ;This is the number of parameters that are transferred over the RS232.                                                    

log_buff_size               EQU .21                 ;42 characters + "L" + <CR><LF> + "0x00" makes 46, so 21 is the max number of log entries that the Serial TX buffer can take.
max_idle_volt_check_delay   EQU .3                  ;Number of 50ms delays used before reading the ADC value.
min_res_volt_val_h			EQU	0x00				;The minimum RES_VOLT / 4 value that the opamps can carry without railing
min_res_volt_val_l			EQU	0x10  				;Currently set to 0.3V (this assumes a Rail-to-Rail opamp.

;List of Log Messages

    ;Simple messages (with appended clock timestamp)
bat1_dischrg_start          EQU .1                  ;DISCHARGING Battery #1 started.     
bat1_chrg_start             EQU .2                  ;CHARGING Battery #1 started.
bat2_dischrg_start          EQU .3                  ;DISCHARGING Battery #2 started.
bat2_chrg_start             EQU .4                  ;CHARGING Battery #2 started.
idle_start                  EQU .5                  ;IDLE period started.
off_start                   EQU .6                  ;OFF period started.
end_cycle					EQU .7					;End of cycle log

;List of Alarm Flags in ALARM1_STAT

alm_bat1_no_dischrg_curr    EQU .0                  ;Battery #1, Unable to generate the specified discharge current.
alm_bat1_dischrg_timeout    EQU .1                  ;Battery #1, Maximum allowed discharge duration reached.
alm_bat1_no_chrg_curr       EQU .2                  ;Battery #1, Unable to generate the specified charge current.
alm_bat1_undervolt_end      EQU .3                  ;Battery #1, Insufficient battery voltage during idle stage.
alm_bat2_no_dischrg_curr    EQU .4                  ;Battery #2, Unable to generate the specified discharge current.
alm_bat2_dischrg_timeout    EQU .5                  ;Battery #2, Maximum allowed discharge duration reached.
alm_bat2_no_chrg_curr       EQU .6                  ;Battery #2, Unable to generate the specified charge current.
alm_bat2_undervolt_end      EQU .7                  ;Battery #2, Insufficient battery voltage during idle stage.

;******************************************************************************************************
;Reset vector
; This code will start executing when a reset occurs.

        ORG    0x0000
        GOTO MAIN                                   ;go to start of main code

;******************************************************************************************************
;High priority interrupt vector
; This code will start executing when a high priority interrupt occurs or
; when any interrupt occurs if interrupt priorities are not enabled.

        ORG    0x0008
        BRA    HIGHINT                              ;go to high priority interrupt routine

;******************************************************************************************************
;LOWINT
;Low priority interrupt vector and routine
; This code will start executing when a low priority interrupt occurs.

        ORG    0x0018
LOWINT
        MOVFF STATUS,STATUS_TEMP                    ;save STATUS register
        MOVFF WREG,WREG_TEMP                        ;save working register
        MOVFF BSR,BSR_TEMP                          ;save BSR register
SERIAL_INT_CHECK
		MOVFF FSR0L,LOG_BUFF_FSR0L_SAVE				;Save the current FSR0L register value.
        BTFSS PIE1,TXIE                             ;Is TX interrupt enabled?
        BRA SERIAL_RX_CHECK                         ;No, go to test RX
        BTFSC PIR1,TXIF                             ;Yes, Did USART TX cause interrupt?
        BRA USART_TX_INT                            ;Yes, service it.
SERIAL_RX_CHECK
        BTFSS PIE1,RCIE                             ;Is RX interrupt enabled?
        BRA NO_VALID_INT                            ;No, go to other test
        BTFSC PIR1,RCIF                             ;Yes, Did USART RX cause interrupt?
        BRA USART_RX_INT                            ;Yes, service it.
NO_VALID_INT
        BRA LOWINT_EXIT                             ;no, branch to leave interrupt routine
USART_TX_INT
        BTFSS SERIAL_STAT,new_tx_cycle              ;verify if new character tx cycle initiated
        BRA NO_NU_CYCLE                             ;no, go to next validation
        LFSR FSR0,SERIAL_TX_BUFF                    ;load Serial Tx buffer's base address in FSR register
        BCF SERIAL_STAT,new_tx_cycle                ;yes, clear new Tx cycle flag
        BRA NU_TX_CYCLE								;branch to process a new Tx cycle
NO_NU_CYCLE
        MOVFF SERIAL_TX_INT_FSR0L_SAVE,FSR0L        ;Save TX FSR0L indirect addressing register value
NU_TX_CYCLE
        TSTFSZ INDF0                                ;is there a character in the buffer?
        BRA CHAR_PRESENT                            ;yes, go load it in tx buffer
        BCF SERIAL_STAT,tx_cycle_on					;nom clear the tx_cycle_on flag
        BCF PIE1,TXIE                               ;disable future TX interrupts 
        BCF PIR1,TXIF                               ;Clear interrupt request Flag.
        BRA LOWINT_EXIT_TX                          ;go to end of ISR, restore context, return        
CHAR_PRESENT
        MOVFF INDF0,TXREG                           ;transfer (indirect addressing) character to Tx register 
        CLRF POSTINC0                               ;clear character and increase indirect pointer
        BCF PIR1,TXIF                               ;Clear interrupt request Flag.
        BRA LOWINT_EXIT_TX                          ;go to end of ISR, restore context, return
USART_RX_INT    
        BCF PIR1,RCIF                               ;clear Rx char. available interrupt flag
        MOVLW 06h                                   ;Mask out unwanted bits
        ANDWF RCSTA,W                               ;Check for errors
        BTFSS STATUS,Z                              ;Was either error status bit set?
        BRA RXERROR                                 ;Found error, flag it
        TSTFSZ SERIAL_RX_BUFF                       ;is there a character in the first rx buffer cell?
        BRA NO_NU_RX_CYCLE                          ;yes, go read next character
        LFSR FSR0,SERIAL_RX_BUFF                    ;no, load Serial rx buffer's base address in FSR register
        BRA NU_RX_CYCLE								;Branch to process a new Rx cycle
NO_NU_RX_CYCLE
        MOVFF SERIAL_RX_INT_FSR0L_SAVE,FSR0L        ;Save RX FSR0L indirect addressing register value
NU_RX_CYCLE
        MOVFF RCREG,POSTINC0                        ;Get rx character from USART and increment indirect addr. pointer
        MOVLW low SERIAL_TX_BUFF-.1                 ;Has end of Rx buffer been reached (address 0x17F)?
        CPFSLT FSR0L                                ;"
        BRA END_RX_CYCLE                            ;Yes, go end Rx cycle
        MOVLW .10                                   ;No, Load LF char in Wreg
        CPFSEQ RCREG                                ;Is character received LF
        BRA LOWINT_EXIT_RX                          ;No, go to end of ISR, restore context, return
END_RX_CYCLE
        BSF SERIAL_STAT,lf_eoc_received             ;Yes, lift the end of user command flag
        BRA LOWINT_EXIT                             ; go to end of ISR, restore context, return
RXERROR
		MOVF RCREG,W								;Discard received errored byte
        BCF RCSTA,CREN                              ;Clear receiver status
        BSF RCSTA,CREN                              ; "
LOWINT_EXIT_RX
        MOVFF FSR0L,SERIAL_RX_INT_FSR0L_SAVE        ;Recover previous RX FSR0L indirect addressing register value
        BRA LOWINT_EXIT                             ;Leave interrupt Routine
LOWINT_EXIT_TX
        MOVFF FSR0L,SERIAL_TX_INT_FSR0L_SAVE        ;Recover previous TX FSR0L indirect addressing register value
LOWINT_EXIT
		MOVFF LOG_BUFF_FSR0L_SAVE,FSR0L				;Restore the current FSR0L register value.
        MOVFF BSR_TEMP,BSR                          ;Restore BSR register
        MOVFF WREG_TEMP,WREG                        ;Restore working register
        MOVFF STATUS_TEMP,STATUS                    ;Restore STATUS register
        RETFIE 0                                    ;This exits and re-enable low priority interrupts

;******************************************************************************************************
;HIGHINT 
;High priority interrupt routine
; no need to save status registers for a high priority interrupt routine

HIGHINT    
        RETFIE FAST                                 ;this restores the 3 status registers and exits interrupt routine.

;******************************************************************************************************
;MAIN
;Start of main program
;Initialization Area. This stuff is executed only once and at Reset time.

MAIN    
        MOVLB .0                                    ;Select RAM bank 0 for all variable accesses 
        CLRF INTCON                                 ;Disable interrupts
        MOVLW b'00000000'                           ; "
        MOVWF INTCON2                               ; "
        MOVLW b'00000000'                           ; "
        MOVWF INTCON3                               ; "
        MOVLW b'10000000'                           ;enable interrupt priorities (high/low)
        MOVWF RCON                                  ; "
        MOVLW b'00110010'                           ;select internal oscillator and set it to 500KHz
        MOVWF OSCCON                                ; "
        MOVLW b'10000111'                           ;Timer0 is used for the long duration cycle timer, up to 90 days.
        MOVWF T0CON                                 ;Set Timer0 to operate on internal clock with a prescaler value of 256.
        MOVLW b'10000001'                           ;Timer1 is used for LED and push button clocking. 50 ms overflow is used.
        MOVWF T1CON                                 ;set timer1 and enable it            
        MOVLW b'10000000'                           ;set timer3 and disable it
        MOVWF T3CON                                 ; "        
        MOVLW b'00110000'                           ;disable CCP1 interrupt, enable Rx and Tx interrupts
        MOVWF PIE1                                  ; " 
        MOVLW b'00000000'                           ;do not enable anything on that register
        MOVWF PIE2                                  ; " 
        MOVLW b'00000000'                           ;assign low priority USART Tx and Rx
        MOVWF IPR1                                  ; "
        MOVLW b'00000001'                           ;Enable A/D function and select the reference voltage, but do not convert yet. 
        MOVWF ADCON0                                ; "
        MOVLW b'01111100'                           ;Enable A/D on pins AN0 to AN1.
        MOVWF ADCON1                                ; "
        MOVLW b'10111110'                           ;Set conversion time to 64 Tosc. Set acquisition to 20 x Conversion time.
        MOVWF ADCON2                                ; "
        MOVLW b'00110000'                           ;PORTB set as follows:
        MOVWF TRISB                                 ; 7-output connected to LED #2's red anode
                                                    ; 6-output connected to LED #2's green anode
                                                    ; 5-input for push button. Has internal pull-up.                                
                                                    ; 4-input used as Serial RX
                                                    ; 3-PWM1 output (see also configuration bit CONFIG3H)
                                                    ; 2-output to disable charge MOSFET (through NPN transistor)
                                                    ; 1-output used as Serial TX
                                                    ; 0-output to control Battery 2 Solid State Relay
        MOVLW b'00100011'                           ;PORTA set as follows:    
        MOVWF TRISA                                 ; 7-output connected to LED #1's red anode
                                                    ; 6-output connected to LED #1's green anode 
                                                    ; 5-MCLR (not used as I/O)                                    
                                                    ; 4-Not Used (open drain)
                                                    ; 3-output to control Battery 1 Solid State Relay
                                                    ; 2-output to disable discharge MOSFET
                                                    ; 1-A/D input connected to batteries
                                                    ; 0-A/D input connected to resistor (low side)
STARTUP_DELAY                                       ;This 1 sec delay is added to let the supply voltage stabilize.
;        MOVLW h'E1'                                 ;Load 1 second equivalent (500ns * 256 * (0x10000 - 0xE17B)) into timer0 value
;        MOVWF TMR0H                                 ; "
;        MOVLW h'7B'                                 ; "
;        MOVWF TMR0L                                 ; "
;        BCF INTCON,TMR0IF                           ;Clear the timer0 overflow interrupt flag
STARTUP_DELAY_LOOP  
;        BTFSS INTCON,TMR0IF                         ;Is flag raised (timer0 overflow)?
;        BRA STARTUP_DELAY_LOOP                      ;No, loop to test again
CLEAR_VARIABLES                                     ;Clear all variables, except the RS232 buffers and the EEPROM variables
        LFSR FSR1,0x000                             ;Load indirect addressing register with base address of the variables to clear
CLR_VARS_LOOP
        CLRF POSTINC1                               ;Clear variable and increase indirect addressing pointer
        MOVLW 0xFF                                  ;Have it reached end of buffer addresses
        CPFSEQ FSR1L                                ; "
        BRA CLR_VARS_LOOP                           ;No, loop again. 
INITIALIZE_REGS
        MOVLW pwm_period_init                       ;loads period in timer2 period register (value of 255)
        MOVWF PR2                                   ; "
        MOVLW b'00000100'                           ;set timer2 to prescaler div. by 1, no postscaler (488 Hz PWM frequency) 
        MOVWF T2CON                                 ; and enable it
        MOVLW ccp1_init                             ;enables ccp1 in PWM mode
        MOVWF CCP1CON                               ; "
READ_EEPROM_PARAMS                                  ;Parameters are transfered from Data EEPROM to RAM variables
        CLRF TEMP1_HI                               ;Clear temporary variable TEMP1_HI
        LFSR FSR1,low CYCLE_STAT                    ;Initialize indirect addressing register FSR1
READ_EEPROM_PARAMS_LOOP
        INCF TEMP1_HI,W                             ;Increment temp. var. TEMP1_HI and save result in Wreg
        MOVWF TEMP1_HI                              ;Also transfer value in temp var.
        CALL READ_EEPROM_DATA                       ;Read data from ERPROM address 
        MOVWF POSTINC1                              ;transfer value read from EEPROM to RAM variable
        MOVLW number_param_to_stor                  ;Is this the last address (parameter) to write?
        CPFSEQ TEMP1_HI                             ; "
        BRA READ_EEPROM_PARAMS_LOOP                 ;No, go transfer next address
CONT_INITIALIZE
		CALL CALC_BAT_VARS							;Yes, calculate the variables as a result of the parameters retrieved in the EEPROM
        MOVLW b'00100100'                           ;Enable USART TX, 8 bits, Async, High Speed
        MOVWF TXSTA                                 ; "
        MOVLW .12                             		; Set baud rate to 2400 bauds @ 500KHz Tosc frequency
        MOVWF SPBRG                                 ; "
        CLRF SPBRGH                                 ; "
        MOVLW b'10010000'                           ;Enable Serial port, enable Rx, 8 bits on Rx, Async
        MOVWF RCSTA                                 ; "
        CLRF WRT_EEPROM_POS                         ;clear the EEPROM write address position indicator
INIT_STAGE
		CLRF TEMP1_HI                               ;Clear temporary variable TEMP1_HI
        BTFSS CYCLE_STAT,conditioning_mode          ;Are we in battery conditioning mode?
        BRA INIT_NO_CONDIT                          ;No, go set initial stage without being in conditioning
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, set the off bit in TEMP1_HI
        BRA INIT_STAGE_REQ_STAT                     ;Go execute the stage change request       
INIT_NO_CONDIT
		BTFSC STAGE_STAT,cycle_off_bit              ;Not in Conditioning Mode. Is the saved stage the off stage?
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, set the off bit in TEMP1_HI
		BTFSC STAGE_STAT,idle_bit                   ;No, is the saved stage the idle stage?
		BSF TEMP1_HI,idle_bit                       ;Yes, start in idle stage. Set the idle bit in TEMP1_HI
        BTFSS CYCLE_STAT,pause_off_active           ;No, is the pause between two stages flag active? This indicates a potential blown fuse (overload) before this power up
        BRA CONT_INIT_NO_OVERLOAD                   ;No, start the usual manner
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, start in off stage. Set the off bit in TEMP1_HI
        BCF CYCLE_STAT,pause_off_active             ;Clear the pause off active flag.   
        BRA INIT_STAGE_REQ_STAT                     ;Go execute the stage change request
CONT_INIT_NO_OVERLOAD                               ;Not in Conditioning, not in overload, not in off or Idle.
		BTFSC STAGE_STAT,bat1_chrg_bit              ;Is the saved stage the battery #1 charge stage?
		BSF TEMP1_HI,bat1_disch_bit                 ;Yes, start in Bat #1 discharge stage. Set the Bat #1 discharge bit in TEMP1_HI 
		BTFSC STAGE_STAT,bat2_chrg_bit              ;No, Is the saved stage the battery #2 charge stage?
		BSF TEMP1_HI,bat2_disch_bit                 ;Yes, start in Bat #2 discharge stage. Set the Bat #1 discharge bit in TEMP1_HI 
		BTFSC STAGE_STAT,bat1_disch_bit             ;No, Is the saved stage the battery #1 discharge stage?
		BSF TEMP1_HI,bat1_disch_bit                 ;Yes, start in Bat #1 discharge stage. Set the Bat #1 discharge bit in TEMP1_HI 
		BTFSC STAGE_STAT,bat2_chrg_bit              ;No, Is the saved stage the battery #2 dis charge stage?
		BSF TEMP1_HI,bat2_disch_bit                 ;Yes, start in Bat #2 discharge stage. Set the Bat #1 discharge bit in TEMP1_HI 
		BTFSC STAGE_STAT,.6                         ;No, Is the saved stage bit 6 raised ?
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, dead end due to EEPROM corruption. Set the off bit in TEMP1_HI
		BTFSC STAGE_STAT,.7                         ;No, Is the saved stage bit 6 raised ?
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, dead end due to EEPROM corruption. Set the off bit in TEMP1_HI
		TSTFSZ STAGE_STAT                           ;No, Is the saved stage = 0x00 ?
        BRA INIT_STAGE_REQ_STAT                     ;No, go transfer TEMP1_HI into STAGE_REQ_STAT
		BSF TEMP1_HI,cycle_off_bit                  ;Yes, dead end due to EEPROM corruption. Set the off bit in TEMP1_HI
INIT_STAGE_REQ_STAT
		MOVFF TEMP1_HI,STAGE_REQ_STAT               ;Transfer TEMP1_HI into STAGE_REQ_STAT                              
        MOVLW low CYCLE_STAT                        ;
        MOVWF EEPROM_WRITE_FSR1L_SAVE               ;Set the EEPROM Write saved address to the first parameter address
        CLRF ALARM_STAT1                            ;Clear any existing alarm flags
		LFSR FSR0,LOG_BUFF                          ;Initialize FSR0 pointer to the LOG buffer base address
        MOVLW b'00100010'                           ;Initializr the pushbutton status register              
        MOVWF PUSHBUTT_STAT                         ; "
		BTFSC STAGE_STAT,cycle_off_bit				;If system starts in off stage?
		BRA CONT_INITIALIZE_2                       ;Yes, CLOCK_REG must start at 0 (not 0xFFFFFF) because CTRL_CLOCK is skipped.
		SETF CLOCK_REG_U          					;No, prepare the clock registers for reset to 0x000000 when first entering CTRL_CLOCK routine
		SETF CLOCK_REG_H             				; "
		SETF CLOCK_REG_L             				; "
CONT_INITIALIZE_2
        BSF INTCON,TMR0IF                           ;Clear the timer0 overflow interrupt flag
        CLRF TBLPTRU                                ;Clear the Table read upper address byte
        CLRF PIR1                                   ;Clear all interrupt request flags
        CLRF PIR2                                   ; "
        BSF INTCON,GIEH                             ;enable high priority interrupts
        BSF INTCON,GIEL                             ;enable low priority interrupts

;******************************************************************************************************
;This is the main loop that repeats indefinitely

BAT_CYCLER_MAIN                            
        CALL READ_PUSHBUTT                          ;Read the Pushbotton status
 		CALL CTRL_PUSHBUTT_ACTION                   ;Generate actions as a result of the Pushbutton status
		CALL CTRL_CLOCK                             ;Control the Cycle timer (clock)
		CALL CTRL_CYCLE                             ;Manage the cycle
        CALL CTRL_ADC                               ;Control the ADC reading cycle.
        CALL CTRL_DISCHARGE                         ;Control and end discharge stage
        CALL CTRL_CHARGE                            ;Control and end charge stage
        CALL CTRL_PWM                               ;Control the DAC output (using PWM) for coarse steps
		CALL CTRL_IDLE_MON                          ;Control the idle stage voltage monitoring activity
        CAll CTRL_OUTPUTS                           ;Control the output pins
        CALL CTRL_LEDS                              ;Control the bi-color LEDs based on the Charge/Discharge states
        CALL CTRL_PARAM_INFO_TX                     ;Control contents of the parameter line sent over serial port
        CALL CTRL_COMMAND_RX                        ;Control reception and process User Commands received over serial port
        CALL CTRL_PARAM_UPD_EEPROM                  ;Control writing in Flash EEPROM of parameters
        CALL CTRL_TIMER1_50MS                       ;Update the 50ms timer registers
        CALL SEND_LOG_INFO                          ;Control the transmission of the log information
		CALL SEND_VOLT_CURR_INFO                    ;Control the transmission of the voltage and current information
		BCF AD_STAT,new_adc_results					;Clear the ADC values
        GOTO BAT_CYCLER_MAIN                        ;Loop back to beginning

;******************************************************************************************************
;READ_PUSHBUTT
;This sub-routine reads the status of the manual stage override button, debounces it and flags its status 
;to other sub-routines.

READ_PUSHBUTT
        MOVF PUSHBUTT_STAT,W                        ;Move the push button stat register into Wreg
        XORWF PORTB,W                               ;XOR it with port B
        MOVWF TEMP1_HI                              ;Transfer result into TEMP1_HI
        BTFSS TEMP1_HI,pushbutt_debounced    		;Are the states of the pin and debounced state different?
        BRA CTRL_PUSHBUTT_RET                       ;No, leave routine
        BTFSS PUSHBUTT_STAT,pushbutt_in_debouncing  ;Yes, is the pin being debounced?
        BRA START_BUTT_DEBOUNCING                   ;No, go start debouncing process
CHECK_DEBOUNCING_END        
        MOVLW butt_debouncing_limit                 ;Yes, check if the debouncing delay is expired
        CPFSLT BUTT_DEBOUNCING_CTR                  ; "
        BRA END_BUTT_DEBOUNCING                     ;Yes, go end debouncing
        BRA CTRL_PUSHBUTT_RET                       ;No, leave routine
END_BUTT_DEBOUNCING
        BCF PUSHBUTT_STAT,pushbutt_in_debouncing    ;Clear the pushbutton in debouncing flag
        MOVLW pushbutt_pin_pos                      ;Transfer the pushbutton state into the debounced flag.
        ANDWF PORTB,W                               ; "
        BCF PUSHBUTT_STAT,pushbutt_debounced        ; "
        IORWF PUSHBUTT_STAT,F                       ; "
        BRA CTRL_PUSHBUTT_RET                       ; Leave routine
START_BUTT_DEBOUNCING
        BSF PUSHBUTT_STAT,pushbutt_in_debouncing    ;Set the pushbutton in debouncing flag
        CLRF BUTT_DEBOUNCING_CTR                    ;Clear the debouncing counter
CTRL_PUSHBUTT_RET
		RETURN										;Leave routine

;******************************************************************************************************
;CTRL_PUSHBUTT_ACTION
;This sub-routine generates actions on the various cycle stages based on the status of the pushbutton. It manages the
;action delay counter and then request a change of stage based on the current stage.

CTRL_PUSHBUTT_ACTION
		BTFSC PUSHBUTT_STAT,pushbutt_debounced		;Is the pushbutton currently activated?
		BRA TEST_ACTION_DELAY						;No, go test for action delay 
		CLRF ACTION_DELAY_CTR						;Yes, initialize the pushbutton action delay counter
		BTFSS PUSHBUTT_STAT,pushbutt_old_debounced	;Was the pushbutton activated before?
		BRA CTRL_PUSHBUTT_ACTION_RET				;Yes, leave routine
START_ACTION_DELAY
		CLRF ACTION_DELAY_CTR						;No, initialize the pushbutton action delay counter
		BCF PUSHBUTT_STAT,pushbutt_old_debounced	;Activate the old pushbutton flag
		BTFSS PUSHBUTT_STAT,push_butt_action_delay	;Is the pushbutton action delay mode already active?
		BRA XFER_STAGE_STAT                         ;No, go transfer STAGE_STAT register into the temp variable
		MOVFF STAGE_REQ_STAT,TEMP1_HI               ;Yes, transfer STAGE_REQ_STAT register into the temp variable
		BRA CONT_START_ACTION_DELAY                 ;Go continue the action delay start
XFER_STAGE_STAT
		MOVFF STAGE_STAT,TEMP1_HI                   ;Transfer STAGE_STAT register into the temp variable
		BSF PUSHBUTT_STAT,push_butt_action_delay	;Set the action delay flag
CONT_START_ACTION_DELAY
        RLNCF TEMP1_HI,W                            ;Shift left by one bit to move on to the next stage.
		MOVWF STAGE_REQ_STAT					    ;And load result into the stage request register
		BTFSC CYCLE_STAT,num_of_batteries			;Is the system provisioned for two batteries?
		BRA CONT_TEST_TWO_BATS						;Yes, skip the second battery exclusion steps
		BTFSS STAGE_REQ_STAT,bat2_disch_bit         ;No, are we requesting a battery #2 discharge stage?
		BRA CONT_TEST_TWO_BATS						;No, skip the second battery exclusion steps
		RLNCF STAGE_REQ_STAT,F					    ;Yes, shift left by two bits more to skip the battery #2 discharge and charge requests.
		RLNCF STAGE_REQ_STAT,F					    ; "	
CONT_TEST_TWO_BATS
		MOVLW stage_request_bits_mask               ;Is the stage bit higher than the idle stage (meaningless)?
		ANDWF STAGE_REQ_STAT,W                      ; "
		BNZ CTRL_PUSHBUTT_ACTION_RET                ;No, leave routine
		CLRF STAGE_REQ_STAT                         ;Yes, request for a start of entire new cycle (go to off stage)
		BSF  STAGE_REQ_STAT,.0                      ; "
		BRA CTRL_PUSHBUTT_ACTION_RET				;Leave routine
TEST_ACTION_DELAY
		BSF PUSHBUTT_STAT,pushbutt_old_debounced	;de-activate the old pushbutton flag
		BTFSC PUSHBUTT_STAT,push_butt_action_delay	;Is the pushbutton action delay mode active?
		BRA TEST_ACTION_DELAY_EXPIRED				;Yes, go test for action delay expiry	
		BRA CTRL_PUSHBUTT_ACTION_RET				;No, leave routine
TEST_ACTION_DELAY_EXPIRED
		MOVLW action_delay_limit					;Is the action delay time expired?
		CPFSLT ACTION_DELAY_CTR 					; "
		BRA EXPIRE_ACTION_DELAY						;Yes, go exit from action delay mode.
		BRA CTRL_PUSHBUTT_ACTION_RET				;No, leave routine
EXPIRE_ACTION_DELAY
		BCF PUSHBUTT_STAT,push_butt_action_delay    ;Clear the action delay flag
		CLRF CLOCK_REG_L							;Clear the CLOCK_REG registers. This reset the clock to 0 whenever a manual stage control is done.
		CLRF CLOCK_REG_H							; "
		CLRF CLOCK_REG_U							; "
CTRL_PUSHBUTT_ACTION_RET
		RETURN										;Leave routine

;******************************************************************************************************
;CTRL_ADC    
;This section controls the 10-bit ADC sampling process. No interrups are used in this process since it is
;not a critical process.

CTRL_ADC
        BTFSC ADCON0,GO                             ;Is a conversion currently in progress?
        BRA CTRL_ADC_RET                            ;Yes, leave routine
        BTFSS T3CON,TMR3ON                          ;Counting the delay between to sampling/conversion?
        BRA START_AD_DELAY                          ;No, go start the A/D delay counter Timer3
        BTFSS PIR2,TMR3IF                           ;Yes, has the delay expired (interrupt flag raised)?
        BRA CTRL_ADC_RET                            ;No, leave routine
        BCF T3CON,TMR3ON                            ;Yes, turn off Timer3
        BRA PROCESS_AD_CHANNEL                      ;Go process ADC channel
START_AD_DELAY
        MOVLW ad_delay_timer_val_h                  ;Load Timer3 with proper delay value corrensponding to 300usec
        MOVWF TMR3H                                 ; (2 x Tad + 50usec margin) minimum delay between two ADC readings
        MOVLW ad_delay_timer_val_l                  ; "
        MOVWF TMR3L                                 ; "
        BCF PIR2,TMR3IF                             ;Clear the Timer 3 rollover flag
        BSF T3CON,TMR3ON                            ;Start Timer3
        BRA CTRL_ADC_RET                            ;Leave routine
PROCESS_AD_CHANNEL
        MOVFF ADCON0,TEMP1_HI						;Temporarily save ADCON0 into temp variable TEMP1_HI
        RRNCF TEMP1_HI,F							;Shift twice TEMP1_HI variable right one bit without carry
        RRNCF TEMP1_HI,F							; This brings the three A/D channel bits as LSBs
        MOVLW b'00000111'							;Isolate the three A/D channel bits
        ANDWF TEMP1_HI,F							; "
TEST_ADCHAN_0_XFER
		MOVLW .0                                    ;Is it time to read ADC channel 0?
		SUBWF TEMP1_HI,W                            ; "
		BNZ TEST_ADCHAN_1_XFER                      ;No, go test for channel 1
		MOVFF ADRESH,RES_VOLT_HI                    ;Yes, transfer ADC value into channel 0 variable
		MOVFF ADRESL,RES_VOLT_LO                    ; "
		BRA INC_CHAN                                ;Go increment channel number to be read
TEST_ADCHAN_1_XFER
		MOVLW .1                                    ;Is it time to read ADC channel 1?
		SUBWF TEMP1_HI,W                            ; "
		BNZ INC_CHAN                      			;No, Go increment channel number to be read
        BTFSC PORTA,bat1_mosfet_pin                 ;Yes, Is battery #1 solid state relay activated?
        BRA XFER_BAT_VOLT_VAL                       ;Yes, go update the ADC variables
        BTFSS PORTB,bat2_mosfet_pin                 ;No, is battery #2 solid state relay activated?
        BRA INC_CHAN                                ;No, go increase channel number
XFER_BAT_VOLT_VAL
		MOVFF ADRESH,BAT_VOLT_HI                    ;Yes, transfer ADC value into channel 1 variable
		MOVFF ADRESL,BAT_VOLT_LO                    ; "
INC_CHAN		
        INCF TEMP1_HI,F								;Increment the A/D channel number
        MOVLW .2							        ;Have we surpassed the last channel number?
        CPFSEQ TEMP1_HI								; "
		BRA CONT_SET_CHAN                           ;No, go increase channel number
        CLRF TEMP1_HI    							;Yes, reset the channel number to 0    
	    BTFSC CYCLE_STAT,pause_off_active			;Is the pause between two stage active?
	    BRA CONT_SET_CHAN                           ;Yes, continue to the next channel but do not raise the A/D measurements ready
		BSF AD_STAT,new_adc_results                 ;No, raise flag to indicate that new A/D measurements are ready
CONT_SET_CHAN
        RLNCF TEMP1_HI,F							;No, shift twice TEMP1_HI variable left one bit without carry
        RLNCF TEMP1_HI,F							; This brings the three A/D channel bits back to their original position
		MOVLW b'11100011'							;Clear the A/D channel bits into ADCON0
		ANDWF ADCON0,F								; "
		MOVF TEMP1_HI,W								;Load the new channel number bits from TEMP1_HI into ADCON0
		IORWF ADCON0,F								; "
		BSF ADCON0,GO								;Start a new conversion		
CTRL_ADC_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;CTRL_CLOCK
;This sub-routine manages the cycle timer (clock). It restarts the timer when necessary, it end the various stages
;when their maximum duration is reached.

CTRL_CLOCK
        BTFSC STAGE_STAT,cycle_off_bit              ;Is the current stage the off stage?
        BRA CTRL_CLOCK_RET                          ;Yes, leave routine
        BTFSS INTCON,TMR0IF                         ;No, Is Timer0 overflow interrupt flag set?        
        BRA CTRL_CLOCK_RET                          ;No, leave routine
        BCF INTCON,TMR0IF                           ;Yes, clear Timer0 interrupt request flag        
        BSF CYCLE_STAT,check_idle_volt_req          ;Request for an idle stage voltage reading
        MOVLW t0_adjust_val_l                       ;Add the adjustment value to Timer0 (low byte) so that it rollovers every 120 secs
        ADDWF TMR0L,W                               ;
        MOVWF TEMP1_HI                              ;Save result temporarily
        MOVLW t0_adjust_val_h                       ;Add adjustment (high byte) to Timer0 with carry flag from low byte addition
        ADDWFC TMR0H,F                              ; "
        MOVFF TEMP1_HI,TMR0L                        ;Now write Timer0 low byte. This updates the whole 16 bits
INCR_CLOCK_L                                        ;Here we increment the 24 bit clock register
        INFSNZ CLOCK_REG_L,F                        ;Increment CLOCK_REG_L. Has it rolled over? 
        BRA INCR_CLOCK_H                            ;Yes, go increment CLOCK_REG_H
        BRA TEST_NEW_CLOCK_CYCLE                    ;No, go test clock
INCR_CLOCK_H
        INFSNZ CLOCK_REG_H,F                        ;Increment CLOCK_REG_H. Has it rolled over? 
        BRA INCR_CLOCK_U                            ;Yes, go increment CLOCK_REG_H
        BRA TEST_NEW_CLOCK_CYCLE                    ;No, go test clock
INCR_CLOCK_U
        INCF CLOCK_REG_U,F                          ;Increment CLOCK_REG_H.
TEST_NEW_CLOCK_CYCLE                                ;Here we test if a whole new clock cycle must be started
		BTFSC CYCLE_STAT,conditioning_mode          ;Are we in battery conditioning mode?
		BRA TEST_END_CHARGE                         ;Yes, go test end of charge cycle. Do not treat cycle clock
        MOVF CYCLE_CLOCK_LIM_L,W                    ;No, A 24-bit substraction is made TEMP1 - CYCLE_CLOCK_LIM
        SUBWF CLOCK_REG_L,W                         ; "
        MOVF CYCLE_CLOCK_LIM_H,W                    ; "         
        SUBWFB CLOCK_REG_H,W                        ; "  
        MOVF CYCLE_CLOCK_LIM_U,W                    ; "         
        SUBWFB CLOCK_REG_U,W                        ; "  
        BN TEST_END_CHARGE                          ;Is result negative (cycle time limit not reached yet)? Yes, go test charge cycle
        MOVLW end_cycle                    			;No, Load the end of cycle flag as log entry into the log buffer 
        CALL WRITE_MSG_LOG                          ; "      
START_NEW_CYCLE
		CLRF CLOCK_REG_L							;Clear the CLOCK_REG registers. This reset the clock to 0 when a new cycle begins.
		CLRF CLOCK_REG_H                            ; "
		CLRF CLOCK_REG_U                            ; "
		CLRF SAVED_CLOCK_VAL_L                      ;Clear the SAVED_CLOCK_VAL registers. 
		CLRF SAVED_CLOCK_VAL_H                      ;"
		CLRF SAVED_CLOCK_VAL_U                      ;"
        CLRF STAGE_REQ_STAT                         ;Set the stage request to battery #1 discharge (first stage in cycle)
        BSF STAGE_REQ_STAT,bat1_disch_bit           ; "                              
        BRA CTRL_CLOCK_RET                          ;Leave routine
TEST_END_CHARGE        
        MOVLW charge_bit_mask                       ;Is there a charge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BZ TEST_END_DISCHARGE 						;No, go test if discharge stage happening
        MOVFF CLOCK_REG_L,TEMP1_LO					;Yes, transfer the clock register into TEMP1
        MOVFF CLOCK_REG_H,TEMP1_HI					; "  
        MOVF SAVED_CLOCK_VAL_L,W                    ;A 16-bit substraction is made: TEMP1 - SAVED_CLOCK_VAL
        SUBWF TEMP1_LO,F                            ; "
        MOVF SAVED_CLOCK_VAL_H,W                    ; "         
        SUBWFB TEMP1_HI,F                           ; "  
        TSTFSZ TEMP1_HI                             ;Checking for an inverted battery connection. Is the charge duration short?
        BRA CONT_TEST_CHARGE                        ;No, continue with charge treatment
        MOVLW .2                                    ;Yes, are we in the second 120-sec interval?
        SUBWF TEMP1_LO,W                            ; "
        BNZ CONT_TEST_CHARGE                        ;No, continue with charge treatment
        BSF GENERAL1_STAT,check_bat_invert_now      ;Yes, request a check of battery reversal
CONT_TEST_CHARGE
		BTFSS STAGE_STAT,bat1_chrg_bit              ;Is battery #1 being charged?
		BRA TEST_END_CHG_BAT2                       ;No go load battery #2 charge limit
		MOVFF CLOCK_CHARGE_LIM1_H,TEMP3_HI          ;Yes, load battery #1 charge limit into TEMP3
		MOVFF CLOCK_CHARGE_LIM1_L,TEMP3_LO          ; "
		BRA CONT_TST_END_CHG                        ;Go to continue test for end of charge
TEST_END_CHG_BAT2
		MOVFF CLOCK_CHARGE_LIM2_H,TEMP3_HI          ;Load battery #2 charge limit into TEMP3
		MOVFF CLOCK_CHARGE_LIM2_L,TEMP3_LO          ; "
CONT_TST_END_CHG
        MOVF TEMP3_LO,W                             ;A 16-bit substraction is made TEMP1 - TEMP3
        SUBWF TEMP1_LO,W                            ; "
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB TEMP1_HI,W                           ; "  
        BN CTRL_CLOCK_RET                           ;Charge time limit not reached, leave routine.
        RLNCF STAGE_STAT,W                          ;Charge time limit reached. Shift left the stage request register to request the following stage of the corresponding battery
        MOVWF STAGE_REQ_STAT                        ;"
		BTFSC CYCLE_STAT,num_of_batteries			;Is the system provisioned for two batteries?
		BRA CTRL_CLOCK_RET						    ;Yes, skip the second battery exclusion steps and leave routine
		BTFSS STAGE_REQ_STAT,bat2_disch_bit         ;No, are we requesting a battery #2 discharge stage?
		BRA CTRL_CLOCK_RET						    ;No, skip the second battery exclusion steps
		RLNCF STAGE_REQ_STAT,F					    ;Yes, shift left by two bits more to skip the battery #2 discharge and charge requests.
		RLNCF STAGE_REQ_STAT,F					    ; "	
        BRA CTRL_CLOCK_RET                          ;Leave routine      
TEST_END_DISCHARGE        
        MOVLW discharge_bit_mask                    ;Is there a discharge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BZ CTRL_CLOCK_RET                           ;No, leave routine      
        MOVFF CLOCK_REG_L,TEMP1_LO					;Yes, transfer the clock register into TEMP1
        MOVFF CLOCK_REG_H,TEMP1_HI					; "
        MOVF SAVED_CLOCK_VAL_L,W                    ;A 24-bit substraction is made TEMP1 - SAVED_CLOCK_VAL
        SUBWF TEMP1_LO,F                            ; "
        MOVF SAVED_CLOCK_VAL_H,W                    ; "         
        SUBWFB TEMP1_HI,F                           ; "  
		BTFSS STAGE_STAT,bat1_disch_bit             ;Is battery #1 being discharged?
		BRA TEST_END_DISCHG_BAT2                    ;No go load battery #2 discharge limit
		MOVFF CLOCK_DISCHG_LIM1_H,TEMP3_HI          ;Yes, load battery #1 discharge limit into TEMP3
		MOVFF CLOCK_DISCHG_LIM1_L,TEMP3_LO          ; "
		BRA CONT_TST_END_DISCHG                     ;Go to continue test for end of discharge
TEST_END_DISCHG_BAT2
		MOVFF CLOCK_DISCHG_LIM2_H,TEMP3_HI          ;Load battery #2 charge limit into TEMP3
		MOVFF CLOCK_DISCHG_LIM2_L,TEMP3_LO          ; "
CONT_TST_END_DISCHG
        MOVF TEMP3_LO,W                             ;A 24-bit substraction is made TEMP1 - TEMP3
        SUBWF TEMP1_LO,W                            ; "
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB TEMP1_HI,W                           ; "  
        BN CTRL_CLOCK_RET                           ;Discharge time limit not reached? Yes, leave routine.
		BTFSS STAGE_STAT,bat1_disch_bit             ;No, Is battery #1 being discharged?
        BSF ALARM_STAT1,alm_bat2_dischrg_timeout    ;No, raise the battery #2 timeout alarm 
		BTFSS STAGE_STAT,bat2_disch_bit             ;Yes, Is battery #2 being discharged?
        BSF ALARM_STAT1,alm_bat1_dischrg_timeout    ;No, raise the battery #2 timeout alarm
        RLNCF STAGE_STAT,W                          ;Yes, shift left the stage request register to request a charge stage of the corresponding battery
        MOVWF STAGE_REQ_STAT                        ;"
CTRL_CLOCK_RET
        RETURN										;Leave routine

;******************************************************************************************************
; CTRL_DISCHARGE
;This sub-routine manages the discharge stage. It verifies if the discharge threshold voltage is reached.
;It orders the PWM to adjusts its duty cycle based on the current measured. It raises logs and flags alarms 
;if anomalies are detected.

CTRL_DISCHARGE
        MOVLW discharge_bit_mask                    ;Is there a discharge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BZ CTRL_DISCHARGE_RET                       ;No ,leave routine
        BTFSS AD_STAT,new_adc_results				;No, are there new ADC readings available?
        BRA CTRL_DISCHARGE_RET						;No, leave routine
TST_DISCHG_LIM_BAT
		BTFSS STAGE_STAT,bat1_disch_bit             ;Is it battery #1 being discharged?
		BRA TST_DISCHG_LIM_BAT2                     ;No, go test bat 2 discharge limit
		MOVFF DISCHRG_LIM1_HI,TEMP3_HI              ;Transfer the Bat #1 discharge limit into TEMP3
		MOVFF DISCHRG_LIM1_LO,TEMP3_LO              ; "
		BRA CONT_TST_DISCHG_LIM                     ;Go treat the discharge limit
TST_DISCHG_LIM_BAT2
		MOVFF DISCHRG_LIM2_HI,TEMP3_HI              ;Transfer the Bat #2 discharge limit into TEMP3    
		MOVFF DISCHRG_LIM2_LO,TEMP3_LO              ; "
CONT_TST_DISCHG_LIM
        MOVF TEMP3_LO,W                             ;A 16-bit substraction is made between DISCHRG_LIM and BAT_VOLT
        SUBWF BAT_VOLT_LO,W                         ; "
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB BAT_VOLT_HI,W                        ; "  
        BN  ABORT_DISCHARGE                      	;Is result negative (low voltage reached)? Yes, go abort cycle.
		MOVFF BAT_VOLT_HI,TEMP1_HI				    ;Transfer the battery voltage ADC reading into TEMP1 variable
		MOVFF BAT_VOLT_LO,TEMP1_LO				    ; "
        MOVF RES_VOLT_LO,W                       	;A 16-bit substraction is made:  BAT_VOLT - RES_VOLT
        SUBWF TEMP1_LO,F                         	; "
        MOVF RES_VOLT_HI,W                       	; "         
        SUBWFB TEMP1_HI,F                        	; "  
	    BN INC_DISCHARGE_CURRENT					;Voltage drop negative accross resistor? Due to resistor tolerances. Increase current
		BTFSS STAGE_STAT,bat1_disch_bit             ;Is it battery #1 being discharged?
		BRA TST_DISCHG_DROP_BAT2                    ;No, go test test Battery #2 for pwm railing
        BTFSS PWM_STAT,pwm_rail_max                 ;Yes, is the pwm railing flag raised?
        BRA CONT_XFER_DISCHRG_VOLT1                 ;No, go continue to control discharge current
        BSF ALARM_STAT1,alm_bat1_no_dischrg_curr    ;Yes, raise the insufficient current alarm on battery #1 
CONT_XFER_DISCHRG_VOLT1
		MOVFF TARGET_DISCHRG_DROP1_H,TEMP3_HI       ;Transfer the Bat #1 discharge voltage drop into TEMP3
		MOVFF TARGET_DISCHRG_DROP1_L,TEMP3_LO       ; "
		BRA CONT_TST_DISCHG_DROP                    ;Go test discharge drop
TST_DISCHG_DROP_BAT2
        BTFSS PWM_STAT,pwm_rail_max                 ;Is the pwm railing flag raised?
        BRA CONT_XFER_DISCHRG_VOLT2                 ;No, go continue to control discharge current
        BSF ALARM_STAT1,alm_bat2_no_dischrg_curr    ;Yes, raise the insufficient current alarm on battery #2
CONT_XFER_DISCHRG_VOLT2
		MOVFF TARGET_DISCHRG_DROP2_H,TEMP3_HI       ;Transfer the Bat #2 discharge voltage drop into TEMP3
		MOVFF TARGET_DISCHRG_DROP2_L,TEMP3_LO       ; "
CONT_TST_DISCHG_DROP
		MOVLW min_res_volt_val_l					;Is RES_VOLT lower than the opamp lowest rail?
        SUBWF RES_VOLT_LO,W                         ; "
        MOVLW min_res_volt_val_h                    ; "         
        SUBWFB RES_VOLT_HI,W                        ; "  
        BNN CONT_DISCHRG_RES_VOLT_OK               	;No? Continue current calculation? 
		BTFSS STAGE_STAT,bat1_disch_bit             ;Yes, is it battery #1 being discharged?
		BRA SET_BAT2_DISCHRG_CURR_LIMIT_ALM         ;No, go set battery #2 alarm
SET_BAT1_DISCHRG_CURR_LIMIT_ALM
        BSF ALARM_STAT1,alm_bat1_no_dischrg_curr	;Yes, raise the insufficient current alarm on battery #1
		BRA CTRL_DISCHARGE_RET						;Go leave routine		
SET_BAT2_DISCHRG_CURR_LIMIT_ALM
;        BSF ALARM_STAT1,alm_bat2_no_dischrg_curr	;Raise the insufficient current alarm on battery #2
		BRA CTRL_DISCHARGE_RET						;Go leave routine		
CONT_DISCHRG_RES_VOLT_OK
        MOVF TEMP3_LO,W                	            ;A 16-bit substraction is made:  (BAT_VOLT - RES_VOLT) - TARGET_DISCHRG_DROP
        SUBWF TEMP1_LO,F    						; " 
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB TEMP1_HI,F                        	; "  
		BN INC_DISCHARGE_CURRENT					;Is current below target? Yes, go increase the discharge current.
		MOVF TEMP1_LO,W								; Is the 16 bit difference 0 (target discharge current achieved)?
		IORWF TEMP1_HI,W							; "
        BZ CTRL_DISCHARGE_RET	                    ;Yes, leave routine
DEC_DISCHARGE_CURRENT									
		BSF PWM_STAT,dec_current			        ;No, discharge current above target. Decrease discharge current
		BRA CTRL_DISCHARGE_RET						;Go leave routine		
INC_DISCHARGE_CURRENT
		BSF PWM_STAT,inc_current			        ;No, discharge current above target. Decrease discharge current
		BRA CTRL_DISCHARGE_RET						;Go leave routine
ABORT_DISCHARGE
        BCF PWM_STAT,pwm_rail_max                   ;Yes, raise the pwm railing flag
        RLNCF STAGE_STAT,W                          ;Shift left the stage request register to request a charge stage of the corresponding battery
        MOVWF STAGE_REQ_STAT                        ;"
CTRL_DISCHARGE_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;CTRL_CHARGE
;This sub-routine manages the charge stage. It orders the PWM to adjusts its duty cycle based on the current measured. 
;It raises logs and flags alarms if anomalies are detected.

CTRL_CHARGE
        MOVLW charge_bit_mask                       ;Is there a charge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BZ CTRL_CHARGE_RET                          ;No , go clear the new ADV value flag and leave routine
        BTFSS AD_STAT,new_adc_results				;Yes, are there new ADC readings available?
        BRA CTRL_CHARGE_RET							;No, leave routine
TST_CHRG_CURRENT
		MOVFF RES_VOLT_HI,TEMP1_HI					;Yes, transfer the resistor voltage ADC reading into TEMP1 variable
		MOVFF RES_VOLT_LO,TEMP1_LO					; "
        MOVF BAT_VOLT_LO,W                       	;A 16-bit substraction is made:  RES_VOLT - BAT_VOLT
        SUBWF TEMP1_LO,F                         	; "
        MOVF BAT_VOLT_HI,W                       	; "         
        SUBWFB TEMP1_HI,F                        	; "  
	    BN INC_CHARGE_CURRENT						;Voltage drop negative accross resistor? Due to resistor tolerances. Go increase current.
        BTFSS GENERAL1_STAT,check_bat_invert_now    ;Time to check if battery is inverted (or shorted)?
        BRA CHECK_RAILING_VOLT1                     ;No, go continue to test 
        BCF GENERAL1_STAT,check_bat_invert_now      ;Yes, clear the flag
        TSTFSZ BAT_VOLT_HI                          ;Is the battery high voltage other than zero?
        BRA CHECK_RAILING_VOLT1                     ;Yes, go continue to transfer the charge voltage drop
        MOVLW .4                                    ;No, is the battery low voltage close to or at zero?
		CPFSLT BAT_VOLT_LO							; "
        BRA CHECK_RAILING_VOLT1                     ;No, go continue to transfer the charge voltage drop
        BTFSS STAGE_STAT,bat1_chrg_bit              ;Is it battery #1 being charged?
	    BRA RAISE_BAT2_CHG_ALM                      ;No, go test test Battery #2 charge current
        BSF ALARM_STAT1,alm_bat1_no_chrg_curr       ;Set the "battery #1 unable to generate proper current" alarm
        BRA ABORT_CHARGE                            ;Abort charge cycle
RAISE_BAT2_CHG_ALM
        BSF ALARM_STAT1,alm_bat2_no_chrg_curr       ;Set the "battery #2 unable to generate proper current" alarm
        BRA ABORT_CHARGE                            ;Abort charge cycle
CHECK_RAILING_VOLT1
		BTFSS STAGE_STAT,bat1_chrg_bit              ;Is it battery #1 being charged?
		BRA CHECK_RAILING_VOLT2                     ;No, go test test Battery #2 charge current
        BTFSS PWM_STAT,pwm_rail_max                 ;Yes, is the pwm railing flag raised?
        BRA CONT_XFER_CHRG_VOLT1                    ;No, go continue to treat charge current
        BSF ALARM_STAT1,alm_bat1_no_chrg_curr       ;Yes, Set the "battery #1 unable to generate proper current" alarm
        BRA ABORT_CHARGE                            ;Abort charge cycle
CHECK_RAILING_VOLT2
        BTFSS PWM_STAT,pwm_rail_max                 ;Is the pwm railing flag raised?
        BRA CONT_XFER_VOLT2                         ;No, go continue to treat charge current
        BSF ALARM_STAT1,alm_bat2_no_chrg_curr       ;Yes, Set the "battery #2 unable to generate proper current" alarm
        BRA ABORT_CHARGE                            ;Abort charge cycle
CONT_XFER_CHRG_VOLT1
		MOVFF TARGET_CHRG_DROP1_H,TEMP3_HI          ;No, transfer the Bat #1 charge voltage drop into TEMP3
		MOVFF TARGET_CHRG_DROP1_L,TEMP3_LO          ; "
		BRA CONT_TST_CHG_DROP                       ;Go test charge drop  
CONT_XFER_VOLT2
		MOVFF TARGET_CHRG_DROP2_H,TEMP3_HI          ;Transfer the Bat #2 charge voltage drop into TEMP3
		MOVFF TARGET_CHRG_DROP2_L,TEMP3_LO          ; "
CONT_TST_CHG_DROP        
		MOVF TEMP3_LO,W                	            ;A 16-bit substraction is made:  (SUPPLY_VOLT - RES_VOLT) - TARGET_CHRG_DROP
        SUBWF TEMP1_LO,F    						; " 
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB TEMP1_HI,F                        	; "  
		BN INC_CHARGE_CURRENT						;Is current below target? Yes, go increase the charge current.
		MOVF TEMP1_LO,W								; Is the 16 bit difference 0 (target charge current achieved)?
		IORWF TEMP1_HI,W							; "
        BZ CTRL_CHARGE_RET	                     	;Yes, leave routine
DEC_CHARGE_CURRENT									
		BSF PWM_STAT,dec_current					;No, charge current above target. Decrease charge current
		BRA CTRL_CHARGE_RET							;Go clear the ADC values		
INC_CHARGE_CURRENT
		BSF PWM_STAT,inc_current					;No, charge current above target. Decrease charge current
		BRA CTRL_CHARGE_RET							;Go clear the ADC values
ABORT_CHARGE
        BCF PWM_STAT,pwm_rail_max                   ;Yes, raise the pwm railing flag
        RLNCF STAGE_STAT,W                          ;Shift left the stage request register to request the following stage of the corresponding battery
        MOVWF STAGE_REQ_STAT                        ;"
	    BTFSC CYCLE_STAT,num_of_batteries			;Is the system provisioned for two batteries?
	    BRA CTRL_CHARGE_RET						    ;Yes, skip the second battery exclusion steps and leave routine
	    BTFSS STAGE_REQ_STAT,bat2_disch_bit         ;No, are we requesting a battery #2 discharge stage?
	    BRA CTRL_CHARGE_RET						    ;No, skip the second battery exclusion steps
	    RLNCF STAGE_REQ_STAT,F					    ;Yes, shift left by two bits more to skip the battery #2 discharge and charge requests.
	    RLNCF STAGE_REQ_STAT,F					    ; "	
CTRL_CHARGE_RET
        RETURN										;Leave routine

;******************************************************************************************************
;CTRL_PWM    
;This section controls the 10-bit Pulse Width Modulator's duty cycle (acting like a DAC) to control the
;charge and discharge current. This routine works does not require an interrupt routine to reload the duty cycle.

CTRL_PWM   
        MOVLW charge_bit_mask                       ;Is there a charge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BNZ PWM_STAGE_ON                   	    	;Yes, go treat the PWM
        MOVLW discharge_bit_mask                    ;No, is there a discharge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BZ CLEAR_PWM_REGS                   	    ;No, leave routine
PWM_STAGE_ON
		BTFSC CYCLE_STAT,pause_off_active			;Is the pause between two stage active?
		BRA CLEAR_PWM_REGS                   	    ;Yes, leave routine	
        BTFSC PWM_STAT,inc_current          		;Charge/discharge current increase required? 
        BRA DAC_INC                                 ;Yes, goto execute current increase.
        BTFSC PWM_STAT,dec_current          		;No, charge/discharge current decrease required? 
        BRA DAC_DEC                                 ;Yes, goto execute current decrease
        BRA CTRL_PWM_RET                   	        ;No, leave routine
DAC_INC                                             ;Execute DAC increase
        BCF PWM_STAT,inc_current            		;Clear the charge current increase required flag
        MOVLW h'03'                                 ;Verify if upper end of L register range reached 
        CPFSEQ PWM_DUTY_L                           ; "
        BRA INC_L_OK                                ;No, go to increase L
        MOVLW h'FF'                                 ;Yes, verify if upper end of H register reached 
        CPFSEQ PWM_DUTY_H                           ; "
        BRA INC_H_OK                                ;No, go to increase H and L routine
        BSF PWM_STAT,pwm_rail_max                   ;Yes, raise the pwm railing flag
        BRA PWM_NEW_CYCLE                           ;Do not increase DAC and Leave routine
INC_H_OK    
        INCF PWM_DUTY_H,F                           ;Increase H register
INC_L_OK
        INCF PWM_DUTY_L,F                           ;Increase L register
        MOVLW h'04'                                 ;Verify if upper end of L register range reached 
        CPFSLT PWM_DUTY_L                           ; "
        CLRF PWM_DUTY_L                             ;Yes, clear register
        BRA SET_NEW_PWM_VAL                         ;No, go load new PWM value
DAC_DEC    
        BCF PWM_STAT,dec_current            		;Clear the charge current increase required flag
        TSTFSZ PWM_DUTY_L                           ;Verify if lower end of L register range reached 
        BRA DEC_L_OK                                ;No, go to decrease L
        TSTFSZ PWM_DUTY_H                           ;Yes, verify if lower end of H register range reached 
        BRA DEC_H_OK                                ;No, go to decrease H and L routine
        BRA PWM_NEW_CYCLE                           ;Yes, do not decrease DAC and Leave routine
DEC_H_OK    
        DECF PWM_DUTY_H,F                           ;Decrease H register
DEC_L_OK
        DECF PWM_DUTY_L,F                           ;Decrease L register
        MOVLW h'FF'                                 ;Verify if lower end of L register range reached 
        CPFSEQ PWM_DUTY_L                           ; "
        BRA SET_NEW_PWM_VAL                         ;No, go load new PWM value
        MOVLW h'03'                                 ;Yes, load register with rollover value of 3.
        MOVWF PWM_DUTY_L                            ; "
SET_NEW_PWM_VAL
        MOVFF PWM_DUTY_H,CCPR1L                     ;load initial eight duty cycle MSb's in CCP1 register        
        BCF CCP1CON,4                               ;clear two duty cycle LSb's in CCP register
        BCF CCP1CON,5                               ; "
        MOVF PWM_DUTY_L,W                           ;load Wreg with the 2 duty cycle LSbs
        MULLW .16                                   ;this shifts the two duty cycle LSb's in position 4 and 5.
        MOVF PRODL,W                                ;move multiplication result back in Wreg
        IORWF CCP1CON,F                             ;transfer bits 4 and 5 in CCP register                                    
PWM_NEW_CYCLE
        BCF PIR1,TMR2IF                             ;clear TIMER2 interrupt request flag        
        BRA CTRL_PWM_RET                            ;leave routine
CLEAR_PWM_REGS
        MOVLW init_pwm_duty_h                       ;Initialize the PWM duty registers for 25% of full range to save time.
        MOVWF PWM_DUTY_H                            ; "
        CLRF PWM_DUTY_L                             ;Clear the PWM duty L register
        CLRF CCPR1L                                 ;Clear eight duty cycle MSb's in CCP1 register        
        BCF CCP1CON,4                               ;Clear two duty cycle LSb's in CCP register
        BCF CCP1CON,5                               ; "
CTRL_PWM_RET
        RETURN                                      ;leave routine

;******************************************************************************************************
;CTRL_CYCLE
;This sub-routine manages the change of stage based on the request it receives from other sub-routines. It also
;sets the proper log messages. It also manages the pause inserted right after a stage change with all outputs in 
;the off state. 

CTRL_CYCLE
		BTFSC PUSHBUTT_STAT,push_butt_action_delay	;Is the pushbutton action delay mode active?
        BRA CTRL_CYCLE_END                          ;Yes, leave routine        
TEST_PAUSE_ACTIVE
        MOVF STAGE_REQ_STAT,W                       ;No, is there a stage change request pending?
        BNZ TEST_CONDITIONING                       ;No, go test for conditioning mode
		BTFSC CYCLE_STAT,pause_off_active           ;Yes, is the outputs off pause active?
		BRA TST_PAUSE_COUNT		                    ;Yes, go test the pause count
        BRA CTRL_CYCLE_END                          ;No, leave routine        
TEST_CONDITIONING
        BTFSS CYCLE_STAT,conditioning_mode          ;Are we in battery conditioning mode?
        BRA CHANGE_STAGE                            ;No, go update the stage
        BTFSS STAGE_REQ_STAT,idle_bit               ;Yes, is there a request for the idle stage?
        BRA CHANGE_STAGE                            ;No, go update the stage
        CLRF STAGE_REQ_STAT                         ;Yes, set the stage request to battery #1 discharge (first stage in cycle)
        BSF STAGE_REQ_STAT,bat1_disch_bit           ; "                                       
        INCF CONDIT_CYCLE_CTR,F                     ;Increment the conditioning cycle counter
        MOVF MAX_CONDIT_CYCLES,W                    ;Have we reached the maximum number of conditioning cycles?
        CPFSEQ CONDIT_CYCLE_CTR                     ; "
        BRA CHANGE_STAGE                            ;No, go update the stage
        CLRF CONDIT_CYCLE_CTR,F                     ;Yes, clear the conditioning cycle counter
        CLRF STAGE_REQ_STAT                         ; Set the stage request to off stage
        BSF STAGE_REQ_STAT,cycle_off_bit            ; "                                       
        BCF CYCLE_STAT,conditioning_mode            ;Leave the conditioning mode
CHANGE_STAGE
        MOVFF STAGE_REQ_STAT,STAGE_STAT             ;Transfer the stage request into real stage
        CLRF STAGE_REQ_STAT                         ;Clear the request register
        BSF GENERAL1_STAT,write_cycle_req           ;Set the flag to write the EEPROM
TEST_OFF_STAGE
        BTFSS STAGE_STAT,cycle_off_bit              ;Is the new stage the off stage?
        BRA CONT_CHANGE_STAGE                       ;No go continue to process new stage
        CLRF ALARM_STAT1                            ;Yes, clear the Alarm register
		CLRF CLOCK_REG_U          					;Prepare the clock registers for reset to 0x000000 when entering CTRL_CLOCK routine next time
		CLRF CLOCK_REG_H             				; "
		CLRF CLOCK_REG_L             				; "
		CLRF SAVED_CLOCK_VAL_L                      ; "
		CLRF SAVED_CLOCK_VAL_H                      ; "
		CLRF SAVED_CLOCK_VAL_U                      ; "
CONT_CHANGE_STAGE
		CLRF PAUSE_OUTS_OFF_CTR                     ;Clear the outputs off pause counter
		BSF CYCLE_STAT,pause_off_active             ;Set the outputs off pause active flag
		BCF AD_STAT,new_adc_results					;Clear any leftover ADC value
		MOVFF CLOCK_REG_U,SAVED_CLOCK_VAL_U         ;Save the clock register values to calculate the beginning of the stage later
		MOVFF CLOCK_REG_H,SAVED_CLOCK_VAL_H         ; "
		MOVFF CLOCK_REG_L,SAVED_CLOCK_VAL_L         ; "
SET_BAT1_DISCHRG_LOGS
        BTFSS STAGE_STAT,bat1_disch_bit             ;Are we starting a battery #1 discharge stage?
        BRA SET_BAT2_DISCHRG_LOGS                   ;No, go test for a log for battery #2 discharge stage
        BCF ALARM_STAT1,alm_bat1_no_dischrg_curr    ;Yes, clear the battery #1 alarm latches   
        BCF ALARM_STAT1,alm_bat1_dischrg_timeout    ; "
        MOVLW bat1_dischrg_start                    ;Load the proper log
        BRA CALL_LOG_WRT_MSG                        ; "
SET_BAT2_DISCHRG_LOGS        
        BTFSS STAGE_STAT,bat2_disch_bit             ;Are we starting a battery #2 discharge stage?
        BRA SET_BAT1_CHRG_LOGS                      ;No, go test for a log for battery #1 charge stage
        BCF ALARM_STAT1,alm_bat2_no_dischrg_curr    ;Yes, clear the battery #2 alarm latches
        BCF ALARM_STAT1,alm_bat2_dischrg_timeout    ; "
        MOVLW bat2_dischrg_start                    ;Load the proper log
        BRA CALL_LOG_WRT_MSG                        ; "
SET_BAT1_CHRG_LOGS
        BTFSS STAGE_STAT,bat1_chrg_bit              ;Are we starting a battery #1 charge stage?
        BRA SET_BAT2_CHRG_LOGS                      ;No, go test for a log for battery #2 charge stage
        BCF ALARM_STAT1,alm_bat1_no_chrg_curr       ;Yes, clear the battery #1 alarm latches
        BCF ALARM_STAT1,alm_bat1_undervolt_end      ; "
        MOVLW bat1_chrg_start                       ;Load the proper log
        BRA CALL_LOG_WRT_MSG                        ; "
SET_BAT2_CHRG_LOGS
        BTFSS STAGE_STAT,bat2_chrg_bit              ;Are we starting a battery #2 discharge stage?
        BRA SET_BAT2_IDLE_LOGS                      ;No, go test for a log for the idle stage
        BCF ALARM_STAT1,alm_bat2_no_chrg_curr       ;Yes, clear the battery #2 alarm latches    
        BCF ALARM_STAT1,alm_bat2_undervolt_end      ; "
        MOVLW bat2_chrg_start                       ;load the proper log
        BRA CALL_LOG_WRT_MSG                        ; "
SET_BAT2_IDLE_LOGS
        BTFSC STAGE_STAT,idle_bit                   ;No, Are we starting an idle stage?
        MOVLW idle_start                            ;Yes, load the proper log
        BTFSC STAGE_STAT,cycle_off_bit              ;No, Are we starting off cycle stage?
        MOVLW off_start                             ;Yes, load the proper log
CALL_LOG_WRT_MSG
        CALL WRITE_MSG_LOG                          ;Load the log entry into the log buffer       
        BRA CTRL_CYCLE_END                          ;Leave routine        
TST_PAUSE_COUNT
		MOVLW pause_outs_limit                      ;Is the pause insertion between two stages expired?
		CPFSEQ PAUSE_OUTS_OFF_CTR                   ; "
		BRA CTRL_CYCLE_END                          ;No, go leave routine
		BCF CYCLE_STAT,pause_off_active             ;Yes, clear the pause insertion active register
        BSF GENERAL1_STAT,write_cycle_req           ;Set the flag to write the EEPROM       
CTRL_CYCLE_END
		RETURN										;Leave routine

;******************************************************************************************************
;CTRL_IDLE_MON
;This sub-routine checks the battery voltages during the idle stage every 120 seconds and raises alarms accordingly.

CTRL_IDLE_MON
        BTFSS STAGE_STAT,idle_bit                   ;Are we in the idle stage?
        BRA NOT_IDLE_CLR_ALMS                       ;No, clear an alarms and leave routine
        BTFSC CYCLE_STAT,check_idle_volt_req        ;Yes, is there a request for an idle stage voltage check?      
        BRA START_IDLE_VOLT_CHECK                   ;Yes , leave routine
        BTFSC CYCLE_STAT,checking_idle_volt         ;No, is there a idle voltage check currently happening?
        BRA CONT_IDLE_VOLT_CHECK                    ;Yes, go treat it
        BRA CTRL_IDLE_MON_RET                       ;No , go leave routine
START_IDLE_VOLT_CHECK
        BCF CYCLE_STAT,check_idle_volt_bat_num      ;Set the voltage check battery number to Battery #1
        BSF CYCLE_STAT,checking_idle_volt           ;Raise the idle voltage being checked flag
        CLRF IDLE_VOLT_CHK_CTR                      ;Clear the idle voltage check counter
        BRA CTRL_IDLE_MON_RET						;Go leave routine
CONT_IDLE_VOLT_CHECK
		CLRF TEMP1_HI                               ;Clear temporary variable TEMP1_HI
        MOVLW max_idle_volt_check_delay             ;Is the idle volt delay reached
        CPFSEQ IDLE_VOLT_CHK_CTR                    ; "
        BRA CTRL_IDLE_MON_RET                       ;No, leave routine
		BTFSC CYCLE_STAT,check_idle_volt_bat_num    ;Yes, is it battery #1 being tested?
		BRA TST_VOLT_BAT2                           ;No, go test bat 2 voltage
		MOVFF DISCHRG_LIM1_HI,TEMP3_HI              ;Yes, Transfer the Bat #1 discharge limit into TEMP3
		MOVFF DISCHRG_LIM1_LO,TEMP3_LO              ; "
		BSF TEMP1_HI,alm_bat1_undervolt_end         ;Transfer battery #1 low voltage alarm into TEMP1_HI		    
		BRA CONT_TST_VOLT_LIM                       ;Go treat the under-voltage limit
TST_VOLT_BAT2
		MOVFF DISCHRG_LIM2_HI,TEMP3_HI              ;Transfer the Bat #2 discharge limit into TEMP3    
		MOVFF DISCHRG_LIM2_LO,TEMP3_LO              ; "
		BSF TEMP1_HI,alm_bat2_undervolt_end		    ;Transfer battery #2 low voltage alarm into TEMP1_HI
CONT_TST_VOLT_LIM
        MOVF TEMP3_LO,W                             ;A 16-bit substraction is made between DISCHRG_LIM and BAT_VOLT
        SUBWF BAT_VOLT_LO,W                         ; "
        MOVF TEMP3_HI,W                             ; "         
        SUBWFB BAT_VOLT_HI,W                        ; "  
		BNN CLR_BAT_VOLT_ALM                        ;Is result positive or zero? Yes, go clear any low voltage alarm
		MOVF TEMP1_HI,W                             ;No, Transfer the alarm flag into Wreg
		IORWF ALARM_STAT1,F                         ;Transfer the alarm into the alarm register
		BRA TEST_NEXT_BAT_IDLE                      ;Go test the next battery for idle voltage
CLR_BAT_VOLT_ALM
		COMF TEMP1_HI,W                             ;No undervoltage detected. Clear the alarm latch of any idle undevoltage alarm.
		ANDWF ALARM_STAT1,F                         ; "
TEST_NEXT_BAT_IDLE
		BTFSS CYCLE_STAT,num_of_batteries			;Is the system provisioned for two batteries?
		BRA END_IDLE_VOLT_PASS				        ;No, go end the idle voltage check pass
		BTFSC CYCLE_STAT,check_idle_volt_bat_num    ;Yes, is battery #2 being tested?
		BRA END_IDLE_VOLT_PASS					    ;Yes, go end the idle voltage check pass    
		BSF CYCLE_STAT,check_idle_volt_bat_num    	;No, Set battery #2 as being tested
        CLRF IDLE_VOLT_CHK_CTR                      ;Clear the idle voltage check counter
		BRA CTRL_IDLE_MON_RET                       ;Go leave routine.		
NOT_IDLE_CLR_ALMS
		BCF ALARM_STAT1,alm_bat1_undervolt_end      ;Clear the alarms
		BCF ALARM_STAT1,alm_bat2_undervolt_end      ; "
END_IDLE_VOLT_PASS
		BCF CYCLE_STAT,checking_idle_volt           ;Clear the idle voltage being checked flag
		BCF CYCLE_STAT,check_idle_volt_bat_num    	;Clear battery #2 as being tested
CTRL_IDLE_MON_RET
        BCF CYCLE_STAT,check_idle_volt_req          ;Clear the check idle voltage requested flag 
        RETURN										;Leave routine

;******************************************************************************************************
;CTRL_OUTPUTS
;This sub-routine enables/disables the Solid state relays, MOSFET clamping transistors and the PWM output 
;pin based on the current stage and system status.

CTRL_OUTPUTS
        BTFSS CYCLE_STAT,checking_idle_volt         ;Is Cycler checking the idle battery voltages?
        BRA CONT_CTRL_OUTS                          ;No, go control the outputs
TEST_IDLE_CHK_BAT_NUM
		MOVLW max_idle_volt_check_delay-.1          ;Yes. Is the idle stage voltage check counter expired?
		CPFSLT IDLE_VOLT_CHK_CTR		            ; "
		BRA TEST_IDLE_VOLT_NOW                      ;Yes, go test idle voltage now
		BRA SW_OFF_OUTS		                        ;No, go switch all outputs off.
TEST_IDLE_VOLT_NOW
        BTFSS CYCLE_STAT,check_idle_volt_bat_num    ;Is battery #1 being tested during idle?
        BRA SW_BAT1_ON                              ;Yes, go switch battery #1 on 
        BRA SW_BAT2_ON                              ;No, go switch battery #2 on. 
CONT_CTRL_OUTS
		BTFSS CYCLE_STAT,pause_off_active           ;Is the beginning-of-stage pause active?
		BRA CONT_CTRL_OUTS2                         ;No, continue with normal control
		MOVLW pause_outs_all_off_limit              ;Yes, has the all-off pause duration expired?
		CPFSLT PAUSE_OUTS_OFF_CTR                   ; "
		BRA CONT_CTRL_OUTS2                         ;Yes, go control outputs normally. 
		BRA SW_OFF_OUTS                             ;No, go switch all outputs off
CONT_CTRL_OUTS2
		MOVLW discharge_bit_mask                    ;Is there a discharge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BNZ SW_DISCHARGE_OUTS                       ;Yes, go set the outputs for discharge stage
        MOVLW charge_bit_mask                       ;no, Is there a charge stage happening?
        ANDWF STAGE_STAT,W                          ; "
        BNZ SW_CHARGE_OUTS                          ;Yes, go set the outputs for charge stage
SW_OFF_OUTS                                         ;No charge or discharge, turn off all outputs
        BSF PORTA,.2                                ;Turn off Discharge MOSFET
        BSF PORTB,.2                                ;Turn off Charge MOSFET
        BCF PORTB,.0                                ;Turn off Battery #2 Solid State Relay
        BCF PORTA,.3                                ;Turn off Battery #1 Solid State Relay 
SW_OFF_PWM
		CLRF CCP1CON 								;Turn off PWM output 
        BCF PORTB,.3								;Set PB3 output latch low and
        BRA CTRL_OUTPUTS_RET                        ;Leave routine
SW_DISCHARGE_OUTS
        BSF PORTB,.2                                ;Turn off Charge MOSFET
        BCF PORTA,.2                                ;Turn on Discharge MOSFET
		BTFSC CYCLE_STAT,pause_off_active           ;Is the all-off pause active? 
		BRA TEST_BAT_NUM	                        ;Yes, go test for battery number directly
		BRA TURN_ON_PWM                             ;No, go turn on the PWM pin
SW_CHARGE_OUTS
        BSF PORTA,.2                                ;Turn off Discharge MOSFET
	    BCF PORTB,.2                                ;Turn on Charge MOSFET
		BTFSC CYCLE_STAT,pause_off_active           ;Is the all-off pause active? 
		BRA TEST_BAT_NUM		                    ;Yes, go test for battery number directly
TURN_ON_PWM
		BTFSC CCP1CON,.3							;No, is PWM already on?
		BRA TEST_BAT_NUM							;Yes go test what battery to work on
		MOVLW ccp1_init								;No, turn PWM on
		MOVWF CCP1CON								; "
TEST_BAT_NUM
        MOVLW bat1_chg_dischg_mask                  ;Is battery #1 being charged or discharged?
        ANDWF STAGE_STAT,W                          ; "
        BZ SW_BAT2_ON                               ;No, go switch battery #2 on 
SW_BAT1_ON
        BCF PORTB,.0                                ;Yes, turn off Battery #2 Solid State Relay 
        BSF PORTA,.3                                ;Turn on Battery #1 Solid State Relay 
        BRA CTRL_OUTPUTS_RET                        ;Leave routine
SW_BAT2_ON
        BCF PORTA,.3                                ;Turn off Battery #1 Solid State Relay
        BSF PORTB,.0                                ;Turn on Battery #2 Solid State Relay
CTRL_OUTPUTS_RET
		RETURN										;Leave routine

;******************************************************************************************************
;CTRL_LEDS

; This LED control routine will work when a two bi-color LEDs are connected between PORTA, pins 6 and 7 
; and PORTB, pins 6 and 7. When required, the pins will toggle repeatedly to create the perceived color 
; of amber (red and green mixed together).

CTRL_LEDS
		BTFSS PUSHBUTT_STAT,push_butt_action_delay	;Is the pushbutton action delay mode already active?
        BRA XFER_STAT_NO_DELAY                   	;No, go set the current stage LED states 
XFER_STAT_DELAY
        MOVFF STAGE_REQ_STAT,TEMP1_HI               ;Yes, transfer the stage request status byte to TEMP1_HI
		BRA TEST_LEDS                            	;Go test the LEDs
XFER_STAT_NO_DELAY
        MOVFF STAGE_STAT,TEMP1_HI                   ;Transfer the current stage status byte to TEMP1_HI
TEST_LEDS
        BTFSC TEMP1_HI,cycle_off_bit              	;Is the cycle in OFF state?
        BRA SET_LEDS_OFF                            ;Yes, go set the LEDs to OFF
TEST_LED1
        BTFSC TEMP1_HI,bat1_chrg_bit              	;No, is battery #1 being charged? 
        BRA SET_LED1_RED                            ;Yes, go set LED #1 red.
        BTFSC TEMP1_HI,bat1_disch_bit             	;No, is battery #1 being discharged?
        BRA SET_LED1_AMBER                          ;Yes, go set LED #1 amber.    
        BRA SET_LED1_GREEN                          ;No, go set LED #1 green    
SET_LED1_RED
        BSF PORTA,led_red_anode                     ;Set LED#1 red anode high
        BCF PORTA,led_green_anode                   ;Set LED#1 green anode low
        BRA TEST_LED2                               ;Go set LED #2
SET_LED1_AMBER                                      
        BTG PORTA,led_red_anode                     ;Toggle LED #1 red anode
        BTFSS PORTA,led_red_anode                   ;Is LED #1 red anode high?
        BSF PORTA,led_green_anode                   ;No, set LED #1 green anode high
        BTFSC PORTA,led_red_anode                   ;Yes, is LED #1 red anode low?
        BCF PORTA,led_green_anode                   ;No, set LED #1 green anode low 
        BRA TEST_LED2                               ;Yes, go test LED #2
SET_LED1_GREEN
        BCF PORTA,led_red_anode                     ;Set LED #1 red anode low 
        BSF PORTA,led_green_anode                   ;Set LED #1 green anode high
TEST_LED2
		BTFSC CYCLE_STAT,num_of_batteries			;Is the system provisioned for two batteries?        
    	BRA TEST_LED2_ON                            ;Yes, go treat LED #2
    	BCF PORTB,led_red_anode                     ;No, Set LED #2 red anode low
    	BCF PORTB,led_green_anode                   ;Set LED #2 green anode low
    	BRA TEST_LEDS_FLASH_ACTION_DEL              ;Go test for flashing LEDs        
TEST_LED2_ON
		BTFSC TEMP1_HI,bat2_chrg_bit                ;is battery #2 being charged?
        BRA SET_LED2_RED                            ;Yes, go set LED #2 red.
        BTFSC TEMP1_HI,bat2_disch_bit               ;No, is battery #2 being discharged?
        BRA SET_LED2_AMBER                          ;Yes, go set LED #2 amber.
        BRA SET_LED2_GREEN                          ;No, go set LED #2 green
SET_LED2_RED
        BSF PORTB,led_red_anode                     ;Set LED#2 red anode high
        BCF PORTB,led_green_anode                   ;Set LED#2 green anode low
        BRA TEST_LEDS_FLASH_ACTION_DEL              ;Go test for flashing LEDs
SET_LED2_AMBER
        BTG PORTB,led_red_anode                     ;Toggle LED #2 red anode
        BTFSS PORTB,led_red_anode                   ;Is LED #2 red anode high?
        BSF PORTB,led_green_anode                   ;No, set LED #2 green anode high
        BTFSC PORTB,led_red_anode                   ;Yes, is LED #2 red anode low?
        BCF PORTB,led_green_anode                   ;No, set LED #2 green anode low
        BRA TEST_LEDS_FLASH_ACTION_DEL              ;Go test for flashing LEDs
SET_LED2_GREEN
        BCF PORTB,led_red_anode                     ;Set LED #2 red anode low 
        BSF PORTB,led_green_anode                   ;Set LED #2 green anode high
TEST_LEDS_FLASH_ACTION_DEL
		BTFSS PUSHBUTT_STAT,push_butt_action_delay	;Is the pushbutton action delay mode already active?
        BRA TEST_ALARM_LEDS_FLASH                   ;No, go test for alarms
        MOVLW led_flash_limit                       ;Yes, has the flashing counter limit been reached?
        CPFSLT LED_FLASH_CTR                        ; "
        BRA FLASH_LEDS                              ;Yes, go toggle the pin modes
        BRA CHECK_FLASH_LED                         ;No,                         
TEST_ALARM_LEDS_FLASH
        MOVF ALARM_STAT1,W                          ;Are there any alarms latched?
        BZ CTRL_LEDS_RET                            ;No, go turn on LEDs
        MOVLW led_alm_flash_limit                   ;Yes, has the flashing counter alarm limit been reached?
        CPFSLT LED_FLASH_CTR                        ; "
        BRA FLASH_LEDS_ALM                          ;Yes, go toggle LEDs
        BRA CHECK_FLASH_LED_ALM                         
FLASH_LEDS
        CLRF LED_FLASH_CTR                          ;Clear the LED flashing counter
        BTG GENERAL1_STAT,leds_flashing              ;Toggle LED #1 pin mode
CHECK_FLASH_LED
        BTFSC GENERAL1_STAT,leds_flashing            ;Toggle LED #1 pin mode
        BRA SET_LEDS_OFF
        BRA CTRL_LEDS_RET                           ;No, leave routine
FLASH_LEDS_ALM
        CLRF LED_FLASH_CTR                          ;yes, clear the LED flashing counter
        BTG GENERAL1_STAT,leds_flashing              ;Toggle LED #1 pin mode
CHECK_FLASH_LED_ALM
        BTFSS GENERAL1_STAT,leds_flashing            ;Toggle LED #1 pin mode
        BRA CTRL_LEDS_RET                           ;Leave routine
        MOVLW bat1_alm_mask                         ;Are there any alarms against battery #1?
        ANDWF ALARM_STAT1,W                         ; "
        BTFSC STATUS,Z
        BRA FLASH_LED2_ALM
        BCF PORTA,led_red_anode                     ;Set LED #1 red anode low
        BCF PORTA,led_green_anode                   ;Set LED #1 green anode low
FLASH_LED2_ALM
        MOVLW bat2_alm_mask                         ;Are there any alarms against battery #2?
        ANDWF ALARM_STAT1,W                         ; "
        BTFSC STATUS,Z
        BRA CTRL_LEDS_RET                           ;Leave routine
        BRA SET_LED2_OFF                            ;No, leave routine
SET_LEDS_OFF
        BCF PORTA,led_red_anode                     ;Set LED #1 red anode low
        BCF PORTA,led_green_anode                   ;Set LED #1 green anode low
SET_LED2_OFF
        BCF PORTB,led_red_anode                     ;Set LED #2 red anode low
        BCF PORTB,led_green_anode                   ;Set LED #2 green anode low
CTRL_LEDS_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;CTRL_PARAM_INFO_TX
;This piece of code processes the transmission of the parameter string to the serial port. It transfers the 
;parameters and formats the ASCII string.

CTRL_PARAM_INFO_TX
        BTFSC SERIAL_STAT,tx_cycle_on               ;Is the transmit cycle already happening?     
        BRA CTRL_PARAM_INFO_TX_RET                  ;Yes, exit routine   
        BTFSC SERIAL_STAT,send_parameters           ;No, Is it requested to send parameter string?
        BRA TREAT_PARAM_TX                          ;Yes, go prepare the parameter string   
        BRA CTRL_PARAM_INFO_TX_RET                  ;No, leave routine
TREAT_PARAM_TX
        CLRF TEMP2_HI                               ;Clear TEMP2_HI. Will be used as a counter
        LFSR FSR1,CYCLE_STAT                        ;Load the Parameters' base address in FSR1 register
        LFSR FSR2,SERIAL_TX_BUFF                    ;Load Serial Tx buffer's base address in FSR2 register
        MOVLW "P"                                   ;Load "P" character
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW firmware_version                      ;Load the firmware version value in TEMP1_HI 
        MOVWF TEMP1_HI                              ; "
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
PARAM_TX_XFER_LOOP
        INCF TEMP2_HI,F                             ;Increment parameter counter
        MOVFF POSTINC1,TEMP1_HI                     ;Load the current parameter register in TEMP1_HI                
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
        MOVLW number_param_to_stor                  ;Has the maximum number of parameters to transfer been reached
        CPFSEQ TEMP2_HI                             ; "
        BRA PARAM_TX_XFER_LOOP                      ;No, loop to treat the next parameter
SEND_CR_LF
        MOVLW "\r"                                  ;Yes, Load CR
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW "\n"                                  ;Load LF
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW 00h                                   ;Load Null
        MOVWF POSTINC2                              ; into Tx buffer
END_TX_COMMAND
        BSF SERIAL_STAT,new_tx_cycle                ;Raise new TX cycle flag
        BCF SERIAL_STAT,send_parameters             ;Clear the parameter string request flag       
        BSF SERIAL_STAT,tx_cycle_on                 ;Raise the transmit cycle flag       
        BSF PIE1,TXIE                               ;Enable TX interrupts              
CTRL_PARAM_INFO_TX_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;SEND_LOG_INFO
;This piece of code processes the transmission of the Log message and clock strings to the serial port. 

SEND_LOG_INFO
        BTFSC SERIAL_STAT,tx_cycle_on               ;Is a parameter transmit cycle already happening?     
        BRA SEND_LOG_INFO_RET                       ;Yes, exit routine   
        BTFSC SERIAL_STAT,send_log_req              ;No, Is it requested to send the Log info?
        BRA TREAT_LOG_TX                            ;Yes, go prepare the parameter string   
        BTFSC SERIAL_STAT,send_current_clock        ;No, Is it requested to send the current clock?
        BRA TREAT_CLOCK_TX                          ;Yes, go prepare the clock string   
        BRA SEND_LOG_INFO_RET                       ;No, leave routine
TREAT_LOG_TX
        LFSR FSR1,LOG_BUFF                          ;Load the Parameters' base address in FSR1 register
        LFSR FSR2,SERIAL_TX_BUFF                    ;Load Serial Tx buffer's base address in FSR2 register
        MOVLW "L"                                   ;Load "L" character
        MOVWF POSTINC2                              ; into Tx buffer
LOG_TX_XFER_LOOP
        MOVFF POSTINC1,TEMP1_HI                     ;Load the current parameter register in TEMP1_HI                
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
        MOVLW low LOG_BUFF + log_buff_size ;+ .1     ;Has the bottom of the log buffer been reached?
        CPFSEQ FSR1L                                ; "
        BRA LOG_TX_XFER_LOOP                        ;No, loop to treat the next parameter
        BCF SERIAL_STAT,send_log_req                ;Clear the send log request flag       
		BRA SEND_LOG_CR_LF							;Yes, go load CR
TREAT_CLOCK_TX
        LFSR FSR2,SERIAL_TX_BUFF                    ;Load Serial Tx buffer's base address in FSR2 register
        MOVLW "K"                                   ;Load "L" character
        MOVWF POSTINC2                              ; into Tx buffer
		MOVFF CLOCK_REG_U,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
		MOVFF CLOCK_REG_H,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
		MOVFF CLOCK_REG_L,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
        BCF SERIAL_STAT,send_current_clock        	;Clear the send the current clock flag
SEND_LOG_CR_LF
        MOVLW "\r"                                  ;Yes, Load CR
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW "\n"                                  ;Load LF
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW 00h                                   ;Load Null
        MOVWF POSTINC2                              ; into Tx buffer
END_LOG_TX_COMMAND
        BSF SERIAL_STAT,new_tx_cycle                ;Raise new TX cycle flag
        BSF SERIAL_STAT,tx_cycle_on                 ;Raise the transmit cycle flag       
        BSF PIE1,TXIE                               ;Enable TX interrupts              
SEND_LOG_INFO_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;SEND_VOLT_CURR_INFO
;This piece of code processes the transmission of the Log message and clock strings to the serial port. 

SEND_VOLT_CURR_INFO
        BTFSC SERIAL_STAT,tx_cycle_on               ;Is a parameter transmit cycle already happening?     
        BRA SEND_VI_INFO_RET                       ;Yes, exit routine   
        BTFSS SERIAL_STAT,send_current_v_i          ;No, Is it requested to send the Log info?
        BRA SEND_VI_INFO_RET                       ;No, leave routine
        LFSR FSR2,SERIAL_TX_BUFF                    ;Load Serial Tx buffer's base address in FSR2 register
        MOVLW "V"                                   ;Load "L" character
        MOVWF POSTINC2                              ; into Tx buffer
SEND_RES_VOLT
		MOVFF RES_VOLT_HI,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
		MOVFF RES_VOLT_LO,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
SEND_BAT_VOLT
		MOVFF BAT_VOLT_HI,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
		MOVFF BAT_VOLT_LO,TEMP1_HI
        CALL HEX_2_ASCII                            ;Convert to two ASCII characters
        MOVFF TEMP1_HI,POSTINC2                     ;Conversion routine returns TEMP1_HI/TEMP1_HI. Load first Hex char into Tx buffer.
        MOVFF TEMP1_LO,POSTINC2                     ; and second hex char into Tx buffer.
        MOVLW "\r"                                  ;Yes, Load CR
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW "\n"                                  ;Load LF
        MOVWF POSTINC2                              ; into Tx buffer
        MOVLW 00h                                   ;Load Null
        MOVWF POSTINC2                              ; into Tx buffer
        BCF SERIAL_STAT,send_current_v_i                ;Clear the send log request flag       
        BSF SERIAL_STAT,new_tx_cycle                ;Raise new TX cycle flag
        BSF SERIAL_STAT,tx_cycle_on                 ;Raise the transmit cycle flag       
        BSF PIE1,TXIE                               ;Enable TX interrupts              
SEND_VI_INFO_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;CTRL_COMMAND_RX
;This piece of code processes the user commands received on the serial port and updates the parameters whenever necessary.

CTRL_COMMAND_RX
        BTFSC SERIAL_STAT,tx_cycle_on               ;Is a Tx cycle being processed?
        BRA USART_RX_RET                            ;Yes, leave (Indirect addressing reg. FSR2 shared with CTRL_TX_STRING)
        BTFSS SERIAL_STAT,lf_eoc_received           ;No, has end of command flag (LF) been detected by interrupt routine
        BRA USART_RX_RET                            ;No, leave routine     
        BCF PIE1,RCIE                               ;Yes, disable RX interrupts                 
VALID_RX_COMMAND
        CLRF TEMP2_HI                               ;Yes, clear the TEMP2_HI variable. Will be used as a counter
        LFSR FSR1,CYCLE_STAT                        ;Load the Parameters' first address in FSR1 register
        LFSR FSR2,SERIAL_RX_BUFF                    ;Load Serial rx buffer's base address in FSR2 register
        MOVFF POSTINC2,TEMP1_LO                     ;Transfer first char into temp variable
TST_U_COMMAND
        MOVLW "U"                                   ;Load character U into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the parameters update command?
        BRA TST_L_COMMAND                           ;No, go to next validation
        CLRF TEMP2_HI                               ;Clear the TEMP2_HI variable. Will be used as a counter
U_XFER_LOOP
        INCF TEMP2_HI,F                             ;Increment the TEMP2_HI counter
        MOVFF POSTINC2,TEMP1_HI                     ;Transfer high character of hex into TEMP1_HI
        MOVFF POSTINC2,TEMP1_LO                     ;Transfer low character of hex into TEMP1_LO
        CALL ASCII_2_HEX                            ;Convert to hex value. Zero Flag is valid here.
        MOVFF TEMP1_HI,POSTINC1                     ;Transfer hex value into the parameter variable
        MOVLW number_param_to_xfer                  ;Has the maximum number of parameters to store been reached
        CPFSEQ TEMP2_HI                             ; "       
        BRA U_XFER_LOOP                             ;No, loop to continue to treat the U command transfer
		CALL CALC_BAT_VARS							;Update the variables using the coefficients sent by the user
        BSF GENERAL1_STAT,write_cycle_req           ;Yes, set the flag to write the EEPROM with the DAC value and all other parameters
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_L_COMMAND       
        MOVLW "L"                                   ;Load character L into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the Send Log command?
        BRA TST_S_COMMAND                           ;No, go to next validation
        BSF SERIAL_STAT,send_log_req                ;Request to send the whole log to the serial Port
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_S_COMMAND       
        MOVLW "S"                                   ;Load character S into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the Send parameters command?
        BRA TST_K_COMMAND                           ;No, go to next validation
        BSF SERIAL_STAT,send_parameters             ;Yes, set the send parameters information flag
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_K_COMMAND       
        MOVLW "K"                                   ;Load character K into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the Send current Clock value?
        BRA TST_V_COMMAND                           ;No, go to next validation
        BSF SERIAL_STAT,send_current_clock          ;Yes, set the send current clock information flag
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_V_COMMAND       
        MOVLW "V"                                   ;Load character V into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the Send current Voltage/Current value?
        BRA TST_C_COMMAND                           ;No, go to next validation
        BSF SERIAL_STAT,send_current_v_i            ;Yes, set the send current clock information flag
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_C_COMMAND       
        MOVLW "C"                                   ;Load character C into Wreg
        CPFSEQ TEMP1_LO                             ;Is it the Clear Log command?
        BRA TST_R_COMMAND                           ;No, go to next validation
CLEAR_BUFF_VARIABLES                                ;Clear all variables, except the RS232 buffers and the EEPROM variables
        LFSR FSR0,LOG_BUFF                          ;Load indirect addressing register with base address of the variables to clear
CLR_BUFF_VARS_LOOP
        CLRF POSTINC0                               ;Clear variable and increase indirect addressing pointer
        MOVLW low LOG_BUFF + log_buff_size + 1      ;Have it reached end of buffer addresses
        CPFSEQ FSR0L                                ; "
        BRA CLR_BUFF_VARS_LOOP                      ;No, loop again. 
        CLRF ALARM_STAT1                            ;Clear the alarm register
        LFSR FSR0,LOG_BUFF                          ;Reset FSR0 (the log write indirect pointer)
        BRA CLR_RX_BUFF                             ;go empty rx buffer
TST_R_COMMAND
        MOVLW "R"                                   ;Load character R into Wreg
        CPFSEQ TEMP1_LO                             ;Is it Reset command?
        BRA CLR_RX_BUFF                             ;No, go empty rx buffer
        RESET                                       ;Execute device reset
CLR_RX_BUFF
        BCF SERIAL_STAT,lf_eoc_received             ;Clear the end of command flag
        LFSR FSR2,SERIAL_RX_BUFF                             ;load Serial rx buffer's base address in FSR register.
CLR_RX_BUFF_LOOP
        CLRF POSTINC2                               ;Clear Rx buffer cell and increase indirect addressing pointer
        MOVLW low SERIAL_TX_BUFF                    ;Has the whole Rx buffer been erased?
        CPFSEQ FSR2L                                ; "
        BRA CLR_RX_BUFF_LOOP                        ;No, loop again.
USART_RX_RET                                        
        BSF PIE1,RCIE                               ;Yes, enable RX interrupts                    
USART_RX_NOINT_RET
        RETURN                                      ;exit routine

;******************************************************************************************************
;CTRL_PARAM_UPD_EEPROM
;This portion of code writes the parameters to data EEPROM. This is a multi-pass process. Before the
;next byte write cycle can be processed, s/w must wait for the previous byte write cycle to complete.

CTRL_PARAM_UPD_EEPROM
        BTFSS GENERAL1_STAT,write_cycle_req         ;Is a Parameter Write cycle required or in progress?
        BRA UPDATE_PARAM_RET                        ;No, leave routine
        BTFSC SERIAL_STAT,tx_cycle_on               ;Is a Tx cycle being processed?
        BRA UPDATE_PARAM_RET                        ;Yes, leave (EEPROM access of Data and Program must not happen concurrently)
        BTFSC EECON1,WR                             ;Yes, is Write cycle happening
        BRA UPDATE_PARAM_RET                        ;Yes, leave routine
EXECUTE_WRITE                    
        CLRF FSR1H
        MOVFF EEPROM_WRITE_FSR1L_SAVE,FSR1L
        MOVLW .0                                    ;No, is is time to load the valid Data Flag (00) parameter in the first location?
        CPFSEQ WRT_EEPROM_POS                       ; "
        BRA WRITE_NON_FLAG                                  ;No, go write next parameter
        CLRF TEMP1_LO                               ;Yes, load 0 in TEMP1_LO
        BRA WRITE_EEPROM_NOW                        ;Go write parameter in EEPROM
WRITE_NON_FLAG
        MOVFF POSTINC1,TEMP1_LO       
WRITE_EEPROM_NOW
        MOVF WRT_EEPROM_POS,W
        CALL WRITE_EEPROM_DATA                      ;Call EEPROM writing routine
        INCF WRT_EEPROM_POS,F                       ;Increment EEPROM Writing Position
        MOVFF FSR1L,EEPROM_WRITE_FSR1L_SAVE
        MOVLW number_param_to_stor + .1                  ;Is this the last address (parameter) to write?
        CPFSEQ WRT_EEPROM_POS                       ; "
        BRA UPDATE_PARAM_RET                        ;No, Leave routine
FORCE_END_WRITE_CYCLE
        MOVLW low CYCLE_STAT
        MOVWF EEPROM_WRITE_FSR1L_SAVE
        BCF GENERAL1_STAT,write_cycle_req           ;Clear the write cycle flag     
        CLRF WRT_EEPROM_POS                         ;Clear write position 
UPDATE_PARAM_RET
        RETURN                                      ;Leave routine

;******************************************************************************************************
;CTRL_TIMER1_50MS

CTRL_TIMER1_50MS
        BTFSS PIR1,TMR1IF                           ;Is Timer1 overflow interrupt flag set?        
        BRA CTRL_TIMER1_50MS_RET                    ;No, leave routine
        BCF PIR1,TMR1IF                             ;Yes, clear Timer1 interrupt request flag        
        MOVLW t1_adjust_val_l                       ;Add the adjustment value to Timer1 (low byte) so that it rollovers every 50 milliseconds
        ADDWF TMR1L,W                               ;
        MOVWF TEMP1_HI                              ;Save result temporarily
        MOVLW t1_adjust_val_h                       ;Add adjustment (high byte) to Timer1 with carry flag from low byte addition
        ADDWFC TMR1H,F                              ; "
        MOVFF TEMP1_HI,TMR1L                        ;Now write Timer1 low byte. This updates the whole 16 bits        
		INCF ACTION_DELAY_CTR,F						;Increment the action delay counter
        INCF BUTT_DEBOUNCING_CTR,F                  ;Increment the pushbutton debouncing counter
        INCF LED_FLASH_CTR,F                        ;Increment the LED flash counter
		INCF PAUSE_OUTS_OFF_CTR,F					;Increment the pause with outputs off counter
        INCF IDLE_VOLT_CHK_CTR,F                    ;Increment the idle voltage check counter
CTRL_TIMER1_50MS_RET
		RETURN										;Leave routine

;******************************************************************************************************
;WRITE_MSG_LOG

WRITE_MSG_LOG
        MOVWF INDF0                                 ;Load Log message into the log buffer cell
        SWAPF INDF0,F
APPEND_CLK_TIMESTAMP                            
        MOVF CLOCK_REG_U,W                         ;Swap the Clock register U byte and load into Wreg
        IORWF POSTINC0,F                               ;Do an OR with the same log buffer cell to load the U byte
        MOVFF CLOCK_REG_H,POSTINC0                  ;Add the cycle clock timestamp to the log buffer
        MOVFF CLOCK_REG_L,POSTINC0                  ; "
CHK_LOG_BUFF_FULL
     	CLRF TEMP1_HI
    	MOVLW low LOG_BUFF + log_buff_size + .3
     	CPFSLT FSR0L
     	BRA FIFO_LOG_LOOP2                             ;Yes, go shift Buffer to free up 3 cells for next entry.
     	BRA WRITE_MSG_LOG_RET
FIFO_LOG_LOOP2
     	LFSR FSR0,LOG_BUFF+.1
FIFO_LOG_LOOP1
     	MOVFF POSTDEC0,POSTINC0
     	INCF FSR0L,F
     	MOVLW low LOG_BUFF + log_buff_size + .3
     	CPFSEQ FSR0L
     	BRA FIFO_LOG_LOOP1       
INC_LOOP_2
     	DECF FSR0L,F
     	CLRF INDF0
     	INCF TEMP1_HI,F
     	MOVLW .3
     	CPFSEQ TEMP1_HI
     	BRA FIFO_LOG_LOOP2
     	MOVLW low LOG_BUFF + log_buff_size
     	MOVWF FSR0L
WRITE_MSG_LOG_RET
		RETURN										;Leave routine

;******************************************************************************************************
;CALC_BAT_VARS
;This code converts the values sent by the Windows tool into the right variables used in the calculations.

CALC_BAT_VARS
CALC_CYCLE_DURATION
		MOVLW one_day_equiv_coeff_h
		MOVWF M1_H
		MOVLW one_day_equiv_coeff_l
		MOVWF M1_L
		MOVFF CYCLE_DURATION,M2_L
		CALL MULT_16B_x_8B
;	CLRF CYCLE_CLOCK_LIM_U		;Used for short Cycle tests
;	CLRF CYCLE_CLOCK_LIM_H		;Used for short Cycle tests
;	MOVLW .02					;Used for short Cycle tests
;	MOVWF CYCLE_CLOCK_LIM_L		;Used for short Cycle tests
		MOVFF R1_U,CYCLE_CLOCK_LIM_U
		MOVFF R1_H,CYCLE_CLOCK_LIM_H
		MOVFF R1_L,CYCLE_CLOCK_LIM_L
CALC_BAT1_CHG_LIM
        MOVFF BAT1_CHRG_30M_SLICE_INDEX,TEMP1_HI
        INCF TEMP1_HI,F
		MOVLW thirty_min_equiv_coeff
		MULWF TEMP1_HI
;	CLRF CLOCK_CHARGE_LIM1_H			;Used for short Cycle tests
;	MOVLW .01							;Used for short Cycle tests
;	MOVWF CLOCK_CHARGE_LIM1_L			;Used for short Cycle tests
		MOVFF PRODH,CLOCK_CHARGE_LIM1_H		
		MOVFF PRODL,CLOCK_CHARGE_LIM1_L
CALC_BAT2_CHG_LIM
        MOVFF BAT2_CHRG_30M_SLICE_INDEX,TEMP1_HI
        INCF TEMP1_HI,F
		MOVLW thirty_min_equiv_coeff
		MULWF TEMP1_HI
;	CLRF CLOCK_CHARGE_LIM2_H			;Used for short Cycle tests
;	MOVLW .01							;Used for short Cycle tests
;	MOVWF CLOCK_CHARGE_LIM2_L			;Used for short Cycle tests
		MOVFF PRODH,CLOCK_CHARGE_LIM2_H		
		MOVFF PRODL,CLOCK_CHARGE_LIM2_L
CALC_BAT1_CHG_CURR
        MOVLW high CURRENT_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low CURRENT_COEFFS_LOCN
        MOVWF TBLPTRL
		RLNCF BAT1_CHRG_CURR_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVFF TABLAT,TARGET_CHRG_DROP1_H
        TBLRD*+
        MOVFF TABLAT,TARGET_CHRG_DROP1_L
CALC_BAT2_CHG_CURR
        MOVLW high CURRENT_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low CURRENT_COEFFS_LOCN
        MOVWF TBLPTRL
		RLNCF BAT2_CHRG_CURR_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVFF TABLAT,TARGET_CHRG_DROP2_H
        TBLRD*+
        MOVFF TABLAT,TARGET_CHRG_DROP2_L
CALC_BAT1_DISCHG_CURR
        MOVLW high CURRENT_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low CURRENT_COEFFS_LOCN
        MOVWF TBLPTRL
		RLNCF BAT1_DISCHRG_CURR_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVFF TABLAT,TARGET_DISCHRG_DROP1_H
        TBLRD*+
        MOVFF TABLAT,TARGET_DISCHRG_DROP1_L
CALC_BAT2_DISCHG_CURR
        MOVLW high CURRENT_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low CURRENT_COEFFS_LOCN
        MOVWF TBLPTRL
		RLNCF BAT2_DISCHRG_CURR_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVFF TABLAT,TARGET_DISCHRG_DROP2_H
        TBLRD*+
        MOVFF TABLAT,TARGET_DISCHRG_DROP2_L
CALC_BAT1_DISCHG_VOLT
        MOVLW high VOLTAGE_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low VOLTAGE_COEFFS_LOCN
        MOVWF TBLPTRL
		MOVF BAT1_DISCHRG_VOLT_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVF BAT1_NUM_CELLS,W
        MULWF TABLAT
		MOVFF PRODH,DISCHRG_LIM1_HI
		MOVFF PRODL,DISCHRG_LIM1_LO
CALC_BAT2_DISCHG_VOLT
        MOVLW high VOLTAGE_COEFFS_LOCN
        MOVWF TBLPTRH
        MOVLW low VOLTAGE_COEFFS_LOCN
        MOVWF TBLPTRL
		MOVF BAT2_DISCHRG_VOLT_INDEX,W
        ADDWF TBLPTRL,F
        BTFSC STATUS,C
        INCF TBLPTRH,F
        TBLRD*+
        MOVF BAT2_NUM_CELLS,W
        MULWF TABLAT
		MOVFF PRODH,DISCHRG_LIM2_HI
		MOVFF PRODL,DISCHRG_LIM2_LO
CALC_BAT1_MAX_DISCH_DUR
        MOVLW one_hour_equiv_coeff
		MULWF BAT1_MAX_DISCH_HOUR_LIM
		MOVFF PRODH,CLOCK_DISCHG_LIM1_H		
		MOVFF PRODL,CLOCK_DISCHG_LIM1_L
CALC_BAT2_MAX_DISCH_DUR
        MOVLW one_hour_equiv_coeff
		MULWF BAT2_MAX_DISCH_HOUR_LIM
		MOVFF PRODH,CLOCK_DISCHG_LIM2_H		
		MOVFF PRODL,CLOCK_DISCHG_LIM2_L
CALC_BAT_VARS_RET
		RETURN

;******************************************************************************************************
;HEX_2_ASCII
;This code converts an hexadecimal byte to the equivalent two ASCII characters.

HEX_2_ASCII                                         ;The hex value is expected to be in TEMP1_HI when the routine is called
        MOVLW .58                                   ;Load ASCII equivalent of 10 into temporary var.
        MOVWF TEMP3_LO            
LO_NIBBLE
        MOVF TEMP1_HI,W                             ;Move hex value into Wreg
        ANDLW h'0F'                                 ;isolate lower nibble
        ADDLW .48                                   ;add ASCII ofset
        CPFSGT TEMP3_LO                             ;verify if ASCII char is higher than "9"
        ADDLW .7                                    ;yes, add additional ASCII offset
        MOVWF TEMP1_LO                              ;No, low ASCII characters is returned in TEMP1_LO
HI_NIBBLE
        SWAPF TEMP1_HI,W                            ;load nibble-swapped hex value into Wreg
        ANDLW h'0F'                                 ;isolate lower nibble
        ADDLW .48                                   ;add ASCII ofset
        CPFSGT TEMP3_LO                             ;verify if ASCII char is higher than "9"
        ADDLW .7                                    ;yes, add additional ASCII offset
        MOVWF TEMP1_HI                              ;No, high ASCII characters is returned in TEMP1_HI   
        RETURN                                      ;Leave routine. Result are in TEMP1_HI and TEMP1_LO

;******************************************************************************************************
;ASCII_2_HEX  
;This code converts two ASCII characters representing a hexadecimal byte to its numerical equivalent.

ASCII_2_HEX                                         ; The ASCII characters are expected to be in TEMP1_HI and TEMP1_LO when the routine is called
        MOVLW .10                                    ;load the offset required to test for gap between ASCII code for 9 and A
        MOVWF TEMP3_LO            
HIGH_CHAR
        MOVLW .48                                   ;load ASCII offset equiv. into Wreg
        SUBWF TEMP1_HI,W                            ;substract 48 from ASCII character
        CPFSGT TEMP3_LO                             ;is ASCII character higher than "9" (like A,B,C,D,E,F)
        ADDLW .249                                  ;yes, substract an additional ASCII offset of 7 (adding 249 does the same)
        MOVWF TEMP1_HI                              ;no, load result into TEMP1_HI
        SWAPF TEMP1_HI,F                            ;swap nibbles to move to higher hex position   
LO_CHAR
        MOVLW .48                                   ;load ASCII offset equiv. into Wreg
        SUBWF TEMP1_LO,W                            ;substract 48 from ASCII character
        CPFSGT TEMP3_LO                             ;is ASCII character higher than "9" (like A,B,C,D,E,F)
        ADDLW .249                                  ;yes, substract an additional ASCII offset of 7 (adding 249 does the same)
        ADDWF TEMP1_HI,F                            ;add the result to the higher hex result 
        RETURN                                      ;Leave routine. Result is in TEMP1_HI

;******************************************************************************************************
;MULT_16B_x_8B
;This sub-routine executes an unsigned 16-bit x 8-bit multiply
;16-bit value is M1_H/L, 8-bit value is M2_L and 24-bit result is R1_U/H/L
; Actual algoritm is: R1 = M1 * M2 = (256 * M1_H * M2_L) + (M1_L x M2_L)

MULT_16B_x_8B
		CLRF R1_L                                   ;First clear the multiply result registers
		CLRF R1_H                                   ; "
		CLRF R1_U                                   ; "
		MOVF M1_H,W                                 ;Load the 8 MSbs of M1 in Wreg 
		MULWF M2_L                                  ;Multiply Wreg by the 8 LSb of M2
		MOVFF PRODH,R1_U                            ;Transfer multiply result into highest 16 bits of R1
		MOVFF PRODL,R1_H                            ; "
		MOVF M1_L,W                                 ;Load the 8 LSbs of M1 in Wreg
		MULWF M2_L                                  ;Multiply Wreg by the 8 LSb of M2
		MOVF PRODL,W                                ;Transfer 8 LSbs of result into Wreg
		ADDWF R1_L,F                                ;Add those to 8 LSbs of multiply result R1 
		MOVF PRODH,W                                ;Transfer 8 MSbs of multiply result into Wreg
		ADDWFC R1_H,F                               ;Add Wreg to 8 middle bits of R1, including carry flag    
		MOVLW .0                                    ;Add carry flag to the highest bits of R1
		ADDWFC R1_U,F                               ; "
		RETURN                                      ;Leave routine

;******************************************************************************************************
;READ_EEPROM_DATA
;This sub-routine reads one byte of Data EEPROM

READ_EEPROM_DATA
        MOVWF EEADR                                 ;Address to read is in Wreg  
        BCF EECON1, CFGS                            ;Point to DATA memory
        BCF EECON1,EEPGD                            ;Point to DATA memory
        BSF EECON1,RD                               ;EEPROM Read
        MOVF EEDATA,W                               ;value read is moved to Wreg 
        RETURN          

;******************************************************************************************************
;WRITE_EEPROM_DATA
;This sub-routine writes one byte of Data EEPROM

WRITE_EEPROM_DATA
        MOVWF EEADR                                 ;Data Memory Address to write is in Wreg
        MOVFF TEMP1_LO,EEDATA                       ;Data memory value is in TEMP1_LO
        BCF EECON1, CFGS                            ;Point to DATA memory
        BCF EECON1, EEPGD                           ;Point to DATA memory
        BCF INTCON, GIE                             ;Disable Interrupts
        BSF EECON1, WREN                            ;Enable writes
        MOVLW h'55'                     
        MOVWF EECON2                                ;Write 55h
        MOVLW h'AA'
        MOVWF EECON2                                ;Write AAh
        BSF EECON1, WR                              ;Set WR bit to begin write
        BCF EECON1, WREN                            ;Enable writes
        BSF INTCON, GIE                             ;Enable Interrupts
        RETURN

;******************************************************************************************************
		;The following program memory parameters make a lookup table of 46 16-bit current monitor values corresponding to the 51 current choices in the Windows tool for a 10 ohm resistor
        ORG 0x0E00
CURRENT_COEFFS_LOCN 
        ; 5 Ohm coefficients
        DB 0x00, 0x0D, 0x00, 0x0F, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1A, 0x00, 0x1C, 0x00, 0x1F
        DB 0x00, 0x21, 0x00, 0x24, 0x00, 0x26, 0x00, 0x29, 0x00, 0x2C, 0x00, 0x2E, 0x00, 0x31, 0x00, 0x33
        DB 0x00, 0x36, 0x00, 0x38, 0x00, 0x3B, 0x00, 0x3D, 0x00, 0x40, 0x00, 0x43, 0x00, 0x45, 0x00, 0x48
        DB 0x00, 0x4A, 0x00, 0x4D, 0x00, 0x4F, 0x00, 0x52, 0x00, 0x54, 0x00, 0x57, 0x00, 0x5A, 0x00, 0x5C
        DB 0x00, 0x5F, 0x00, 0x61, 0x00, 0x64, 0x00, 0x66, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x6E, 0x00, 0x71
        DB 0x00, 0x73, 0x00, 0x76, 0x00, 0x78, 0x00, 0x7B, 0x00, 0x7D, 0x00, 0x80

VOLTAGE_COEFFS_LOCN
		;The following parameters make a lookup table of 15 8-bit voltage monitor values corresponding to the 27 voltage choices in the Windows tool
		DB 0x0A, 0x0D, 0x0F, 0x12, 0x14, 0x17, 0x1A, 0x1C, 0x1F, 0x21, 0x24, 0x26, 0x29, 0x2C, 0x2E, 0x31 
        DB 0x33, 0x36, 0x38, 0x3B, 0x3D, 0x40, 0x43, 0x45, 0x48, 0x4A, 0x4D

;******************************************************************************************************

;Default parameter values added into Data EEPROM locations, starting at address 0x00
        ORG 0xF00000
PARAMETERS_LOCN
		;Make sure the "number_param_to_stor" and "number_param_to_xfer" constant is updated when changing the number of parameters saved.
		;  NULL , CYCLE_STAT , ALARM_STAT1 , CYCLE_DURATION , BAT1_CHRG_30M_SLICES , BAT2_CHRG_30M_SLICES , BAT1_CHRG_CURR_INDEX , BAT2_CHRG_CURR_INDEX 
		DE 0x00	,    0x00    ,    0x00     ,     0x15       ,        0x17          ,         0x17         ,			0x10         ,        0x03     
		;   BAT1_DISCHRG_CURR_INDEX , BAT2_DISCHRG_CURR_INDEX , BAT1_DISCHRG_VOLT_INDEX , BAT2_DISCHRG_VOLT_INDEX , MAX_CONDIT_CYCLES , BAT1_CAPACITY , BAT2_CAPACITY , BAT1_NUM_CELLS 
		DE           0x23           ,         0x0B            ,           0x09          ,          0x09           ,       0x03        ,     0x14      ,     0x14       ,     0x06 
		;   BAT2_NUM_CELLS , DEFAULTS_CHECKED , BAT1_MAX_DISCH_HOUR_LIM , BAT1_MAX_DISCH_HOUR_LIM , STAGE_STAT , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK , BLANK
        DE       0x06      ,      0x03        ,           0x08          ,           0x08          ,    0x01    ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF ,  0xFF
		;Then we fill the remaining data EEPROM values (except the last two) with FF. This makes the PIC burner happy.
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
        DE 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF

;******************************************************************************************************

;The last Data EEPROM value stores the SW VERSION to allow to read the software version from PICMicro burner,  
;whitout running the software. It is not saved or retrieved by software. It is located in the last Data EEPROM Byte.
;             BLANK  ,   SW VERSION
        DE    0xFF   ,      .1

;******************************************************************************************************

;End of program
        END
