/*********************************************

J. Koehler
Comox, BC, Canada

All rights reserved

Program to run the automatic noise figure indicator electronics.

PB6 is an output and goes to the MOSFET switch.  When it is high, +28VDC
appears at the 'Noise Source' output connector.

AD0 is the A/D input connected to the AD8307 logarithmic detector chip

AD1 is connected to the front panel switch.  The voltage appearing here
depends on the switch position:
`On    0VDC
 Auto  2.5 VDC
 Off   5.0 VDC
 
The mode of the operation is set through the serial port.  Sending a single 'T',
(upper or lower case) sets it to a mode where the excess noise is displayed as
a temperature.  This is the default after being programmed.  Sending a 'B' so it
changes the mode to one where the excess noise is displayed as a noise figure in
dB.  The mode is stored in EEPROM so it stays in whatever mode it was last set to.
 
The serial port is also used for calibration ('c'), for setting the ENR of ths noise
source ('e') or for displaying ('d') the internal stored values.

4 May, 2002  The circuit was modified by adding a 4.7K resistor/0.1 uF capacitor low-pass
filter to PD5 and making it an output.  Then code was added to make this a 10-bit PWM
output and the value of (1024 * Ratio) modulus 1024 was sent to it.
 
14 July, 2006 started amending code to use GCC compiler and ATMega32
microprocessor with 14.7 MHz clock

19 July, 2006  Finished!
*********************************************/

#include <lcd.h>
#include <stdlib.h>
#include <math.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/eeprom.h>
#include <avr/sleep.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/io.h>


#define fcpu 14745600
#define BAUDRATE 19200
#include "uart0.c"
#define UART_char_has_come_in() rbuflen()

#define TurnNoiseOn PORTB = PORTB | (1 << PB6)
#define TurnNoiseOff PORTB = PORTB & ~(1 << PB6)
#define NumberAveraged 100
#define NumberOfRepetitions 50
#define NumberOfStoredCalibrationPoints 5
#define SetSwitchPressed ((PINB & (1 << PB7)) == 0)
#define CR 13
#define LF 10
#define T0_ON TCCR0 = 3
#define T0_OFF TCCR0 = 0

// global variables

char str[16];
float A, B, Pc_sys, Ph_sys, HotdBm, ColddBm, Ratio, T_e, T_sys, Gain;
float F_system, F_dut, ENR, Pc_dut, Ph_dut;
char MeasuredSystemParameters;
volatile int adc_value;
volatile long int tick;
char mode_RAM;
float T_hot_RAM;
float x_RAM[NumberOfStoredCalibrationPoints];
float y_RAM[NumberOfStoredCalibrationPoints];
static char mode __attribute__((section (".eeprom"))) = 'T';
static float T_hot __attribute__((section (".eeprom"))) = 13854.0;
static float x[NumberOfStoredCalibrationPoints] __attribute__((section (".eeprom")))
= {102.4, 204.8, 307.2, 409.6, 512.0};
static float y[NumberOfStoredCalibrationPoints] __attribute__((section (".eeprom"))) 
= {-70.0,-50.0, -30.0, -10.0, 10.0}; 

// prototype declarations

void initialize(void);
float GetAverageValue(void);	// the averaged channel 0 value, averaged over NumberAveraged readings
void DoLeastSquare(void);
void DoCalibrate(void);
void GetENR(void);
void Measure(void);
float GetNumber(void);		// gets float value from serial port
void DumpData(void);		// displays internal data stored
void delay_ms(int i);
int read_adc(unsigned char channel);
 
int main(void) {

  char ch;
  
  initialize();
  DoLeastSquare();
  

/*  int v;
  v = 0;
  while (1)
    {
    wdt_reset();
    lcd_clrscr(0);
    lcd_gotoxy(0, 0, 0);
    v = read_adc(1);
    itoa(v, str, 10);
    lcd_puts(str, 0);
    delay_ms(1000);
    }
*/

  while (1) {
  
    wdt_reset();
  
    if (SetSwitchPressed) {
      MeasuredSystemParameters = 1;
      lcd_clrscr(0);
      lcd_gotoxy(0, 0, 0);
      lcd_puts("Measuring ...", 0);
      Pc_sys = 0.0;
      Ph_sys = 0.0;
      for (ch = 0; ch < 10; ch++) {
        Measure();
        Pc_sys += pow(10.0, ColddBm / 10.0);
        Ph_sys += pow(10.0, HotdBm / 10.0);
        }
      Ratio = Ph_sys / Pc_sys;
      Pc_sys /= 10.0;
      Ph_sys /= 10.0;
      T_sys = (T_hot_RAM - (290.0 * Ratio)) / (Ratio - 1.0);
      F_system = ENR / (Ratio - 1.0);
      lcd_clrscr(0);
      lcd_gotoxy(0, 0, 0);
      dtostrf((double) HotdBm, 6, 2, str);
      lcd_puts("Hot dBm: ", 0);
      lcd_puts(str, 0);
      dtostrf((double)ColddBm, 6, 2, str);
      lcd_gotoxy(0, 1, 0);
      lcd_puts("Cold dBm: ", 0);
      lcd_puts(str, 0);
      delay_ms(5000);
      lcd_clrscr(0);
      lcd_gotoxy(0, 0, 0);
      lcd_puts("System Temp.", 0);
      lcd_gotoxy(0, 1, 0);
      dtostrf((double)T_sys, 10, 1, str);
      lcd_puts(str, 0);
      delay_ms(5000);
      }
    
    if (UART_char_has_come_in()) {
      ch = UART_getchar();
      switch (ch) {
        case 'c':
        case 'C': DoCalibrate();
                  //
                  // Then, recalcaulate the values for A and B
                  //
                  DoLeastSquare();
                  break;
        case 'e':
        case 'E': GetENR();
                  break;
        case 'd':
        case 'D': DumpData();
                  break;
        case 't':
        case 'T': mode_RAM = 'T';
                  eeprom_busy_wait();
                  eeprom_write_byte(&mode, mode_RAM);
                  break;
        case 'b':
        case 'B': mode_RAM = 'B';
                  eeprom_busy_wait();
                  eeprom_write_byte(&mode, mode_RAM);
                  break;
        default : break;
         }
      }
      
    if (read_adc(1) > 700) {	// switch is set to OFF
      TurnNoiseOff;
      ColddBm = A + (GetAverageValue() * B);
      dtostrf((double)ColddBm, 6, 2, str);
      lcd_clrscr(0);
      lcd_gotoxy(0, 0, 0);
      lcd_puts("Noise Off", 0);
      lcd_gotoxy(0, 1, 0);
      lcd_puts(str, 0);
      lcd_gotoxy(7, 1, 0);
      lcd_puts("dBm", 0);
      delay_ms(500);
      }
       
    if (read_adc(1) < 300) {	// switch is set to ON
      TurnNoiseOn;
      HotdBm = A + (GetAverageValue() * B);
      dtostrf((double)HotdBm, 6, 2, str);
      lcd_clrscr(0);
      lcd_gotoxy(0, 0, 0);
      lcd_puts("Noise On", 0);
      lcd_gotoxy(0, 1, 0);
      lcd_puts(str, 0);
      lcd_gotoxy(7, 1, 0);
      lcd_puts("dBm", 0);
      delay_ms(500);
      }
       
    if ((read_adc(1) >= 300) && (read_adc(1) <= 700)) {	// switch is set to AUTO
      if (MeasuredSystemParameters) {
        Measure();
        Ratio = pow(10.0, (HotdBm - ColddBm) / 10.0);
        Pc_dut = pow(10.0, ColddBm / 10.0);
        Ph_dut = pow(10.0, HotdBm / 10.0);
        F_dut = ENR / (Ratio - 1);	// noise figure of DUT + system in tandem
        OCR1A = ((int) (Ratio  * 1024.0)) % 1024;	// analogue output
        if ((Ratio - 1.0) > 0.1) {
	  Gain = (Ph_dut -Pc_dut) / (Ph_sys - Pc_sys);
// Now calculate the true noise figure, compensating for the gain and noise figure of measuring system
          F_dut -= ((F_system - 1) / Gain);	// gain correction
          T_e = (F_dut - 1) * 290.0;		// excess temperature of the DUT
          lcd_clrscr(0);
          lcd_gotoxy(0, 0, 0);
          if (mode_RAM == 'T') {
            lcd_puts("Te = ", 0);
            lcd_gotoxy(6, 0, 0);
            dtostrf((double)T_e, 10, 0, str);
            lcd_puts(str, 0);
            }
          else {
            lcd_puts("NF (dB) = ", 0);
            dtostrf((double)(10.0 * log10(F_dut)), 10, 2, str);
            lcd_puts(str, 0);
            }
          lcd_gotoxy(0, 1, 0);
          lcd_puts("Gain (dB): ", 0);
          dtostrf((double)( 10.0 * log10(Gain)), 5, 1, str);
          lcd_puts(str, 0);
          }
        else {
          lcd_clrscr(0);
          lcd_gotoxy(0, 0, 0);
          lcd_puts("Te too high", 0);
          }
        }
      else {
        lcd_clrscr(0);
        lcd_gotoxy(0, 0, 0);
        lcd_puts("Sys. parameters", 0);
        lcd_gotoxy(0, 1, 0);
        lcd_puts("not measured", 0);
        delay_ms(1000);
        }
      }
    }
  return 0;
  }

void delay_ms(int i)
  {
  tick = 0;
  while (tick < i) wdt_reset();
  
  }
void DumpData() {
/*

	Sends vales of internal readings to serial port
*/

  int i;
  
  OutStr("Value       Power\n\r");
  OutStr("(Volts)     (dBm)\n\r");
  for (i = 0; i < 5; i++) {
    dtostrf((double)(x_RAM[i] / 204.8), 10, 2, str);
    OutStr(str);
    OutStr("       ");
    dtostrf((double)y_RAM[i], 10, 2, str);
    OutStr(str);
    UART_putchar(CR);
    UART_putchar(LF);
    }
  UART_putchar(LF);
  OutStr("ENR = ");
  
  dtostrf((double)(10.0 * log10 ((T_hot_RAM - 290.0) /290.0)), 10, 2, str);
  OutStr(str);
  OutStr(" dBm\n\r");

  if (mode_RAM == 'T') OutStr ("Mode is temperature");
    else OutStr("Mode is dB");
  OutStr("\n\r");
  delay_ms(200);
  }
  
void Measure(void) {

 int n;

 for (n= 0; n < NumberOfRepetitions; n++) {
    TurnNoiseOn;
    delay_ms(1);
    HotdBm = A + (GetAverageValue() * B);
    TurnNoiseOff;
    delay_ms(1);
    ColddBm = A + (GetAverageValue() * B);
    }
  }

float GetNumber(void) {

  char ch;
  int i;
  
  for (i = 0; i < 16; i++) str[i] = 0;
  i = 0;
  ch = UART_getchar();
  while (ch != CR) {
    UART_putchar(ch);
    str[i] = ch;
    i++;
    ch = UART_getchar();
    }
    
  return atof(str); 
  }

void DoCalibrate(void){
/*
  	In this routine, the calibrate is done by communication
through the serial RS-232 port.  The pseudo-code is:

	For NumberOfStoredCalibrationPoints do
	  display what step it is;
	  get the dBm from the serial port
	  read the A/D converter value
	  store both numbers in the x andy arrays
*/

  int n;
  
  OutStr("Calibration procedure\n\r");
  
  for (n = 0; n < NumberOfStoredCalibrationPoints; n++) {
    OutStr("Data point #  ");
    itoa(n + 1, str, 10);
    OutStr(str);
    OutStr(": dBm: ");
    y_RAM[n] = GetNumber();
    OutStr(" press Set when ready");
    while (!SetSwitchPressed) wdt_reset();	// wait till switch is pressed
    UART_putchar(CR);
    UART_putchar(LF);
    x_RAM[n] =  GetAverageValue();
    }
    
  OutStr("Value table\n\r");
  
  for (n = 0; n < NumberOfStoredCalibrationPoints; n++) {
    dtostrf((double)y_RAM[n], 10, 2, str);
    OutStr(str);
    OutStr("     ");
    dtostrf((double)x_RAM[n], 10, 2, str);
    OutStr(str);
    UART_putchar(CR);
    UART_putchar(LF);
    }
    
  eeprom_busy_wait();
  eeprom_write_block(&x_RAM, &x, 20);
  eeprom_busy_wait();
  eeprom_write_block(&y_RAM, &y, 20);
  
  }
  

void GetENR(void){
/*

 Reads the ENR of the noise source from the serial RS-232 port, converts it to
 degrees Kelvin and stores it in eeprom
 
 */
   
  OutStr("Type ENR of source (dB): ");
  T_hot_RAM = 290.0 * (1.0 + pow(10.0, GetNumber() / 10.0));
  UART_putchar(CR);
  UART_putchar(LF);
  OutStr("Source Temp: ");
  dtostrf((double)T_hot_RAM, 10, 2, str);
  OutStr(str);
  OutStr(" degrees\n\r");
  eeprom_busy_wait();
  eeprom_write_block(&T_hot_RAM, &T_hot, 4);
  }


void DoLeastSquare(void) {
  
  float sx, sy, sxx, sxy, det;
  int i;
  
  sx = 0.0;
  sy = 0.0;
  sxx = 0.0;
  sxy = 0.0;
  for (i = 0; i < NumberOfStoredCalibrationPoints; i++) {
    sx += x_RAM[i];
    sy += y_RAM[i];
    sxy += x_RAM[i] * y_RAM[i];
    sxx += x_RAM[i] * x_RAM[i];
    }
  det = (NumberOfStoredCalibrationPoints * sxx) - (sx * sx);
  A = ((sy * sxx) - (sx * sxy)) / det;
  B = ((NumberOfStoredCalibrationPoints * sxy) - (sx * sy)) / det;
  }
   

float GetAverageValue(void) {

  float sum;
  int n;

  sum = 0.0;
  for (n = 0; n < NumberAveraged; n++) sum += read_adc(0);
  return sum / NumberAveraged;
  }

int read_adc(unsigned char channel)
  {
  unsigned char i;
  
  ADMUX = channel & 0x07;
/*
  ADCSRA |= (1 << ADSC);
  while (ADSRA & (1 << ADSC)) wdt_reset();
  return ADCW;
*/

  T0_OFF;
  set_sleep_mode(SLEEP_MODE_ADC);
  
  i = 0;
  while (i++ < 100) wdt_reset();
  
  sleep_mode();
  T0_ON;
  
  return adc_value;  
  }
  
void initialize(void) {


// Input/Output Ports initialization
// Port A
PORTA=0x00;
DDRA=0x00;

// Port B
DDRB=0x40;	// PB7 is an input, PB6 is an output
PORTB=0x80;	// set pull-up resistor for PB7

// Port C
PORTC=0x00;
DDRC=0x00;

// Port D
PORTD=0x00;
DDRD=0x20;	//PD5 is an output for PWM using OCR1A

// Timer/Counter 0 initialization
// Clock source: System Clock
// Clock value: Timer 0 enabled
// Mode: Output Compare
// OC0 output: Disconnected
OCR0 = 230;	// this gives interrupts about once a mS
TIMSK = 1 << OCIE0;
T0_ON;	// clock is divided by 64 giving 230.4 KHz

// Timer 1 used to do PWM
// it is clocked at cpu rate and uses 10 bits
TCCR1A = 0x83;  // PWM, 10 bit
TCCR1B = 0x01;	// clock divided by 1



// UART initialization
UART_first_init();	// initializes it to 19200 baud
OutStr("PANFI has been turned on\n\r");

// Analog Comparator initialization
// Analog Comparator: Off
// Analog Comparator Input Capture by Timer/Counter 1: Off
ACSR=0x80;

// ADC initialization
// ADC Clock frequency: 2457.600 kHz
ADCSRA = (1 << ADEN) | (1 << ADIE) | (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);

lcd_init(LCD_DISP_ON, 0);

// get eeprom values into their similar arrays in RAM
eeprom_busy_wait();
eeprom_read_block(&x_RAM, &x, 20);
eeprom_busy_wait();
eeprom_read_block(&y_RAM, &y, 20);
eeprom_busy_wait();
eeprom_read_block(&T_hot_RAM, &T_hot, 4);
eeprom_busy_wait();
mode_RAM = eeprom_read_byte(&mode);
ENR = 10.0 * log10 ((T_hot_RAM - 290.0) /290.0);

// Global enable interrupts
sei();

DumpData();

// enable watchdog timer every 2 seconds
wdt_enable(WDTO_2S);

}

SIGNAL (SIG_ADC)
  {
  adc_value = ADCW;
  }
  
SIGNAL (SIG_OUTPUT_COMPARE0)
  {
  TCNT0 = 0;
  wdt_reset();
  tick++;
  }
