/*
  Yaesu CAT Display
  for the Yaesu FT-450/FT-950/FT-1200 and others
  written by Glen Popiel - KW5GP

  Released under the GPLv3 license
*/

#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>  // SPI library so we can communicate with the TFT display
#include <AltSoftSerial.h>  // AltSoft Serial Library - better than the SoftwareSerial library

//#define debug // Uncomment to send debug information to the Serial Monitor

#define CAT_serial_rate 4800  // Define the radio CAT baud rate - Default is 4800 baud
#define comm_speed 9600 // Define the Arduino Serial Monitor baud rate
#define timeout_delay 5000  // The delay to signify a data timeout from the radio
#define update_delay 5000 // The delay between Arduino TFT display updates

#define TFT_CS     10  // Assign the TFT CS to pin 10
#define TFT_RST    7  // Assign the TFT RST to pin 7
#define TFT_DC     6  // Assign the TFT DC to pin 6

#define tft_delay 10  // set the TFT command delay to 10ms

#define get_VFO_A "FA;" // Get VFO-A command
#define get_VFO_B "FB;" // Get VFO-B command
#define get_vfo "VS;" // Get which VFO is selected
#define get_mode "MD0;"   // Get Mode command

boolean first_pass = true;  // Variable used to determine if first pass through the main loop
char rx_char; // Variable for each character received from the radio
unsigned long timeout, current_millis;  // Variables to handle communication timeout errors
bool receiving, timeout_error;  // Variables for the receive radio data process
String CAT_buffer;  // String to hold the received CAT data
String VFO_A, VFO_B, vfo, mode; // Variables to hold the parsed and formatted CAT data

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Initialize the TFT display instance

AltSoftSerial altSerial;  // Initialize the software serial instance

void setup()
{
#ifdef debug
  Serial.begin(comm_speed); // Set the Arduino Serial Monitor baud rate
  while (!Serial) ; // wait for Arduino Serial Monitor to open
  Serial.println("FT-450/900/1200 CAT Display Begin");
#endif

  //Start the Software Serial port
  altSerial.begin(CAT_serial_rate); // Start the software serial port to the transceiver

  tft.initR(INITR_18BLACKTAB);   // initialize a 1.8" TFT with ST7735S chip, black tab
  delay(tft_delay);
  tft.fillScreen(ST7735_BLACK); // Clear the display - fill with BLACK background
  delay(tft_delay);
  tft.setRotation(1); // Set the screen rotation
  delay(tft_delay);
  tft.setTextWrap(false); // Turn off Text Wrap
  delay(tft_delay);
  tft.setTextSize(3); // Set the Font Size
  delay(tft_delay);
  tft.setTextColor(ST7735_BLUE); //Set the Text Color
  delay(tft_delay);
  tft.setCursor(40, 10);  //Set the Cursor and display the startup screen
  delay(tft_delay);
  tft.print("KW5GP");
  delay(tft_delay);
  tft.setTextSize(2);
  delay(tft_delay);
  tft.setCursor(50, 60);
  delay(tft_delay);
  tft.print("Yaesu");
  delay(tft_delay);
  tft.setCursor(15, 80);
  delay(tft_delay);
  tft.print("Transceiver");
  delay(tft_delay);
  tft.setCursor(15, 100);
  delay(tft_delay);
  tft.print("CAT Display" );

  delay(5000);  //Wait 5 seconds then clear the startup message
  tft.fillScreen(ST7735_BLACK); // Clear the display
  delay(tft_delay);

  update_display(); // update the display
  first_pass = false; // turn off the update display first pass flag
}

void loop()
{
  send_command(get_VFO_A);  // Send the Get VFO-A Frequency Command

  get_response();  // Receive the VFO-A frequency response

  format_string();  // Format the VFO Response data for display
  VFO_A = CAT_buffer;

#ifdef debug
  Serial.print("VFO-A: ");
  Serial.println(VFO_A);
#endif

  send_command(get_VFO_B);  // Send the Get VFO-B Frequency Command


  get_response();  // Receive the VFO-B frequency response

  format_string();  // Format the VFO Response data for display
  VFO_B = CAT_buffer;

#ifdef debug
  Serial.print("VFO-B: ");
  Serial.println(VFO_B);
#endif

  send_command(get_vfo);    // Send the VFO Selected command

  get_response();   // Receive the VFO Selected response
  vfo = CAT_buffer.substring(2);

#ifdef debug
  Serial.print("VFO: ");
  Serial.println(vfo);
#endif

  send_command(get_mode);   // Send the Get Mode Command

  get_response();   // Receive the Get Mode response
  mode = CAT_buffer.substring(3);

  convert_mode(); // Convert the CAT Mode response into the corresponding text

#ifdef debug
  Serial.print("Mode: ");
  Serial.println(mode);
#endif

  update_display(); // Update the TFT display

  delay(update_delay);  // Delay until the next display update
}

void get_response() // Receives the CAT command response from the radio
{

#ifdef debug
  Serial.print("   Start RX  ");
#endif

  // Set a timeout value
  current_millis = millis();  // Get the current time
  timeout = current_millis + timeout_delay; // Calculate the timeout time
  // Check for millis() rollover condition - the Arduino millis() counter rolls over about every 47 days

  if (timeout < current_millis) // We've calculated the timeout during a millis() rollover event
  {
    timeout = timeout_delay; // Go ahead and calculate as if we've rolled over already (adds a few millis to the timeout delay)
  }

  // Get ready to receive CAT response from the radio
  receiving = true;
  timeout_error = false;
  CAT_buffer = "";
  do
  {
    if (millis() > timeout) // We've exceeded the timeout delay
    {
      // We timed out - exit
      receiving = false;
      timeout_error = true;

      CAT_buffer = "";
      break;
    }
    if (altSerial.available() && receiving) // If there's a character in the rx buffer and we're ok to receive
    {
      rx_char = altSerial.read(); // Get the incoming character
      if (rx_char == ';') // ";" indicates the end of the response from the radio
      {
        receiving = false;  // Turn off the ok to receive flag
      }  else
      {
        CAT_buffer = CAT_buffer + rx_char;  // Add the received character to the CAT rx string
      }

    }
  } while (receiving);  // Keep looping while we're ok to receive data from the radio

#ifdef debug
  if (timeout_error)
  {
    Serial.println("   Timeout ");
  } else
  {
    Serial.print("   Received complete: ");
    Serial.println(CAT_buffer);

  }
#endif
}

void send_command(String CAT_command) // Sends the CAT command string to the radio
{
  // Send the CAT Command

#ifdef debug
  Serial.print("Sending: ");
  Serial.print(CAT_command);
#endif

  altSerial.print(CAT_command); // Sends the CAT command to the radio

#ifdef debug
  Serial.print("  Send Complete  ");
#endif
}

void convert_mode()  // Convert the mode to the mode name
{
  // Mode is a single character string

  switch (mode[0])// convert the single character mode string to a char variable
  {
    case '1': // LSB
      mode = "LSB";
      break;

    case '2': // USB
      mode = "USB";
      break;

    case '3': // CW
      mode = "CW";
      break;

    case '4': // FM
      mode = "FM";
      break;

    case '5': // AM
      mode = "AM";
      break;

    case '6': // DATA (RTTY-LSB)
      mode = "DATA (RTTY-LSB)";
      break;

    case '7': // CW-R
      mode = "CW-R";
      break;

    case '8': // USER-L
      mode = "USER-L";
      break;

    case '9': // DATA (RTTY-USB)
      mode = "DATA (RTTY-USB)";
      break;

    case 'B': // FM-N
      mode = "FM-N";
      break;

    case 'C': // USER-U
      mode = "USER-U";
      break;

    default: // Unknown mode
      mode = " ";
      break;

#ifdef debug
      Serial.print("Mode String: ");
      Serial.println(mode);
#endif
  }
}


void update_display() // Updates the TFT display
{
  if (first_pass) // Only do this part the first time the function is called
  {
    // Clear the screen and display normal operation
    tft.fillScreen(ST7735_BLACK); // Clear the display
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 1
    delay(tft_delay);
    tft.setTextColor(ST7735_BLUE); // Set the text color to BLUE
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 2
    delay(tft_delay);
    tft.setCursor(30, 5);
    delay(tft_delay);
    tft.print("Yaesu CAT Display");  // Display screen title
    tft.setCursor(5, 40);
    delay(tft_delay);
    tft.print("VFO A :");  // Display VFO A
    tft.setCursor(5, 60);
    delay(tft_delay);
    tft.print("VFO B :");  // Display VFO B
    delay(tft_delay);
    tft.setCursor(5, 80);
    delay(tft_delay);
    tft.print("Mode  :");  // Display Mode
    delay(tft_delay);
  } else
  {
    clear_data();  // Clear the value display area

    tft.setCursor(50, 40);
    delay(tft_delay);
    tft.print(VFO_A); // Display VFO-A Value

    tft.setCursor(50, 60);
    delay(tft_delay);
    tft.print(VFO_B); // Display VFO-B Value

    tft.setCursor(50, 80);
    delay(tft_delay);
    tft.print(mode);  // Display mode

    if (vfo == "0")
    {
      // VFO-A is active
      tft.setCursor(115, 40); // Select the active VFO line
    }

    if (vfo == "1")
    {
      // VFO-B is active
      tft.setCursor(115, 60); // Select the active VFO line
    }

    tft.print("Active"); // Indicate the active VFO

  }
}

void clear_data() //Clears the data area of the display
{
  tft.fillRect(50, 40, 110, 50, ST7735_BLACK); // Clear the CAT data area
  delay(tft_delay);
}

void format_string()  // Formats the VFO data RX string data for display
{
  CAT_buffer = CAT_buffer.substring(2);  // First, strip off the CAT command echo
  CAT_buffer = CAT_buffer.substring(0, 2) +  "." + CAT_buffer.substring(2, 5) +  "." + CAT_buffer.substring(5); // Add periods to separate digits
  if (CAT_buffer.startsWith("0"))
  {
    // It's Below 10MHZ - strip off the leading zero
    CAT_buffer = CAT_buffer.substring(1);
  }
}

