/* Yaesu FT-950/1200/2000/3000/5000 Rotor Controller
 * Implements the Yaesu FT series transceiver front panel rotor control features to drive a 
 * standard CDE/HyGain rotor controller
 * 
 * written by Glen Popiel - KW5GP
 * 
 * uses Adafruit ADS1X15 and MCP4725 libraries
 *  
*/

#define debug_mode 1  // Set to 1 to enable the serial monitor diagnostic data, set to 0 to turn off diagnostic data

// Include the libraries we need
#include <Wire.h>
#include <Adafruit_ADS1015.h>
#include <Adafruit_MCP4725.h>

// Instantiate the A/D object
Adafruit_ADS1115 ads; 

// Instantiate the D/A object
Adafruit_MCP4725 dac;

// Define the I/O pins and brake delay
#define CW_in_pin 2 // the Clockwise drive input pin from the rig
#define CCW_in_pin 3  // The Counter-clockwise input pin from the rig
#define CW_relay_pin 5 // The Clockwise relay drive pin on the rotor controller
#define CCW_relay_pin 4 // The Counter-clockwise relay drive pin on the rotor controller
#define Brake_pin 6 // The brake relay pin on the rotor controller
#define brake_delay 500 // The brake engage/disengage delay time in milliseconds

boolean rotating = false; // Variable to indicate that the rotor is being turned from the front panel

int rig_cal_zero = 61;  // The D/A counts representing the 0 degree point on the rig
int rig_cal_360 = 2850; // The D/A counts representing the 360 degree point on the rig
int rig_cal_450 = 3522; // The D/A counts representing the 450 degree point on the rig (for the Yaesu G-450 and SA series rotor controllers)
int rig_cal_max = rig_cal_450;  // Default to the 450 degree calibration setting
int rotate_degrees = 360; // Set the maximum rotation degrees of the rotor

int read_adc;  // Variable to hold the A/D value
int rotor_cal_zero = 45;  // The A/D count value representing the 0 point on the rotor controller
int rotor_cal_max = 7116;  // The A/D count value representing the max point on the rotor controller
int rotor_degrees, actual_degrees, rig_degrees;  // Variables to keep track of positioning data in degrees

float volts_per_count = .001245;  // Used in diagnostic data to display the actual voltage output by the D/A
float multiplier = 0.0078125; // Used in diagnostic data to display the actual A/D read voltage

void setup()
{
  pinMode(CW_in_pin, INPUT_PULLUP); // Configure the rig clockwise input pin with Arduino internal pullup resistor enabled
  pinMode(CCW_in_pin, INPUT_PULLUP); // Configure the rig counterclockwise input pin with Arduino internal pullup resistor enabled
  pinMode(CW_relay_pin, OUTPUT);  // Configure the clockwise drive relay for output
  pinMode(CCW_relay_pin, OUTPUT);  // Configure the counterclockwise drive relay for output
  pinMode(Brake_pin, OUTPUT);  // Configure the brake drive relay for output
  
  if (debug_mode)
  {
    Serial.begin(9600);
    Serial.println("Rotor Controller Initializing");
  }

  // If the rotor controller only does 360 degrees instead of 450 degrees, change the maximum D/A value to match
  if(rotate_degrees == 360)
  {
    rig_cal_max = rig_cal_360;
  }
  if (debug_mode)
  {
    Serial.println("Starting D/A converter");
    Serial.println("Setting A/D for differential reading from AIN0 (P) and AIN1 (N)");
    Serial.println("ADC Range: +/- 0.256V (1 bit = .0078125mV for the ADS1115)");
  }
  
  dac.begin(0x62);  // Start the MCP4725 D/A converter

  // The ADC input range (or gain) can be changed via the following
  // functions, but be careful never to exceed VDD +0.3V max, or to
  // exceed the upper and lower limits if you adjust the input range!
  // Setting these values incorrectly may destroy your ADC!
  //                                                                ADS1015  ADS1115
  //                                                                -------  -------
  // ads.setGain(GAIN_TWOTHIRDS);  // 2/3x gain +/- 6.144V  1 bit = 3mV      0.1875mV (default)
  // ads.setGain(GAIN_ONE);        // 1x gain   +/- 4.096V  1 bit = 2mV      0.125mV
  // ads.setGain(GAIN_TWO);        // 2x gain   +/- 2.048V  1 bit = 1mV      0.0625mV
  // ads.setGain(GAIN_FOUR);       // 4x gain   +/- 1.024V  1 bit = 0.5mV    0.03125mV
  // ads.setGain(GAIN_EIGHT);      // 8x gain   +/- 0.512V  1 bit = 0.25mV   0.015625mV
  
  ads.setGain(GAIN_SIXTEEN);    // Set the ADS gain to 16x gain  +/- 0.256V  1 bit = 0.125mV  0.0078125mV
  
  ads.begin();  // Start the ADS 1115 A/D converter
}

void loop()
{
  // If we were rotating but both rig control pins are now high, stop rotating
  if ((rotating) && digitalRead(CW_in_pin) && digitalRead(CCW_in_pin)) 
  {
    // stop rotation
    if (debug_mode)
    {
      Serial.println("Stopping Rotation");
    }
    digitalWrite(CCW_relay_pin, LOW); // De-energize the rotate relays
    digitalWrite(CW_relay_pin, LOW);
    delay(brake_delay); // Wait for the rotation to stop
    digitalWrite(Brake_pin, LOW); // engage the rotor brake
    rotating = false; // Reset the rotating flag    
  }
 
  // Check the input pins and start rotating if necessary
  if(digitalRead(CW_in_pin) == LOW && !rotating)  // If the Clockwise rig input pin is low and we're not rotating, start rotating clockwise
  {
    // start rotating clockwise
    if (debug_mode)
    {
      Serial.println("Starting CW rotation");
    }
    digitalWrite(Brake_pin, HIGH); // release the brake
    delay(brake_delay); // Wait for the brake to release
    digitalWrite(CW_relay_pin, HIGH); // Energize the CW rotate relay
    rotating = true;  // Set the rotating flag
  }
  else if (digitalRead(CCW_in_pin) == LOW && !rotating) // If the Counter-clockwise rig input pin is low and we're not rotating, start rotating counterclockwise
  {
    Serial.println("Starting CCW rotation");
    // start rotating counter-clockwise
    digitalWrite(Brake_pin, HIGH); // release the brake
    delay(brake_delay); // Wait for the brake to release
    digitalWrite(CCW_relay_pin, HIGH); // Energize the CCW rotate relay
    rotating = true;  // Set the rotating flag    
  }

  read_adc = ads.readADC_Differential_0_1();  // Read the A/D using Differential mode
    
  if (debug_mode)
  {
    Serial.print("Differential: "); Serial.print(read_adc); Serial.print("("); Serial.print(read_adc * multiplier); Serial.print("mV)  ");
  }

  rotor_degrees = map(read_adc, rotor_cal_zero, rotor_cal_max, 0, 360);  // Map the Current Azimuth to Degrees
  
  if(rotor_degrees > 360) // if we're over 360 degrees, set the value to 360 degrees (and recheck your calibration values)
  {
    rotor_degrees = 360;
  }
  if (rotor_degrees < 0)// if we're under 0 degrees, set the value to 0 degrees (and recheck your calibration values)
  {
    rotor_degrees = 0;
  }
  
  /* Correction Since Azimuth Reading starts at Meter Center Point on the CDE/Hygain rotor controller
   *  Note - this is for a CDE/Hygain controller with North (0/360) at center, you will need to change (or remove) this
   *  for a controller with South at the center meter reading
   *  
   *  You could also change the Yaesu rig Menu 011 DISP RTR STU setting to get around all this, but this sketch assumes you
   *  don't want to play with your rig Menu settings
  */
  if (rotor_degrees > 180)
  {
    actual_degrees = rotor_degrees - 180;
  } else 
  { 
    actual_degrees = rotor_degrees + 180;
  }
  if (debug_mode)
  {
    Serial.print("Degrees Read: "); Serial.print(rotor_degrees); Serial.print("  Converted Degrees: "); Serial.print(actual_degrees);
  }

  /* Now calculate and send the unconverted rotor degrees to the D/A since the Yaesu display starts at 0 degrees and goes to 
   *  360 degrees based on the D/A position voltage
   *  
   *  
   * We also need to tweak the degree value since we're using a 0 degree center reading on the controller meter
   * and the Yaesu rig controls will not allow you to rotate counterclockwise if the reading is at 0 or clockwise 
   * if the reading is at 450 degrees, so we cheat a bit and make it 1 degree to the good so the rig will let us 
   * turn counterclockwise at 0 degrees. Changing the Yaesu rig Menu 011 DISP RTR STU setting also avoids this issue
   * but this sketch assumes you don't want to play with your rig Menu settings
   *  
   */
   
  if (actual_degrees <1 )  
  {
   actual_degrees = 1;
  }
  
  rig_degrees = map(actual_degrees, 0, 360, rig_cal_zero, rig_cal_max); // Map the degree value to the D/A voltage needed to send to the rig
  dac.setVoltage(rig_degrees, false);  // Set the D/A Voltage - note, the second parameter is false since we don't need to save this value to the D/A's EEPROM

  if (debug_mode)
  {
    Serial.print("  A/D Count: "); Serial.print(rig_degrees); Serial.print("   A/D Voltage: "); Serial.println(rig_degrees * volts_per_count);
  }
}
